<?php
require_once(__DIR__ . '/class.file.php');
require_once(__DIR__ . '/class.script.php');
require_once(__DIR__ . '/../functions.system.php');
// Functions for intercepting and handling php errors
require_once(__DIR__ . '/../functions.error.php');

set_error_handler("errorHandler");
register_shutdown_function("fatalErrorHandler");

/**
 * Page to be rendered in Enterprise
 */
abstract class Page extends File
{
  protected $pageName;
  protected $title;
  protected $scripts = [];
  protected $favIcon = "/img/favicon.png";
  protected $arguments = [];
  protected $argumentsAccept = [];
  protected $argumentSignature = [];
  protected $globalsList;
  protected $html = "";
  protected $data;
  
  function __construct(
    $_fileName,
    $_argumentsAccept,
    $_argumentSignature = [],
    $_globalsAccept = []
  ) {
    session_init();
    $pageName = get_class($this);
    $this->setPageName($pageName);
    parent::__construct(
      $_fileName
    );
    $this->setArgumentsAccept($_argumentsAccept);
    $_argumentSignature = !empty($_argumentSignature) ? $_argumentSignature : [];
    $this->setArgumentSignature($_argumentSignature);
    $this->parseArguments();
    $this->setGlobalsAccept($_globalsAccept);
    $this->acceptGobals();
  }

  abstract protected function handleArguments();

  protected function handleHistory() {
    $GLOBALS['isPopup'] = false; // Ensure this page gets added to history array
    historyAdd();
    session_restart();
    historyTrim();
  }

  protected function prepare() {
    $scriptsHead = "";
    $scriptsTop = "";
    $scriptsBottom = "";
    foreach ($this->scripts as $script) {
      $script->interpolate($this->getData());
      switch ($script->getPosition()) {
        case Script::SCRIPT_POSITION_HEAD:
          $scriptsHead .= $script->getTag() . PHP_EOL;
          break;
        case Script::SCRIPT_POSITION_TOP:
          $scriptsTop .= $script->getTag() . PHP_EOL;
          break;
        case Script::SCRIPT_POSITION_BOTTOM:
          $scriptsBottom .= $script->getTag() . PHP_EOL;
          break;
      }
    }
    $this->interpolate(
      [
        'title' => $this->getTitle(),
        'favIcon' => $this->getFavIcon(),
        'scriptsHead' => $scriptsHead,
        'scriptsTop' => $scriptsTop,
        'content' => $this->getContent(),
        'scriptsBottom' => $scriptsBottom
      ]
    );

    $this->setContent(appendSystemVersion($this->getContent(), getSystemVersion()));
  }

  public function render($echo = true) {
    $this->prepare();
    if ($echo) {
      $this->setHeaders();
      echo($this->getContent());
    } else {
      return $this->getContent();
    }
  }

  public function setTitle($_title) {
    $this->title = $_title;
  }

  public function getTitle() {
    return $this->title;
  }

  public function setFavIcon($_favIcon) {
    $this->favIcon = $_favIcon;
  }

  public function getFavIcon() {
    return $this->favIcon;
  }

  public function setScripts($_scripts) {
    $this->scripts = $_scripts;
  }

  public function getScripts() {
    return $this->scripts;
  }

  public function setData($_data) {
    $this->data = $_data;
  }

  public function getData() {
    return $this->data;
  }

  protected function parseArguments() {
    // TODO: Cater for variable type checks
    // TODO: Create class 'Argument'
    $length = count($this->getArgumentSignature());
    $this->setArgumentsAccept(array_pad($this->getArgumentsAccept(), $length, ""));
    $argumentsAccept = $this->getArgumentsAccept();
    $argumentSignature = $this->getArgumentSignature();
    if (empty($argumentSignature) && !empty($argumentsAccept)) {
      throw new Exception("ERROR: '".$this->getPageName()."' accepts arguments, but does not define a signature");
    }
    if (gettype($argumentSignature) !== 'array') {
      throw new Exception("ERROR: '".$this->getPageName()."' can only accept an argument signature of type 'array'");
    }
    if (!empty($argumentSignature)) {
      foreach ($argumentsAccept as $key=>$name) {
        if (!isset($argumentSignature[$key])) {
          throw new Exception("ERROR: Cannot find argument '$name' in argument signature defined for ".$this->getPageName()."'");
        }
        $this->arguments[$argumentSignature[$key]] = $argumentsAccept[$key];
      }
    }
  }

  public function setArguments($_arguments) {
    $this->arguments = $_arguments;
  }

  public function getArguments() {
    return $this->arguments;
  }

  public function getArgument($_argument) {
    $argumentType = gettype($_argument);
    if (in_array($argumentType, ['object', 'array'])) {
      throw new Exception("ERROR: Cannot get argument of type '$argumentType'");
    }
    $arguments = $this->getArguments();
    if (isset($arguments[$_argument])) {
      return $arguments[$_argument];
    }
    return null;
  }

  public function setArgumentSignature($_argumentSignature) {
    $this->argumentSignature = $_argumentSignature;
  }

  public function getArgumentSignature() {
    return $this->argumentSignature;
  }

  public function setArgumentAccept($_argumentAccept) {
    $this->argumentAccept = $_argumentAccept;
  }

  public function getArgumentAccept() {
    return $this->argumentAccept;
  }

  protected function acceptGobals() {
    if (!empty($this->globalsAccept)) {
      if (!in_array('history', $this->globalsAccept)) {
        $this->globalsAccept['history'] = "";
      }
      foreach ($this->globalsAccept as $name=>$value) {
        $this->globalsList[$name] = isset($GLOBALS[$name]) 
          ? $GLOBALS[$name] : $this->globalsAccept[$name];
      }
    }
  }

  public function setGlobals($globalsList) {
    if (!empty($globalsList)) {
      foreach ($globalsList as $name=>$value) {
        $GLOBALS[$name] = isset($globalsList[$name]) 
          ? $globalsList[$name] : "";
        $this->setGlobal($name, $GLOBALS[$name]);
      }
    }
  }

  public function setGlobal($name, $value) {
    $this->globalsList[$name] = $value;
  }

  public function getGlobal($name, $checkExists = true) {
    if (isset($this->globalsList[$name])) {
      return $this->globalsList[$name];
    } else {
      if ($checkExists) {
        throw new Exception("ERROR: Global '{$name}' not found");
      } else {
        return false;
      }
    }
  }

  public function setArgumentsAccept($_argumentsAccept) {
    $this->argumentsAccept = $_argumentsAccept;
  }

  public function getArgumentsAccept() {
    return $this->argumentsAccept;
  }

  public function setGlobalsAccept($_globalsAccept) {
    $this->globalsAccept = $_globalsAccept;
  }

  public function getGlobalsAccept() {
    return $this->argumentAccept;
  }

  public function setPageName($_pageName) {
    $this->pageName = $_pageName;
  }

  public function getPageName() {
    return $this->pageName;
  }

  protected function setHeaders() {
    // Prevent caching of dynamic pages
    // Current date and time in RFC 1123 date format
    $today = gmdate("D, d M Y H:i:s") . " GMT";
    // Immediate expiry
    header('Expires: -1');
    // Must be current timestamp for IE dialup users
    header('Last-Modified: ' . $today);
    // HTTP/1.0 header
    header('Pragma: no-cache');
    // HTTP/1.1 header (see RFC 2616)
    header('Cache-Control: no-cache, no-store, must-revalidate');
    // Neither post-check nor pre-check exist in the RFC
    header("Cache-Control: post-check=0, pre-check=0", false);
  }

  public static function redirect($url, $fields=false) {
    if(!headers_sent() && empty($fields)) {
      header("Location: $url");
      die();
    }

    echo '<html>
      <body>
        <form id="redirect_form" method="POST" action="'.$url.'">
    ';
    if(is_array($fields)) {
      foreach($fields as $name=>$value) {
        echo '<textarea style="display:none" name="'.$name.'">'.t_encodeHTMLField($value).'</textarea>'."\n";
      }
    }
    echo '
        </form>
        <script>document.getElementById("redirect_form").submit();</script>
      </body>
    </html>';
    die();
  }
}
