<?php

/**
 * functions.error.php - Functions used when dealing with php error handling
 */

global $errors;
$errors = array();

function errorHandler($error_number, $error_message, $error_file="", $error_line=1) {
	global $errors;

	$errors[] = errorHandlerCreate($error_number,$error_message);

	// Execute the internal PHP error handler as well
	return false;
}

function fatalErrorHandler() {
	global $errors;

	# Getting last error
	$error = error_get_last();

	$errors[] = errorHandlerCreate($error['type'],$error['message']);

	# Checking if last error is a fatal error 
	if(($error['type'] === E_ERROR) || ($error['type'] === E_USER_ERROR)) {
		redirectTo("reservation.php?3000+1",array('errors'=>json_encode($errors)));
	}
}

function errorHandlerTypeDesc($errorNumber) {
	// define an assoc array of error string
	// in reality the only entries we should
	// consider are E_WARNING, E_NOTICE, E_USER_ERROR,
	// E_USER_WARNING and E_USER_NOTICE
	$types = array (
		E_ERROR              => 'Error',
		E_WARNING            => 'Warning',
		E_PARSE              => 'Parsing Error',
		E_NOTICE             => 'Notice',
		E_CORE_ERROR         => 'Core Error',
		E_CORE_WARNING       => 'Core Warning',
		E_COMPILE_ERROR      => 'Compile Error',
		E_COMPILE_WARNING    => 'Compile Warning',
		E_USER_ERROR         => 'User Error',
		E_USER_WARNING       => 'User Warning',
		E_USER_NOTICE        => 'User Notice',
		E_STRICT             => 'Runtime Notice',
		E_RECOVERABLE_ERROR  => 'Catchable Fatal Error',
		E_DEPRECATED         => 'Deprecated',
		E_USER_DEPRECATED    => 'User Deprecated'
	);

	if(in_array($errorNumber,array_keys($types))) {
		return $types[$errorNumber];
	} else {
		return "Unknown";
	}
}

function errorHandlerURL() {
	$url = (isset($_SERVER['REQUEST_SCHEME'])?$_SERVER['REQUEST_SCHEME']:"http") . "://" . 
		$_SERVER['SERVER_NAME'] . ":" . $_SERVER['SERVER_PORT'] . $_SERVER['REQUEST_URI'];
	if(strlen($url) > 100) {
		$url = substr($url,0,100) . "...";
	}

	return $url;
}

function errorHandlerCreate($error_number,$error_message) {
	global $principal_name_short;
	global $principal_id;
	global $dbcode;
	global $debugEnabled;
	global $systemVersion;
	global $lDB;
	global $userid;

	$error = array();

	$error['Client'] = $principal_name_short . " ($principal_id $dbcode ". 
		($lDB->isMaster?"master":"slave") . ")";

	// user who generated the error
	$user = $lDB->get("
		SELECT
			TRIM(CONCAT(pr_persona.pr_name_first, ' ', pr_persona.pr_name_last))
		FROM
			pr_persona
		WHERE
			pr_persona.pr_persona_ix = '$userid'
	",4);
	if(empty($user)) {
		$user = "Unknown";
	}
	$error['User'] = $user . " ($userid)";

	// url of the error
	$error['URL'] = errorHandlerURL();

	// current system version
	$error['Version'] = $systemVersion;

	// timestamp for the error entry
	$error['Timestamp'] = date("Y-m-d H:i:s (T)");

	$error['Error type'] = errorHandlerTypeDesc($error_number) . " (" .$error_number. ")";
	$error['Error message'] = $error_message;

	$e = new Exception;
	$error['Trace'] = explode("\n",$e->getTraceAsString());
	if($debugEnabled) {
		// Commented out as this throws a Allowed Memory Size Exhausted error. Might want to try again in future
		//$error['Backtrace'] = explode("\n",print_r(debug_backtrace(), true));
	}

	return array_merge(array('raw'=>$error), errorHandlerRender($error));
}

function errorHandlerRender($error) {
	global $errors;

	$errorCount = sizeof($errors)+1;

	$errorText = "[Error #$errorCount]\n";
	$errorHTML = '
		<table cellpadding="0" cellspacing="0" width="100%" style="margin-bottom:10px;">
		<col style="width: 100px" />
		<col />
		<tr>
			<td colspan="2" class="txtb txti">Error #'.$errorCount.'</td>
		</tr>
	';
	foreach($error as $key=>$item) {
		$errorText .= $key . ":";
		$errorHTML .= '
			<tr>
				<td valign="top">'.$key.':</td>
				<td>
		';
		if(!is_array($item)) {
			$errorText .= ' ' . $item . "\n";
			$errorHTML .= t_encodeHTMLField($item);
		} else {
			$errorText .= "\n";
			foreach($item as $traceKey=>$traceItem) {
				$errorText .= "\t$traceItem\n";
				$errorHTML .= str_replace(" ","&nbsp;",t_encodeHTMLField($traceItem))."<br />";
			}
		}
		$errorHTML .= '
				</td>
			</tr>
		';
	}
	$errorText .= "\n";
	$errorHTML .= '
		</table>
	';

	return array('text'=>$errorText, 'html'=>$errorHTML);
}

function errorHandlerCommon($fatalErrors=array()) {
	global $errors;

	$finalErrors = array_merge($fatalErrors, $errors);

	return showpage(array(
		'errorHandlingContent'=>array_reduce($finalErrors, function($text, $item) {
			return $text . $item['text'];
		},""),
		'errorHandlingContentHTML'=>array_reduce($finalErrors, function($html, $item) {
			return $html . $item['html'];
		},""),
		'errorHandlingContentJSON'=>json_encode(array_reduce($finalErrors, function($json, $item) {
			$json[] = $item['raw'];
			return $json;
		},array()))
	),join("", file(__DIR__ . "/../../../public/html/error_handling_common.htm")));
}

function errorHandlerShowBanner($htmlOutput) {
	global $errors;

	$bannerMessage = "Oops! We're sorry, but something went wrong while displaying this page. <a href=\"Javascript:;\" onclick=\"errorHandlingProcess();\">Let us know about it</a> or <a href=\"Javascript:;\" onclick=\"errorHandlingShowPopup();\">view the details</a>";
	$banner_style_override = "";

	if (substr($errors[0]['raw']['Error message'], 0, 8) == 'special:') {
		// In order to display a custom error banner message, the error message must start with 'special:'
		// Example call: errorHandler(1, "special:This is a custom message");
		$bannerMessage = substr($errors[0]['raw']['Error message'], 8) . "<a href=\"Javascript:;\" style=\"margin-left: 20px;\" onclick=\"errorHandlingPopupClose(1);\">Close</a>";
		$banner_style_override = "height: 40px !important; padding-top: 25px;";
	}

	$errorHTML = join("", file(__DIR__ . "/../../../public/html/error_handling_banner.htm")).errorHandlerCommon();
	$errorHTML = str_replace("!banner_message!", $bannerMessage, $errorHTML);
	$errorHTML = str_replace("!banner_style_override!", $banner_style_override, $errorHTML);

	$bodyPos = strpos($htmlOutput, '</body>');
	$bodyPos = ($bodyPos === false) ? strlen($htmlOutput) : $bodyPos;
	$htmlOutput = substr_replace($htmlOutput, $errorHTML, $bodyPos, 0);
	return $htmlOutput;
}

function errorHandlingTest($allowFatal=false, $runTest=true) {
	global $job;

	if($job == 3000) { return; } // don't run tests when on the fatal error form

	$tests = [
		/* 0 */ ['name'=>"Undefined variable (read)", 'code'=>function() {
			$error_test_variable = $error_test_undefined_variable; 
		}],
		/* 1 */ ['name'=>"Undefined variable (increment)", 'code'=>function() {
			$error_test_undefined_variable++;
		}],
		/* 2 */ ['name'=>"Invalid constant", 'code'=>function() {
			$error_test_variable = UNDEFINED_CONSTANT;
		}],
		/* 3 */ ['name'=>"Invalid array index", 'code'=>function() {
			$error_test_array = array();
			for($error_test_counter = 0; $error_test_counter < 5; $error_test_counter++) {
				$error_test_variable = $error_test_array[$error_test_counter];
			}
		}],
		/* 4 */ ['name'=>"Call to undefined function", 'code'=>function() {
			error_test_invalid_function();
		}, 'fatal'=>true],
		/* 5 */ ['name'=>"Memory exhaustion", 'code'=>function() {
			$error_test_variable = "x";
			while (true) {
				$error_test_variable .= $error_test_variable;
			}
		}, 'fatal'=>true]
	];

	if($runTest === true) { // Run all tests
		$testsToRun = $tests;
	} elseif(array_key_exists($runTest,$tests)) {
		$testsToRun = array($tests[$runTest]);
	} else {
		$testsToRun = array();
	}
	foreach($testsToRun as $test) {
		$isFatal = isset($test['fatal']) && $test['fatal'];
		if(!$isFatal || ($isFatal && $allowFatal)) {
			$test['code']();
		}
	}
}
