<?php

/**
 * function.reservation.advanded - Functions relating to advanced reservation information
 */

require_once(__DIR__ . "/class.stock.php");

// checkOptionalComponents - Checks if any optional components exist for this item
// =================================================================================================
function checkOptionalComponents($date,$rateTypeId,$accommTypeId) {
   global $lDB;

   // find the optional components available to this item
   $cellStart = $date[0];
   $cellEnd = $date[1];
   $periodList = "";

   $splitItems = $lDB->get("SELECT rf_period_split FROM rf_default",4);
   if ($splitItems == "2") {
      $where = "
            (rt_period.rt_period_date_from >= '$cellStart' AND rt_period.rt_period_date_from <= '$cellEnd')
            OR (rt_period.rt_period_date_to >= '$cellStart' AND rt_period.rt_period_date_to <= '$cellEnd')
            OR (rt_period.rt_period_date_from < '$cellStart' AND rt_period.rt_period_date_to > '$cellEnd')
      ";
   } else {
      $where = "
            (rt_period.rt_period_date_from <= '$cellStart' AND rt_period.rt_period_date_to >= '$cellStart')
      ";
   }

   $periodList = $lDB->get("
         SELECT
            rt_period.rt_period_ix
         FROM
            rt_period
         WHERE
            $where
         ORDER BY
            rt_period_date_from
   ",3);
   cleanse($periodList);
   $result = $lDB->get("
      SELECT
         COUNT(rt_rate_component_ix)
      FROM
         rt_rate_component
            INNER JOIN rt_rate ON rt_rate_ix = rt_rate_component.rt_rate_id
            INNER JOIN rt_component ON rt_component.rt_component_ix = rt_rate_component.rt_component_id
            INNER JOIN rt_period ON rt_period.rt_period_ix = rt_rate.rt_period_id
            INNER JOIN rt_rate_type ON rt_rate_type.rt_rate_type_ix = rt_rate.rt_rate_type_id
            INNER JOIN rf_currency ON rf_currency.rf_currency_ix = rt_rate_type.rf_currency_id
      WHERE
         rt_rate_component.rt_comp_opt_yn != '0'
         AND rt_rate.rt_rate_type_id = '".$rateTypeId."'
         AND rt_rate.ac_accomm_type_id = '".$accommTypeId."'
         AND rt_rate.rt_period_id IN ('".join("','",$periodList)."')
      ORDER BY
         rt_component.rt_component_desc,
         rt_period.rt_period_date_from
   ",4);

   if ($result != "0") {
      return true;
   }
   return false;
}

// testResConfirmation - verifies that this reservation can be confirmed - returns TRUE or error msg
// =================================================================================================
function testResConfirmation($reservationId) {
   global $lDB;

   $itemList = $lDB->get("SELECT rv_reservation_item_ix FROM rv_reservation_item WHERE rv_reservation_id = '$reservationId'",3);

   // Test 1 - Does the res have items?
   if (sizeof($itemList) < 1) {
      return "Please define items for this reservation.";
   }

   // Test 2 - Do we have sufficient rate data?
   //for ($i=0;$i<sizeof($itemList);$i++) {
   // $resItemId = $itemList[$i];
   foreach($itemList as $resItemId) {
      $test = $lDB->get("
         SELECT
            COUNT(rv_res_item_comp.rv_res_item_comp_id) AS CountOfrv_res_item_comp_id
         FROM
            rv_res_item_comp
         WHERE
            rv_res_item_comp.rv_reservation_item_id = '$resItemId'
      ",4);
      if ($test == "0") {
         return "The reservation rate data is incomplete.";
      }
   }

   // Test 3 - Do we have accurate exchange rate information?
   $invCurr = getInvCurr($reservationId);
   $rateCurr = $lDB->get("
      SELECT
         rt_rate_type.rf_currency_id
      FROM
         rv_reservation
         INNER JOIN rt_rate_type ON rt_rate_type.rt_rate_type_ix = rv_reservation.rt_rate_type_id
      WHERE
         rv_reservation_ix = '$reservationId'
   ",4);
   if ($invCurr != $rateCurr) {
      list($rate, $date) = $lDB->get("
         SELECT
            rv_exchg_rate,
            rv_exchg_expiry_date
         FROM
            rv_reservation
         WHERE
            rv_reservation_ix = '$reservationId'
      ",1);
      if ($date < date("Y-m-d") || $rate == "0.0000") {
         return "Please record a valid accommodation exchange rate.";
      }
   }

   // Test 4 - Does every extra for this reservation have a valid exchange information?
   $extraList = $lDB->get("
      SELECT
         rv_extra.rv_extra_ix as rv_extra_id,
         rv_extra.rf_currency_id,
         rv_extra.rv_extra_exch_rate,
		 rv_extra.rv_extra_exch_expiry,
		 fn_folio.rf_currency_id as folio_currency_id
      FROM
         rv_extra
		 INNER JOIN rf_currency ON rf_currency.rf_currency_ix = rv_extra.rf_currency_id
		 INNER JOIN fn_folio ON fn_folio.fn_folio_ix = rv_extra.fn_folio_id
      WHERE
         rv_extra.rv_reservation_id = '$reservationId'
   ",2);
   foreach($extraList as $item) {
      list($extraId, $extraCurr, $extraRate, $extraDate, $folioCurr) = $item;
      if ($folioCurr != $extraCurr) {
         if ($extraDate < date("Y-m-d") || $extraRate == "0.0000") {
            return "Please record a valid exchange rate for each extra charge.";
         }
      }
   }

   return "TRUE";
}


// check_availability - check the availability for all the items in a reservation by resnum
// ========================================================================================
function check_availability($resnum, $wizStep2=false) {
	global $lDB;

	# Set some defaults
	$error = array();
	$overbooked = false;
	$hasAvailability = false;
	$overbookingdetails = array();

	# Get the list of items and item data from the resnum
	$resultset = $lDB->get("
		SELECT
			rv_reservation_item_ix as rv_reservation_id,
			rv_item_date_arrive,
			rv_item_date_depart,
			ac_accomm_type_id,
			rv_item_accomm_count,
			rv_reservation_item.rv_reservation_item_ix
		FROM
			rv_reservation_item
		WHERE rv_reservation_id = '$resnum'
	",2);
	$oldItems = array_map(function($item) {
		return $item['rv_reservation_item_ix'];
	},$resultset);

	list($hasAvailability, $overbookingdetails) = check_multi_item_availability($resultset,$oldItems,true,true);

	if (!$hasAvailability) {
		if (!$wizStep2) {
			return $overbookingdetails;
		} else {
			// Return more details for Wizard step 2
			foreach($resultset as $item) {
				list($Item_ID, $Date_Arrive, $Date_Depart, $Accomm_Type_ID, $Unit_Count) = $item;
				$result = check_item_availability($Date_Arrive,$Date_Depart,$Accomm_Type_ID,$Unit_Count,$Item_ID);
				if ($result !== true) {
					array_push($error,$result);
				}
			}
			return $error;
		}
	} else {
		return true;
	}
}


// check_item_availability - check the availability for an item
// ============================================================
function check_item_availability($cArrive,$cDepart,$cAccommTypeId,$cRooms,$cItemId=0,$cAllocationList=false) {
	global $lDB;

	$departArray = explode("-",$cDepart);
	$endDate = date("Y-m-d",mktime(0,0,0,$departArray[1],$departArray[2]-1,$departArray[0]));
	$stock = new Stock($cAccommTypeId,$cArrive,$endDate,false,false,false,$cItemId);
	$avail = $stock->ToAvailability();
	if($cAllocationList !== false) {
		$cAllocationArray = explode(":",$cAllocationList);
		$allocation = $stock->ToAllocationAgent(false);
	}
	$overbooked = false;
	$min = 0;
	$error = array();
	$error['days'] = array();
	foreach($avail as $date=>$item) {
		$total = $item - $cRooms;
		if($cAllocationList !== false) {
			foreach($cAllocationArray as $allocId) {
				foreach($allocation as $agent) {
					foreach($agent['allocation_balance'] as $allocItem) {
						if($allocItem['rv_reservation_ix'] == $allocId) {
							$total += isset($allocItem['occupancy'][$date]) ? $allocItem['occupancy'][$date] : 0;
						}
					}
				}
			}
		}
		if($total < 0) {
			$overbooked = true;
			array_push($error['days'],$date);
			if($total < $min) {
				$min = $total;
			}
		}
	}
	if(!$overbooked) {
		return true;
	} else {
		# Add these values and more to the error output
		$error['date_arrive'] = $cArrive;
		$error['date_depart'] = $cDepart;
		$error['unit_count'] = $cRooms;
		$names = $lDB->get("
			SELECT
				ac_accomm_type.ac_accomm_desc,
				pr_persona.pr_name_last
			FROM 
				ac_accomm_type
				INNER JOIN pr_persona ON pr_persona.pr_persona_ix = ac_accomm_type.pr_business_id
			WHERE
				ac_accomm_type_ix = '$cAccommTypeId'
		",1);
		$error['accomm_name'] = $names['ac_accomm_desc'];
		$error['property_name'] = $names['pr_name_last'];
		$error['accomm_capacity'] = $stock->ToAccomm();

		# Get the amount of rooms overbooked on the worst day
		$max_overbook = $min * -1;
		$error['max_overbook'] = $max_overbook;

		return $error;
	}	
}

/**
 * check_multi_item_availability() - check availability for one or more itineraries
 *		if $returnDetails is true it will return an array of the overbooking details:
 *		e.g. $details = Array(
 *			[<ac_accomm_type_id>] => Array
 *				(
 *					[<date>] => <availibility>
 *				)
 *		
 *			[WB7] => Array
 *				(
 *					[2014-10-18] => -3
 *					[2014-12-20] => -5
 *				)
 *			...		
 *		
 *		)
 * @param array $newItems
 * @param array $oldItems
 * @param boolean $useLimit
 * @param boolean $returnDetails
 * @return boolean or array(boolean, array)
 **/
function check_multi_item_availability($newItems=array(),$oldItems=array(),$useLimit=true,$returnDetails=false) {
	if($useLimit) {
		$limit = userAvailLimit();
	} else {
		$limit = false;
	}
	$details = array();
	$accommItems = array();
	foreach($newItems as $item) {
		$ac_accomm_type_id = $item['ac_accomm_type_id'];
		if(!array_key_exists($ac_accomm_type_id,$accommItems)) {
			$accommItems[$ac_accomm_type_id] = array();
		}
		$accommItems[$ac_accomm_type_id][] = $item;
	}

	$overbooked = false;
	foreach($accommItems as $ac_accomm_type_id=>$items) {
		$rv_item_date_arrive = "9999-99-99";
		$rv_item_date_depart = "0000-00-00";
		foreach($items as $item) {
			if($item['rv_item_date_arrive'] < $rv_item_date_arrive) $rv_item_date_arrive = $item['rv_item_date_arrive'];
			if($item['rv_item_date_depart'] > $rv_item_date_depart) $rv_item_date_depart = $item['rv_item_date_depart'];
		}
		$departArray = explode("-",$rv_item_date_depart);
		$rv_item_date_end = date("Y-m-d",mktime(0,0,0,$departArray[1],$departArray[2]-1,$departArray[0]));

		$stock = new Stock($ac_accomm_type_id,$rv_item_date_arrive,$rv_item_date_end,false,false,false,join(":",$oldItems));
		$avail = $stock->ToAvailability(true,true,true,$limit);

		$error = array();
		$error['days'] = array();

		foreach($items as $item) {
			if(array_key_exists('allocations',$item)) {
				$cAllocationArray = $item['allocations'];
				$allocation = $stock->ToAllocationAgent(false);
			} else {
				$cAllocationArray = false;
				$allocation = array();
			}
			$min = 0;
			foreach($avail as $date=>$availItem) {
				if($date >= $item['rv_item_date_arrive'] && $date < $item['rv_item_date_depart']) {
					$avail[$date] -= $item['rv_item_accomm_count'];
					if($cAllocationArray !== false) {
						foreach($cAllocationArray as $allocId) {
							foreach($allocation as $agent) {
								foreach($agent['allocation_balance'] as $allocItem) {
									if($allocItem['rv_reservation_ix'] == $allocId) {
										if($allocItem['occupancy'][$date] > $item['rv_item_accomm_count']) {
											$allocRooms = $item['rv_item_accomm_count'];
										} else {
											$allocRooms = $allocItem['occupancy'][$date];
										}

										$avail[$date] += $allocRooms;
									}
								}
							}
						}
					}
				}
			}
		}

		foreach($avail as $date=>$total) {
			if($total < 0) {
				if(!array_key_exists($ac_accomm_type_id,$details)) {
					$details[$ac_accomm_type_id] = array();
				}
				$details[$ac_accomm_type_id][$date] = $total;
				$overbooked = true;
			}
		}
	}
	
	$hasAvailability = ($overbooked == false) ? true : false;
	if ($returnDetails) {
		return array($hasAvailability, $details);
	} else {
		return $hasAvailability;
	}
}



function check_room_availability($rv_item_date_arrive,$rv_item_date_depart,$itemId) {
	global $lDB;

	$itemCount = $lDB->get("SELECT COUNT(*) FROM rv_reservation_item WHERE rv_reservation_item_ix = '$itemId'",4);
	if($itemCount == "0") { // New item, so there is no possibility of a room overlapping
		return true;
	}

	$groupCount = $lDB->get("SELECT COUNT(*) FROM rv_res_item_group WHERE rv_reservation_item_id = '$itemId'",4);
	if($groupCount == "0") { // No groups, so there is no possibility of a room overlapping
		return true;
	}

	$roomList = $lDB->get("SELECT ac_accomm_room_id FROM rv_res_item_group WHERE rv_reservation_item_id = '$itemId'",3);
	$roomCheck = $lDB->get("
		SELECT DISTINCT
			ac_accomm_room.ac_desc
		FROM
			rv_res_item_group
			INNER JOIN rv_reservation_item ON rv_reservation_item.rv_reservation_item_ix = rv_res_item_group.rv_reservation_item_id
			INNER JOIN ac_accomm_room ON ac_accomm_room.ac_accomm_room_ix = rv_res_item_group.ac_accomm_room_id
		WHERE
			rv_res_item_group.rv_reservation_item_id <> '$itemId'
			AND rv_res_item_group.ac_accomm_room_id IN ('".join("','",$roomList)."')
			AND (
				(rv_reservation_item.rv_item_date_arrive >= '$rv_item_date_arrive' AND rv_reservation_item.rv_item_date_arrive < '$rv_item_date_depart')
				OR (rv_reservation_item.rv_item_date_depart > '$rv_item_date_arrive' AND rv_reservation_item.rv_item_date_depart <= '$rv_item_date_depart')
				OR (rv_reservation_item.rv_item_date_arrive < '$rv_item_date_arrive' AND rv_reservation_item.rv_item_date_depart > '$rv_item_date_depart')
			)			
		ORDER BY
			(ac_accomm_room.ac_desc+0),ac_accomm_room.ac_desc
			
	",3);
	if(sizeof($roomCheck) > 0) {
		return $roomCheck;
	}

	return true;
}

function resStatus($resStatusDates){
   $resStatus = "Pending";
   if ($resStatusDates['rv_provision_expiry_date'] > date("Y-m-d")) {
              $resStatus = "Provisional";
   }
   if ($resStatusDates['rv_wait_list_date'] > 0) {
      $resStatus = "Wait listed";
   }
   if ($resStatusDates['rv_confirmation_date'] > 0) {
              $resStatus = "Confirmed";
   }
   if ($resStatusDates['rv_invoice_date'] > 0) {
         $resStatus = "Invoiced";
   }
   if ($resStatusDates['rv_cancel_date'] > 0) {
      $resStatus = "Cancelled";
   }
   return $resStatus;
}

function userAvailLimit() {
	global $userid;
	global $lDB;

	list($sc_grp_avail_limit,$sc_grp_avail_limit_yn) = $lDB->get("
		SELECT
			sc_grp_avail_limit,
			sc_grp_avail_limit_yn
		FROM
			sc_user
			INNER JOIN sc_group ON sc_group.sc_group_id = sc_user.sc_group_id
		WHERE
			sc_user.pr_user_id = '$userid'
	",1);

	if($sc_grp_avail_limit_yn == "0") {
		$sc_grp_avail_limit = false;
	}

	if($sc_grp_avail_limit_yn == "1" && $sc_grp_avail_limit == "1") {
		$sc_grp_avail_limit = $lDB->get("SELECT rf_book_limit FROM rf_default",4);
	}

	return $sc_grp_avail_limit;
}
