<?php

/**
 * functions.reservation.itinerary.php - Functions related to the creation and maintenance of itineraries
 */

require_once(__DIR__ . "/db.pr_agent.php");
require_once(__DIR__ . "/db.pr_agent_prop.php");
require_once(__DIR__ . "/db.pr_business.php");
require_once(__DIR__ . "/db.rf_currency.php");
require_once(__DIR__ . "/db.rt_rate_type.php");
require_once(__DIR__ . "/db.rv_reservation.php");
require_once(__DIR__ . "/db.rv_reservation_item.php");
require_once(__DIR__ . "/db.rv_res_item_comp_rate_grp.php");
require_once(__DIR__ . "/db.rv_res_item_comp_tax.php");
require_once(__DIR__ . "/db.rv_res_item_rate_grp.php");
require_once(__DIR__ . "/db.rv_special.php");
require_once(__DIR__ . "/db.fn_folio.php");
require_once(__DIR__ . "/functions.financial.php");
require_once(__DIR__ . "/class.stock.php");
require_once(__DIR__ . "/class.itinerary.php");
require_once(__DIR__ . "/functions.specials.php");
require_once(__DIR__ . "/db.tc_sequence.php");
require_once(__DIR__ . '/class.audit.php');   

/**
 * calCreateItem() - Create a reservation itinerary item
 * @param $wizResId - rv_reservation_ix
 * @param $itemDataRaw structure - colon (:) separated list with:
 * 		[0] -> ac_accomm_type_id
 * 		[1] -> rv_item_date_arrive (needs / to - conversion)
 * 		[2] -> rv_item_date_depart (needs / to - conversion)
 * 		[3] -> nights (not used)
 * 		[4] -> accomm count
 * 		[5] -> adult count
 * 		[6] -> child count
 * @param $rate_id - rt_rate_type_id
 * @param $compList:
 * 	- array of rt_component_ids for optionals
 * 	- or string "defaultOnly" to only add the default optional components
 * @param $override
 * @param $rv_item_split_yn
 * @param $ovrdDiscAmt - not used
 * @param $itemId
 * @param $allocations
 * @param $allocationLink
 * @param $bl_block_itinerary_ix
 * @param $clearAllBlockAllocations
 * @param $bl_series_id - the block series to which a draw down allocation should belong
 * @return array $items
 **/
function calCreateItem($wizResId, $itemDataRaw, $rate_id, $compList, $override, $rv_item_split_yn, $ovrdDiscAmt="0", $itemId=false, $allocations=false, $allocationLink=false, $bl_block_itinerary_ix=false, $clearAllBlockAllocations=false, $bl_series_id=false, $retainExpiredBlockAllocations=false, $commPerc=false, $showItinErrorPopup=false) {
	global $lDB;

	$ac_accomm_type_id = "";
	$rv_item_date_arrive = "";
	$rv_item_date_depart = "";
	if (is_array($itemDataRaw)) {
		$ac_accomm_type_id = $itemDataRaw['ac_accomm_type_id'];
		$rv_item_date_arrive = $itemDataRaw['rv_item_date_arrive'];
		$rv_item_date_depart = $itemDataRaw['rv_item_date_depart'];
		$numnights = $itemDataRaw['nights'];
		$rateGroupArr = array();
		$rateGroups = $itemDataRaw['rate_groups'];
	} else {
		$data = explode(":",$itemDataRaw);

		$ac_accomm_type_id = $data[0];
		$rv_item_date_arrive = str_replace("/","-",$data[1]);
		$rv_item_date_depart = str_replace("/","-",$data[2]);
		$numnights = $data[3];

		$rateGroupArr = explode("~",$data[4]);
	}
	if (!is_array($compList)) {
		switch ($compList) {
			case 'defaultOnly':
				$components = getDefaultOptionalComponents($ac_accomm_type_id,$rate_id,$rv_item_date_arrive,$rv_item_date_depart);
				$compList = array();
				foreach ($components as $component) {
					$compList[] = $component['rt_component_ix'];
				}
				break;
			default:
				$compList = array();
				break;
		}
	}

	foreach($rateGroupArr as $rateGroupPair){
		$rateGroupPairArr = explode("_", $rateGroupPair);
		$rateGroupPairArr = array_pad($rateGroupPairArr,2,0);
		$rateGroups[] = array(
			"rt_rate_group_ix" => $rateGroupPairArr[0],
			"name"	=> "",
			"qty"	=> $rateGroupPairArr[1]);
	}

	// We don't want to lose the block allocation after recreation of itinerary record, because a certain
	// client now doesn't want Block Allocations to unlink automatically when they become invalid. Bah
	// Lets find if there's a valid Block Allocation attached to this item, and not just a normal allocation.
	// Later on we'll update this block allocation with the new itinerary ID.
	$blockAllocationId = $lDB->get("
		SELECT
			rv_reservation_item.rv_reservation_item_ix
		FROM
			rv_reservation_item
			LEFT JOIN rv_reservation ON rv_reservation.rv_reservation_ix = rv_reservation_item.rv_reservation_id
		WHERE
			rv_reservation_item.rv_link_res_item_id = '" . $itemId . "'
			AND rv_reservation_item.rv_reservation_item_ix IS NOT NULL
			AND rv_reservation.rv_reservation_ix IS NOT NULL
			AND rv_reservation.bl_block_period_id IS NOT NULL
			AND rv_reservation.bl_block_period_id != ''
	", 4);

	if ($clearAllBlockAllocations) {
		clearAllocations($itemId,$clearAllBlockAllocations);
        // prevent re-linking of block allocations
        if (!is_array($allocations)) {
            $allocations = explode(":",$allocations);
        }
        foreach ($allocations as $sequence => $allocation) {
            $isBlockAllocation = !empty($lDB->get("
                SELECT
                    rv_reservation.bl_block_period_id
                FROM
                    rv_reservation
                WHERE
                    rv_reservation.rv_reservation_ix = '$allocation'
                    AND rv_reservation.bl_block_period_id IS NOT NULL
                    AND rv_reservation.bl_block_period_id != ''
            ",4));
            if ($isBlockAllocation) {
                unset($allocations[$sequence]);
            }
        }
	}

	$newBooking = isset($GLOBALS['calendarPage']);

	$itinerary = new Itinerary();
	$itinerary->SetReservationItemId($itemId);
	$itinerary->Create($wizResId,$ac_accomm_type_id,$rv_item_date_arrive,$rv_item_date_depart,$rateGroups,$rate_id,$compList,$override,$allocationLink,$allocations,$rv_item_split_yn,$bl_block_itinerary_ix,$bl_series_id,$retainExpiredBlockAllocations,$commPerc);
	$itinerary->showItinErrorPopup = $showItinErrorPopup;	// whether or not to show an error popup if an itinerary can not be recreated (like zero-night TBA)
	$itinerary->SetDoNotDeleteAllocations(true);			// Do not delete the actual Block Allocation records
	$items = $itinerary->Save($newBooking);

	if (!$clearAllBlockAllocations && !empty($blockAllocationId) && isset($items[0]) && !empty($items[0])) {
		// Re-link block allocation to new itinerary record. There may be more than one itinerary record,
		// because of rate period splitting, so we only use the first one.
		$lDB->put("UPDATE rv_reservation_item SET rv_link_res_item_id = '" . $items[0] . "' WHERE rv_reservation_item_ix = '" . $blockAllocationId . "'");
	}

	if($newBooking) {
		db_rv_reservation_set_expiry($wizResId);
	}
	return $items;
}

function calApplyDifferentConfig($oldItems, $newItems, $mode, $clearAllBlockAllocations) {
	global $lDB;

	if(empty($newItems) && $mode == "edit") {
		foreach($oldItems as $rv_reservation_item_id) {
			$lDB->put("UPDATE rv_reservation_item SET rv_item_split_yn = 0 WHERE rv_reservation_item_ix = '$rv_reservation_item_id'");
		}
		return true;
	}

	// Look up reservation, accommodation type and allocation information from old items
	$rv_reservation_id = "";
	$ac_accomm_type_id = "";
	$allocations = [];
    $oldItemSeries = false;
	foreach($oldItems as $rv_reservation_item_id) {
		$bl_series_id = $GLOBALS['lDB']->get("
            SELECT DISTINCT
                alloc_item.bl_series_id
            FROM
                rv_reservation_item
                LEFT JOIN rv_reservation_item AS alloc_item ON alloc_item.rv_link_res_item_id = rv_reservation_item.rv_reservation_item_ix
                LEFT JOIN rv_reservation AS alloc ON alloc.rv_reservation_ix = alloc_item.rv_reservation_id
            WHERE
                rv_reservation_item.rv_reservation_item_ix = '".$lDB->escape($rv_reservation_item_id)."'
		",4);
        $oldItemSeries = $bl_series_id;
		$excludeBlockAllocations = true;
		if(!empty($bl_series_id)) {
			$excludeBlockAllocations = false;
        } else {
            $bl_series_id = false;
        }

		$item = $lDB->get("
			SELECT
				rv_reservation_item.rv_reservation_id,
				rv_reservation_item.ac_accomm_type_id,
				rv_reservation_item.rv_item_date_arrive,
				rv_reservation_item.rv_item_date_depart,
				rv_reservation.rv_agent_id
			FROM
				rv_reservation_item
				INNER JOIN rv_reservation ON rv_reservation.rv_reservation_ix = rv_reservation_item.rv_reservation_id
			WHERE
				rv_reservation_item.rv_reservation_item_ix = '" . $lDB->escape($rv_reservation_item_id) . "'

		",1);
		$endArray = explode("-",$item['rv_item_date_depart']);
		$item['rv_item_date_end'] = date("Y-m-d",mktime(0,0,0,$endArray[1],$endArray[2]-1,$endArray[0]));
		if(trim($item['rv_agent_id']) == "") {
			$item['rv_agent_id'] = "0";
		}

		if(!empty($item)) {
			$stock = new Stock($item['ac_accomm_type_id'],$item['rv_item_date_arrive'],$item['rv_item_date_end'],false,$item['rv_agent_id'],true);
			$allocationArray = $stock->ToReservationAllocation($rv_reservation_item_id, $excludeBlockAllocations);
		} else {
			$allocationArray = array();
		}
		foreach($allocationArray as $date=>$allocationsPerDay) {
			foreach($allocationsPerDay as $allocationId=>$allocation) {
				if($allocation['selected'] && !in_array($allocationId, $allocations)) {
					$allocations[] = $allocationId;
				}
			}
		}
		$rv_reservation_id = $item['rv_reservation_id'];
		$ac_accomm_type_id = $item['ac_accomm_type_id'];
	}

	if(empty($rv_reservation_id) || empty($ac_accomm_type_id)) {
		return false;
	}

	// Delete old items
	foreach($oldItems as $rv_reservation_item_id) {
		calDeleteItem($rv_reservation_item_id, $rv_reservation_id, false, false, true, true);
	}

	// Create new items
	foreach($newItems as $item) {
		$itemData = join(":",[
			$ac_accomm_type_id,
			$item['arrive'],
			$item['depart'],
			dateSubtract($item['depart'],$item['arrive']),
			$item['config']
		]);
        calCreateItem(
            $rv_reservation_id,
            $itemData,
            $item['rate'],
            $item['optionals'],
            $item['override'],
            "0",
            "0",
            "external",
            $allocations,
            false,
            false,
            $clearAllBlockAllocations,
            $oldItemSeries,
			true,
			$item['comm']
        );
	}
}

/**
 * clearAllocations() - clear all allocations from a reservation itinerary item
 *		If $clearAllBlockAllocations == true then it will clear all allocations
 *		from the whole reservation that belong to the block booking of $itemId
 * @param string $itemId
 * @param boolean $clearAllBlockAllocations
 **/
function clearAllocations($itemId,$clearAllBlockAllocations=false) {
	global $lDB;

	if(!db_rv_reservation_item_exists($itemId)) {
		return false;
	}

	$reservationDetails = array();
	if ($clearAllBlockAllocations) {
		$reservationDetails = $GLOBALS['lDB']->get("
			SELECT
				draw_down_item.rv_reservation_id,
                bl_block_period.bl_block_period_ix,
                bl_block.bl_block_name
			FROM
				bl_block
				INNER JOIN bl_block_period ON bl_block_period.bl_block_id = bl_block.bl_block_ix
				LEFT JOIN rv_reservation allocation ON allocation.bl_block_period_id = bl_block_period.bl_block_period_ix AND allocation.rv_reservation_type_ind = '5' AND allocation.rf_reservation_status_id = '15'
				LEFT JOIN rv_reservation_item allocation_item ON allocation_item.rv_reservation_id = allocation.rv_reservation_ix
				LEFT JOIN rv_reservation_item draw_down_item ON draw_down_item.rv_reservation_item_ix = allocation_item.rv_link_res_item_id
			WHERE
				draw_down_item.rv_reservation_item_ix = '$itemId'
		",1);
	}
	$allocationsSql = "";
	$reservationItemIds = array();
	if (!empty($reservationDetails)) {
		$reservationItemIds = $GLOBALS['lDB']->get("
			SELECT
				rv_reservation_item.rv_reservation_item_ix
			FROM
				rv_reservation_item
			WHERE
				rv_reservation_item.rv_reservation_id = '". $reservationDetails['rv_reservation_id'] ."'
		",3);
		$allocationsSql = "AND rv_reservation.bl_block_period_id = '" . $reservationDetails['bl_block_period_ix'] ."'";
	} else {
		$reservationItemIds = array($itemId);
	}

	$allocations = $lDB->get("
		SELECT
			rv_reservation_item.rv_reservation_item_ix
		FROM
			rv_reservation_item
			INNER JOIN rv_reservation ON rv_reservation.rv_reservation_ix = rv_reservation_item.rv_reservation_id
		WHERE
			rv_reservation_item.rv_link_res_item_id IN ('". implode("','", $reservationItemIds) ."')
			$allocationsSql
	",3);
    if (!empty($reservationDetails)) {
        $reservationAuditTrail = new AuditTrail($reservationDetails['rv_reservation_id'],TYPE_RESERVATION);
        $reservationAuditTrail->save("Block unlinked (".$reservationDetails['bl_block_name'].")");
    }
	foreach($allocations as $allocItemId) {
		// Clear cache
		$reservation_item_details = $lDB->get("SELECT ac_accomm_type_id, rv_item_date_arrive, rv_item_date_depart FROM rv_reservation_item where rv_reservation_item_ix = '$allocItemId'",1);
		$dates = makeDates($reservation_item_details['rv_item_date_arrive'].":".$reservation_item_details['rv_item_date_depart'],dateSubtract($reservation_item_details['rv_item_date_depart'],$reservation_item_details['rv_item_date_arrive']) + 1);
        if (!empty($dates)) {
            $stock = new Stock($reservation_item_details['ac_accomm_type_id'],$reservation_item_details['rv_item_date_arrive'],$reservation_item_details['rv_item_date_depart']);
            $stock->ClearCache(array('agentAllocation','occupancy'), $dates);
        }

		$resId = $lDB->get("SELECT rv_reservation_id FROM rv_reservation_item WHERE rv_reservation_item_ix = '$allocItemId'",4);
		calDeleteItem($allocItemId,$resId);
	}
}

/**
 * drawAllocations() - Draw down from allocations to a reservation
 *	$edit: whether the relevant reservation item is being edited or not
 * @param string $itemId
 * @param array $allocations
 * @param boolean $edit
 **/
function drawAllocations($itemId,$allocations,$edit=false,$bl_series_id=false, $appliedAllocation=array()) {
	global $lDB;

	clearAllocations($itemId);

	// Allow expired allocations to be used for block drawdowns when the user has allocation override access
	$allowExpired = db_sc_group_get_user_setting("sc_grp_res_ovr_alloc_yn") == "1" && !empty($bl_series_id);

	$itemInfo = $lDB->get("
		SELECT
			rv_reservation.rv_reservation_ix,
			rv_reservation.rv_agent_id,
			rv_reservation_item.ac_accomm_type_id,
			rv_reservation_item.rv_item_date_arrive,
			rv_reservation_item.rv_item_date_depart,
			rv_reservation_item.rv_item_accomm_count,
			rv_reservation_item.rv_item_adult_count,
			rv_reservation_item.rv_item_child_count
		FROM
			rv_reservation_item
			INNER JOIN rv_reservation ON rv_reservation.rv_reservation_ix = rv_reservation_item.rv_reservation_id
		WHERE
			rv_reservation_item_ix = '$itemId'
	",1);
	$resId = $itemInfo['rv_reservation_ix'];
	$agentId = $itemInfo['rv_agent_id'];
	$accommId = $itemInfo['ac_accomm_type_id'];
	$startDate = $itemInfo['rv_item_date_arrive'];
	$departDate = $itemInfo['rv_item_date_depart'];
	$departArray = explode("-",$itemInfo['rv_item_date_depart']);
	$endDate = date("Y-m-d",mktime(0,0,0,$departArray[1],$departArray[2]-1,$departArray[0]));
	$rooms = $itemInfo['rv_item_accomm_count'];
	$adult = $itemInfo['rv_item_adult_count'];
	$child = $itemInfo['rv_item_child_count'];
	$stock = new Stock($accommId,$startDate,$endDate,false,false,$allowExpired,$itemId);
	$allocation = false;
	if ($edit) {
		$ignoreAllocations = $lDB->get("
			SELECT DISTINCT
				allocation_item.rv_reservation_id
			FROM
				bl_block_period
				INNER JOIN rv_reservation allocation ON allocation.bl_block_period_id = bl_block_period.bl_block_period_ix AND allocation.rv_reservation_type_ind = '5' AND allocation.rf_reservation_status_id = '15'
				INNER JOIN rv_reservation_item allocation_item ON allocation_item.rv_reservation_id = allocation.rv_reservation_ix
			WHERE
				allocation.rv_reservation_ix IN ('".join("','", $allocations)."')
		",3);
		$ignoreAllocations = array_merge($ignoreAllocations, $appliedAllocation);
		$allocation = $stock->ToAllocationAgent(false, "", false, $ignoreAllocations);
	} else {
		$allocation = $stock->ToAllocationAgent(false);
	}
	$itemAllocation = array();


	// Get Unit and Adult Rate Groups based on sys code
	$unitID = $lDB->get("SELECT rt_rate_group_ix FROM rt_rate_group WHERE rt_rate_group_sys_code = 1",4);
	$adultID = $lDB->get("SELECT rt_rate_group_ix FROM rt_rate_group WHERE rt_rate_group_sys_code = 2",4);
	$childID = $lDB->get("SELECT rt_rate_group_ix FROM rt_rate_group WHERE rt_rate_group_class = 2",4);

	foreach($stock->ToDate() as $date) {
		$roomsLeft = $rooms;
		if(!array_key_exists($date,$itemAllocation)) {
			$itemAllocation[$date] = array();
		}
		foreach($allocations as $allocId) {
			if($roomsLeft > 0) {
				$avail = isset($allocation[$agentId]) && isset($allocation[$agentId]['allocation_balance'][$allocId]['occupancy'][$date]) ? $allocation[$agentId]['allocation_balance'][$allocId]['occupancy'][$date] : 0;
				if($avail >= $roomsLeft) {
					$use = $roomsLeft;
					$roomsLeft = 0;
				} else {
					$use = $avail;
					$roomsLeft -= $avail;
				}
				$itemAllocation[$date][$allocId] = $use;
			}
		}
	}
	foreach($allocations as $allocId) {
		$newStartDate = null;
    $newRooms = 0;
		foreach($itemAllocation as $date=>$allocItem) {
			if(isset($allocItem[$allocId]) && $newStartDate == null) {
				$newStartDate = $date;
				$newRooms = $allocItem[$allocId];
			}
			if(isset($allocItem[$allocId]) && $newRooms != $allocItem[$allocId]) {
				if($newRooms > 0) {
					$numNights = dateSubtract($date,$newStartDate);
					$itemInfo = join(":",array(
						$accommId,
						$newStartDate,
						$date,
						$numNights,
						join("~",array(
							0 => join("_",array(
								0 => $unitID,
								1 => -$newRooms
							)),
							1 => join("_",array(
								0 => $adultID,
								1 => $adult
							)),
							2 => join("_",array(
								0 => $childID,
								1 => $child
							))
						))
					));
					calCreateItem($allocId,$itemInfo,"",array(),"BLANK","0","0",false,false,$itemId,false,false,$bl_series_id);
				}
				$newStartDate = $date;
				$newRooms = $allocItem[$allocId];
			}
		}
		if($newRooms > 0) {
			$numNights = dateSubtract($date,$newStartDate);
			$itemInfo = join(":",array(
				$accommId,
				$newStartDate,
				$departDate,
				$numNights,
				join("~",array(
						0 => join("_",array(
							0 => $unitID,
							1 => -$newRooms
						)),
						1 => join("_",array(
							0 => $adultID,
							1 => $adult
						)),
						2 => join("_",array(
							0 => $childID,
							1 => $child
						))
					))
				));
			calCreateItem($allocId,$itemInfo,"",array(),"BLANK","0","0",false,false,$itemId,false,false,$bl_series_id);
		}
	}
}

/**
 * calDeleteItem() - Delete an itinerary item
 *		If $clearAllBlockAllocations == true then it will clear all allocations
 *		from the whole reservation that belong to the block booking of $itemId
 *		If $clearCache == true it will clear the cache for the itinerary item
 * @param string $itemId
 * @param string $wizResId
 * @param boolean $clearAllBlockAllocations
 * @param boolean $clearCache
 **/
function calDeleteItem($item,&$wizResId,$clearAllBlockAllocations=false,$clearCache=true,$actionEdit=false,$doNotDelete=false, $doNotDeleteAllocations=false) {
	global $userStatusId;
	global $lDB;

	$itinCount = $GLOBALS['lDB']->get("SELECT COUNT(rv_reservation_item_ix) FROM rv_reservation_item WHERE rv_reservation_id = '$wizResId'",4);

	if ($itinCount == 1 && $actionEdit !== true) {
		return;
	}

	$currentSpecials = db_rv_special_get_specials_by_reservation($wizResId);
	$specialOverride = db_rv_reservation_has_special_override($wizResId);
	if (!empty($currentSpecials)) {
		removeSpecialsFromRes($wizResId); // When a reservation is edited, currently applied specials are removed, to be re-applied/re-evaluated later
	}

	if ($lDB->count("rv_reservation_item","rv_reservation_item_ix",$item,1)) {
		$folioId = $lDB->get("SELECT fn_folio_id FROM rv_reservation_item WHERE rv_reservation_item_ix = '$item'",4);
		$compList = $lDB->get("SELECT rv_res_item_comp.rv_res_item_comp_ix FROM rv_res_item_comp WHERE rv_reservation_item_id = '$item'",3);
		foreach($compList as $comp) {
			$lDB->put("DELETE FROM rv_res_item_comp WHERE rv_res_item_comp.rv_res_item_comp_ix = '$comp'");
			db_rv_res_item_comp_tax_delete_by_comp($comp);
			db_rv_res_item_comp_rate_grp_delete_by_comp($comp);
		}

		$groupList = $lDB->get("SELECT rv_res_item_group_ix FROM rv_res_item_group WHERE rv_reservation_item_id = '$item'",3);
		foreach($groupList as $group) {
			$lDB->put("DELETE FROM rv_res_item_group WHERE rv_res_item_group_ix = '$group'");
		}

		$guestList = $lDB->get("SELECT rv_res_item_guest_ix FROM rv_res_item_guest where rv_reservation_item_id = '$item'",3);
		foreach($guestList as $guest) {
			$lDB->put("DELETE FROM rv_res_item_guest WHERE rv_res_item_guest_ix = '$guest'");
		}
		$cacheComponents = array('occupancy');
		$cacheItemDetails = $lDB->get("SELECT rf_reservation_status_id,rv_link_res_item_id FROM rv_reservation_item INNER JOIN rv_reservation ON rv_reservation.rv_reservation_ix = rv_reservation_item.rv_reservation_id WHERE rv_reservation_item_ix = '$item'",1);
		$rf_reservation_status_id = $cacheItemDetails['rf_reservation_status_id'];
		$rv_link_res_item_id = $cacheItemDetails['rv_link_res_item_id'];
		if (in_array($rf_reservation_status_id,array('15','97')) || !empty($rv_link_res_item_id)) {
			$cacheComponents = array('occupancy','agentAllocation');
		}
		$reservation_item_details = $lDB->get("SELECT ac_accomm_type_id, rv_item_date_arrive, rv_item_date_depart FROM rv_reservation_item where rv_reservation_item_ix = '$item'",1);
		$dates = makeDates($reservation_item_details['rv_item_date_arrive'].":".$reservation_item_details['rv_item_date_depart'],dateSubtract($reservation_item_details['rv_item_date_depart'],$reservation_item_details['rv_item_date_arrive']));
        if (!empty($dates)) {
            if($clearCache) {
                $stock = new Stock($reservation_item_details['ac_accomm_type_id'],$reservation_item_details['rv_item_date_arrive'],$reservation_item_details['rv_item_date_depart']);
                $stock->ClearCache($cacheComponents, $dates);
            }

            if ($clearCache) {
                $reservation_item_details = $lDB->get("SELECT ac_accomm_type_id, rv_item_date_arrive, rv_item_date_depart FROM rv_reservation_item where rv_reservation_item_ix = '$item'",1);
                $dates = makeDates($reservation_item_details['rv_item_date_arrive'].":".$reservation_item_details['rv_item_date_depart'],dateSubtract($reservation_item_details['rv_item_date_depart'],$reservation_item_details['rv_item_date_arrive']) + 1);
                $stock = new Stock($reservation_item_details['ac_accomm_type_id'],$reservation_item_details['rv_item_date_arrive'],$reservation_item_details['rv_item_date_depart']);
                $stock->ClearCache(array('accomm'), $dates);
            }
        }

		if (!$doNotDeleteAllocations) {
			clearAllocations($item, $clearAllBlockAllocations);
		}

		$lDB->put("DELETE FROM rv_reservation_item WHERE rv_reservation_item_ix = '$item'");
		db_rv_res_item_rate_grp_delete_by_res_item($item);

		if (isset($GLOBALS['calendarPage'])) {
			$pad = (int) $lDB->get("SELECT rf_res_create_expiry_mins FROM rf_default",4);
			if ($userStatusId > 1) {
				$pad = (int) $lDB->get("SELECT rf_res_create_expiry_internal FROM rf_default",4);
			}
			$total = $pad * 60;
			$expiry = date("Y-m-d H:i:s",mktime(date("H"),date("i"),date("s")+$total,date("m"),date("d"),date("Y")));
			$lDB->put("UPDATE rv_reservation SET rv_create_expiry_date = '$expiry' WHERE rv_reservation_ix = '$wizResId'");
		}

		// Attempt to remove reservation item folio
		db_fn_folio_delete_by_id($folioId);
		db_fn_folio_update_totals($folioId);

	}
	# Delete res and extras if no more items exist -- Shouldn't this be in the reservation delete code? [DB]
	$noItems = $lDB->count("rv_reservation_item","rv_reservation_id",$wizResId,0);
	if ($noItems && isset($GLOBALS['calendarPage']) && $GLOBALS['calendarPage'] && !$doNotDelete) {
		$list = $lDB->get("SELECT rv_extra_ix FROM rv_extra WHERE rv_reservation_id = '".$wizResId."'",3);
		foreach ($list as $extra) {
			$lDB->put("DELETE FROM rv_extra WHERE rv_extra_ix = '$extra'");
		}

		$list = $lDB->get("SELECT fn_folio_ix FROM fn_folio WHERE rv_reservation_id = '".$wizResId."'",3);
		foreach ($list as $folio) {
			$lDB->put("DELETE FROM fn_folio WHERE fn_folio_ix = '$folio'");
		}
		$payList = $lDB->get("SELECT rv_pay_plan_item_ix FROM rv_pay_plan_item WHERE rv_reservation_id = '$wizResId'",3);
		foreach($payList as $payItem) {
			$lDB->put("DELETE FROM rv_pay_plan_item WHERE rv_pay_plan_item.rv_pay_plan_item_ix = '$payItem'");
		}
		$auditList = $lDB->get("SELECT ad_reservation_ix FROM ad_reservation WHERE rv_reservation_id = '$wizResId'",3);
		foreach($auditList as $auditItem) {
			$lDB->put("DELETE FROM ad_reservation WHERE ad_reservation_ix = '$auditItem'");
		}

		$rv_reservation_db = $lDB->get("
			SELECT
				rv_reservation_db
			FROM
				rv_reservation
			WHERE
				rv_reservation_ix = '$wizResId'
		",4);
		$lDB->put("DELETE FROM rv_reservation WHERE rv_reservation_ix = '$wizResId'");
		db_tc_sequence_reset("rv_reservation", $rv_reservation_db);
		unsetWizResId($wizResId);
	} else {
		recalcTotals($wizResId,4);
		updateReservation($wizResId);
	}
	if (!empty($currentSpecials)) {
		specialsAutoApply($wizResId, $currentSpecials, $specialOverride);
	}
}

function clearTimedOutReservations() {
	global $lDB;
	$tempMasterData = $lDB->isMaster;
	$lDB->isMaster = "1";

	if(array_key_exists('calendarPage',$GLOBALS)) {
		$oldCalendarPage = $GLOBALS['calendarPage'];
	} else {
		$oldCalendarPage = false;
	}
	$GLOBALS['calendarPage'] = true;

	$defaultStartDateTime = date("Y-m-d H:i:s");
	$resItemList = $lDB->get("
		SELECT
			rv_reservation.rv_reservation_ix,
			rv_reservation_item.rv_reservation_item_ix,
			rv_reservation.rf_reservation_status_id
		FROM
			rv_reservation
			LEFT JOIN rv_reservation_item ON rv_reservation_item.rv_reservation_id = rv_reservation.rv_reservation_ix
		WHERE
			rv_reservation.rf_reservation_status_id = '25'
			AND rv_reservation.rv_create_expiry_date < '$defaultStartDateTime'
			AND rv_reservation.rv_reservation_db = '$GLOBALS[dbcode]'
	",2);
	$deleted = array();
	foreach($resItemList as $resItem) {
		if($resItem['rf_reservation_status_id'] != "25") {
			continue;
		}
		$deleted[] = $resItem['rv_reservation_ix'];
		calDeleteItem($resItem['rv_reservation_item_ix'],$resItem['rv_reservation_ix'],false,false,true);
	}
	$GLOBALS['calendarPage'] = $oldCalendarPage;
	$lDB->isMaster = $tempMasterData;

	$deleted = array_unique($deleted);
	if(sizeof($deleted) > 0) {
		error_log("clearTimedOutReservations(): ". join(",",$deleted));
	}
	return $deleted;
}

/**
 * getDefaultOptionalComponents() - get all the default optional components for
 * 		the specified accommodation type and rate type that belong to periods
 *		for the given start and end dates
 * @param string $ac_accomm_type_id
 * @param string $rt_rate_type_id
 * @param string $startDate
 * @param string $endDate
 * @return array $defaultOptionalComponents
 **/
function getDefaultOptionalComponents($ac_accomm_type_id,$rt_rate_type_id,$startDate,$endDate) {
	global $lDB;
	$splitItems = $lDB->get("
		SELECT
			rf_period_split
		FROM
			rf_default
	",4);
	if ($splitItems == "2") {
		$sql = "
			SELECT DISTINCT
				rt_period.rt_period_ix
			FROM
				rt_period
				INNER JOIN rt_rate ON rt_rate.rt_period_id = rt_period.rt_period_ix
				INNER JOIN rt_period_dates ON rt_period_dates.rt_period_id = rt_period.rt_period_ix
			WHERE
				rt_rate.ac_accomm_type_id = '".$ac_accomm_type_id."'
				AND	rt_rate.rt_rate_type_id = '".$rt_rate_type_id."'
				AND	(
					(rt_period_dates.rt_period_from  >= '".$startDate."' AND rt_period_dates.rt_period_from < '".$endDate."')
					OR
					(rt_period_dates.rt_period_to >= '".$startDate."' AND rt_period_dates.rt_period_to <= '".$endDate."')
					OR
					(rt_period_dates.rt_period_from <= '".$startDate."' AND rt_period_dates.rt_period_to >= '".$endDate."')
				)
			ORDER BY
				rt_period_dates.rt_period_from
		";
	} else {
		$sql = "
			SELECT
				rt_period.rt_period_ix
			FROM
				rt_period
				INNER JOIN rt_rate ON rt_rate.rt_period_id = rt_period.rt_period_ix
				INNER JOIN rt_period_dates ON rt_period_dates.rt_period_id = rt_period.rt_period_ix	AND	rt_rate.ac_accomm_type_id = '".$ac_accomm_type_id."'	AND	rt_rate.rt_rate_type_id = '".$rt_rate_type_id."'
			WHERE
				(
				rt_period_dates.rt_period_from <= '".$startDate."'
				AND
				rt_period_dates.rt_period_to >= '".$startDate."'
				)
			ORDER BY
				rt_period_dates.rt_period_from
			LIMIT 1
		";
	}

	$periodList = $lDB->get($sql,3);
	$periodList = array_unique($periodList);
	cleanse($periodList);
	$prdCount = sizeof($periodList);

	$defaultOptionalComponents = $GLOBALS['lDB']->get("
		SELECT
			rt_component.rt_component_ix,
			rt_component.rt_component_abbrv
		FROM
			rt_rate_component
			INNER JOIN rt_rate ON rt_rate.rt_rate_ix = rt_rate_component.rt_rate_id
			INNER JOIN rt_component ON rt_component.rt_component_ix = rt_rate_component.rt_component_id
			INNER JOIN rt_period ON rt_period_ix = rt_rate.rt_period_id
			INNER JOIN rt_rate_type ON rt_rate_type.rt_rate_type_ix = rt_rate.rt_rate_type_id
			INNER JOIN rf_currency ON rf_currency.rf_currency_ix = rt_rate_type.rf_currency_id
		WHERE
			rt_rate_component.rt_comp_opt_yn = '1'
			AND rt_rate_component.rt_comp_opt_def_yn = '1'
			AND	rt_rate.rt_rate_type_id = '".$rt_rate_type_id."'
			AND	rt_rate.ac_accomm_type_id = '".$ac_accomm_type_id."'
			AND rt_rate.rt_period_id IN ('".join("','",$periodList)."')
		ORDER BY
			rt_component.rt_component_desc
	",2);
	return $defaultOptionalComponents;
}

function getReservationTerms($rv_reservation_id) {
	global $lDB;

	$terms = $lDB->get("
		SELECT 
			rv_reservation.rt_rate_type_id,
			rt_rate_type.rt_rate_type_desc,
			rt_rate_type.rf_currency_id,
			rv_reservation.rv_commission_perc
		FROM
			rv_reservation
			INNER JOIN rt_rate_type ON rt_rate_type.rt_rate_type_ix = rv_reservation.rt_rate_type_id
		WHERE
			rv_reservation.rv_reservation_ix = '" . $lDB->escape($rv_reservation_id) . "'
	",1);

	return [
		'rt_rate_type_id' => $terms['rt_rate_type_id'],
		'rt_rate_type_desc' => $terms['rt_rate_type_desc'],
		'rf_currency_id' => $terms['rf_currency_id'],
		'rv_commission_perc' => $terms['rv_commission_perc']
	];
}

function getDefaultTerms($pr_agent_id=false, $rf_currency_id=false) {
	global $lDB;
	global $userStatusId;

	if(!db_pr_agent_exists($pr_agent_id)) {
		$pr_agent_id = false;
	}

	$currencyWhere = "";
	if(db_rf_currency_exists($rf_currency_id)) {
		$currencyWhere = "
			AND rt_rate_type.rf_currency_id = '" . $lDB->escape($rf_currency_id) . "'
		";
	}

	if($userStatusId < 2) {
		$default_rate_type_id = $lDB->get("
			SELECT
				rf_default_rate.rf_default_rate_id
			FROM
				rf_default_rate
				INNER JOIN rt_rate_type ON rt_rate_type.rt_rate_type_ix = rf_default_rate.rf_default_rate_id
			WHERE
				rt_rate_type_inactive_yn = '0'
				$currencyWhere
			ORDER BY
				rt_rate_type.rt_rate_type_desc
		",4);
	} else {
		$default_rate_type_id = $lDB->get("
			SELECT
				rf_default.rt_rate_type_id
			FROM
				rf_default
				INNER JOIN rt_rate_type ON rt_rate_type.rt_rate_type_ix = rf_default.rt_rate_type_id
			WHERE
				1
				$currencyWhere
		",4);
	}
	$default_commission_perc = "0.00";

	$agent_commission_perc = null;
	$agent_rate_type_id = null;
	if($pr_agent_id !== false) {
		$agent_commission_perc = $lDB->get("
			SELECT
				pr_agent.pr_agent_commission_perc
			FROM
				pr_agent
			WHERE
				pr_agent.pr_agent_id = '" . $lDB->escape($pr_agent_id) . "'
		",4);
		$agent_rate_type_id = $lDB->get("
			SELECT
				pr_agent_rate.rt_rate_type_id
			FROM
				pr_agent_rate
				INNER JOIN rt_rate_type ON rt_rate_type.rt_rate_type_ix = pr_agent_rate.rt_rate_type_id
			WHERE
				pr_agent_rate.pr_agent_id = '" . $lDB->escape($pr_agent_id) . "'
				$currencyWhere
			ORDER BY
				pr_agent_rate_default_yn DESC,
				rt_rate_type.rt_rate_type_desc ASC
		",4);
	}
	$rt_rate_type_id = $agent_rate_type_id ?? $default_rate_type_id;
	list($rt_rate_type_desc, $rf_currency_id) = $lDB->get("
		SELECT
			rt_rate_type.rt_rate_type_desc,
			rt_rate_type.rf_currency_id
		FROM
			rt_rate_type
		WHERE
			rt_rate_type.rt_rate_type_ix = '" . $lDB->escape($rt_rate_type_id) . "'
	",1);

	return [
		'rt_rate_type_id' => $rt_rate_type_id,
		'rt_rate_type_desc' => $rt_rate_type_desc,
		'rf_currency_id' => $rf_currency_id,
		'rv_commission_perc' => $agent_commission_perc ?? $default_commission_perc
	];
}

function getAgentPropertyTerms($pr_agent_id=false, $rf_currency_id=false, $pr_business_id=false, $rt_rate_type_id=false, $propertyOnly=false) {
	global $lDB;

	if(!db_pr_agent_exists($pr_agent_id)) {
		$pr_agent_id = false;
	}

	$defaultTerms = getDefaultTerms($pr_agent_id, $rf_currency_id);

	if(empty($rf_currency_id) && !empty($defaultTerms['rf_currency_id'])) {
		$rf_currency_id = $defaultTerms['rf_currency_id'];
	}
	if(!db_rf_currency_exists($rf_currency_id)) {
		return false;
	}

	if(!db_pr_business_exists($pr_business_id)) {
		$pr_business_id = false;
	}

	if($pr_agent_id === false) {
		return [
			'rt_rate_type_id' => $defaultTerms['rt_rate_type_id'],
			'rv_commission_perc' => "0.00"
		];
	}

	if(!db_rt_rate_type_exists($rt_rate_type_id)) {
		$rt_rate_type_id = false;
	}

	$propertyTerms = [
		'rt_rate_type_id' => null,
		'rv_commission_perc' => null
	];

	if($pr_business_id !== false && db_pr_agent_prop_exists_by_currency($pr_agent_id, $pr_business_id, $rf_currency_id)) {
		$propertyTerms = $lDB->get("
			SELECT
				rt_rate_type.rt_rate_type_ix AS rt_rate_type_id,
				pr_agent_prop.pr_agent_prop_comm_perc AS rv_commission_perc
			FROM
				pr_agent_prop
				INNER JOIN rt_rate_type ON
					rt_rate_type.rt_rate_type_ix = pr_agent_prop.rt_rate_type_id
					AND rt_rate_type.rf_currency_id = '" . $lDB->escape($rf_currency_id) . "'
			WHERE
				pr_agent_prop.pr_agent_id = '" . $lDB->escape($pr_agent_id) . "'
				AND pr_agent_prop.pr_business_id = '" . $lDB->escape($pr_business_id) . "'
		",1);
	}

	if($rt_rate_type_id !== false && $propertyTerms['rt_rate_type_id'] != $rt_rate_type_id) {
		$propertyTerms = [
			'rt_rate_type_id' => $rt_rate_type_id,
			'rv_commission_perc' => null
		];
	}

	if(
		is_null($propertyTerms['rv_commission_perc'])
		&& $pr_business_id !== false
		&& db_pr_agent_prop_exists_by_all($pr_agent_id, $pr_business_id)
	) {
		$propertyTerms['rv_commission_perc'] = $lDB->get("
			SELECT
				pr_agent_prop.pr_agent_prop_comm_perc AS rv_commission_perc
			FROM
				pr_agent_prop
				LEFT JOIN rt_rate_type ON rt_rate_type.rt_rate_type_ix = pr_agent_prop.rt_rate_type_id
			WHERE
				pr_agent_prop.pr_agent_id = '" . $lDB->escape($pr_agent_id) . "'
				AND pr_agent_prop.pr_business_id = '" . $lDB->escape($pr_business_id) . "'
				AND rt_rate_type.rt_rate_type_ix IS NULL
		",4);
	}

	if($propertyOnly) {
		$agent_commission_perc = null;
		$agent_rate_type_id = null;
	} else {
		$agent_commission_perc = $defaultTerms['rv_commission_perc'];
		$agent_rate_type_id = $defaultTerms['rt_rate_type_id'];
	}

	return [
		'rt_rate_type_id' => $propertyTerms['rt_rate_type_id'] ?? $agent_rate_type_id,
		'rv_commission_perc' => $propertyTerms['rv_commission_perc'] ?? $agent_commission_perc
	];
}

function calCreateItemAudit($items, $rv_reservation_id=false) {
	global $lDB;

	if(empty($items)) {
		return false;
	}
	if($rv_reservation_id === false) {
		$rv_reservation_id = $lDB->get("
			SELECT
				rv_reservation_item.rv_reservation_id
			FROM
				rv_reservation_item
			WHERE
				rv_reservation_item.rv_reservation_item_ix = '" . $lDB->escape($items[0]) . "'
		",4);
	}
	$reservationAuditTrail = new AuditTrail($rv_reservation_id,TYPE_RESERVATION);
	$moduleFieldValues = array();
	foreach ($items as $resItem) {
		$reservationItem = $GLOBALS['lDB']->get("
			SELECT
				rv_reservation_item.rv_reservation_item_ix,
				rv_reservation_item.rv_item_amt_comm,
				rv_reservation_item.rv_item_amt_gross,
				rv_reservation_item.rv_item_amt_nett,
				rv_reservation_item.rv_item_amt_payable,
				rv_reservation_item.rv_item_amt_tax,
				rv_reservation_item.rv_item_comm_perc,
				rv_reservation_item.rv_item_overide_level_ind,
				rv_reservation_item.rv_item_overide_amt,
				rv_reservation_item.rv_item_date_arrive,
				rv_reservation_item.rv_item_date_depart,
				rv_reservation_item.rv_item_adult_count,
				rv_reservation_item.rv_item_child_count,
				rv_reservation_item.rv_item_exch_rate,
				rv_reservation_item.rv_item_nights,
				rv_reservation_item.rv_item_accomm_count,
				rv_reservation_item.rv_item_note_conf,
				rv_reservation_item.rv_item_note_general,
				rv_reservation_item.rv_item_split_yn,
				rv_reservation_item.rv_item_var_perc,
				rv_reservation_item.rt_rate_type_id,
				rv_reservation_item.ac_accomm_type_id,
				rv_reservation_item.pr_business_id
			FROM
				rv_reservation_item
			WHERE
				rv_reservation_item_ix = '$resItem'
		", 1);
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_amt_comm");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_amt_gross");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_amt_nett");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_amt_payable");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_amt_tax");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_comm_perc");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_overide_level_ind");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_overide_amt");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_date_arrive");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_date_depart");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_adult_count");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_child_count");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_exch_rate");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_nights");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_accomm_count");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_note_conf");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_note_general");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_split_yn");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rv_item_var_perc");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.rt_rate_type_id");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.ac_accomm_type_id");
		$moduleFieldValues[] = array("ad_module_field_name"=>"rv_reservation_item.pr_business_id");
		foreach ($moduleFieldValues as $moduleFieldId=>$moduleFieldValue) {
			$moduleFieldName = explode(".", $moduleFieldValue['ad_module_field_name'])[1];
			$reservationAuditTrail->addDetail(
				"",
				$reservationItem[$moduleFieldName],
				$reservationItem['rv_reservation_item_ix'],
				Null,
				DB_AD_RES_DETAIL_ACTION_ADD,
				"resitinerary",
				$moduleFieldValue['ad_module_field_name']
			);
		}
	}
	$reservationAuditTrail->save("Itinerary (Add)");
	updateReservation($rv_reservation_id);
}
