<?php

/*
 * $Id: api.bridge.php,v 1.1 2013-08-19 17:02:52 light Exp $
 *
 * $Log: api.bridge.php,v $
 * Revision 1.1  2013-08-19 17:02:52  light
 * Bug #8252 - API extensions for TVL
 *
 */

/**
 * api.bridge.php - API bridge specific calls
 */

$this->Register(array(
	'br_principal_list'=>array(
		'doc'=>"Return a list of censys principals.",
		'signature'=>array(
			array("struct", "struct"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'ac_get_principal'=>array(
		'doc'=>"When passed a valid username and password, returns all available principals and their codes. Optionally a third parameter may be passed to search for specific principal names. This search may use * or % as wildcard characters.",
		'signature'=>array(
			array("struct","string","string"),
			array("struct","string","string","string")
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_get_link'=>array(
		'doc'=>"When passed a valid username and password, returns all available principals and their codes. Optionally a third parameter may be passed to search for specific principal names. This search may use * or % as wildcard characters.",
		'signature'=>array(
			array("struct","struct"),
			array("struct","struct","string"),
			array("struct","struct","string","boolean")
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_user_create'=>array(
		'doc'=>"Creates a new bridge account. Parameter 1 (required): name. Parameters 2-8 (optional): bridge username, bridge password, admin user, inactive, note, reference.",
		'signature'=>array(
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
			array("struct", "struct", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_user_link_create'=>array(
		'doc'=>"Create a new bridge user linked to a bridge account. Parameters 1-2 (require one or the other): user id, user name. Parameters 3-4 (optional): link first name, link last name.",
		'signature'=>array(
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
			array("struct", "struct", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_user_list'=>array(
		'doc'=>"Return a list of bridge users based on search criteria. Parameters 1-7 (optional): id, name, username, admin yn, inactive yn, total records, starting record",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
			array("struct", "struct", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_user_update'=>array(
		'doc'=>"Update a user. Parameters 1-2 (require one or the other): user id, username. Parameters 3-9 (optional): name, username, password, admin YN, inactive YN, note, reference",
		'signature'=>array(
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
			array("struct", "struct", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string", "string", "string", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_user_delete'=>array(
		'doc'=>"Delete a user. Parameters 1-2 (require one or the other): user id, username. Parameter 3 (optional): permanent. The user record will be marked as Inactive by default. Optional third parameter will determine if a record is made inactive, or deleted permanently: 0 = deactivate (default), 1 = delete",
		'signature'=>array(
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
			array("struct", "struct", "string", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_user_link_list'=>array(
		'doc'=>"Return a list of bridge user links. Parameters 1-2 (require one or the other): user id, username. Parameters 3-6 (optional): link first name, link last name, total records, starting record",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
			array("struct", "struct", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_link_update'=>array(
		'doc'=>"Update a link. Parameter 1 (required): link id. Parameters 2-3 (optional): link first name, link last name",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
			array("struct", "struct", "string", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_link_delete'=>array(
		'doc'=>"Delete a link. Parameter 1 (required): link id. Parameter 2 (optional): permanent. The user record will be marked as Inactive by default. Optional second parameter will determine if a record is made inactive, or deleted permanently: 0 = deactivate (default), 1 = delete",
		'signature'=>array(
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_user_link_principal_list'=>array(
		'doc'=>"Return a list of bridge user link principals. Parameters 1-2 (require one or the other): account id, account username. Parameters 3-7 (optional): link principal id, username, cache YN, records, starting record",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
			array("struct", "struct", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_user_link_principal_list_reverse'=>array(
		'doc'=>"Return a list of bridge users based on link principal ID. Parameters 1-2 (require one or the other): link principal id, link principal username",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_user_link_principal_create'=>array(
		'doc'=>"Creates a new bridge user link principal. Parameters 1-4 (required): link id, principal ID, username, password. Parameter 5 (optional): cache YN",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
			array("struct", "struct", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_link_principal_update'=>array(
		'doc'=>"Update a link principal. Parameter 1 (required):  link principal id. Parameters 2-5 (optional): principal ID, principal username, principal password, cache YN",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
			array("struct", "struct", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_link_principal_delete'=>array(
		'doc'=>"Delete a link principal. Parameter 1 (required): link principal id. Parameter 2 (optional): permanent. The user record will be marked as Inactive by default. Optional second parameter will determine if a record is made inactive, or deleted permanently: 0 = deactivate (default), 1 = delete",
		'signature'=>array(
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_audit_list'=>array(
		'doc'=>"Returns a list of audit entries. Parameters 1-8 (optional): start date, end date, user ID, description search, from value search, to value search, total records, starting record",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
			array("struct", "struct", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string", "string", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_audit_detail_list'=>array(
		'doc'=>"Returns details of an audit entry. Parameter 1 (required): audit entry ID",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_role_list'=>array(
		'doc'=>"Returns a list of roles. Accepts no parameters",
		'signature'=>array(
			array("struct", "struct"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_role_create'=>array(
		'doc'=>"Creates a role. Parameter 1 (required): name",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_user_role_list'=>array(
		'doc'=>"Returns list of user roles. Parameter 1 (required): user ID",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_user_role_create'=>array(
		'doc'=>"Creates a user role. Parameters 1-2 (required): user ID, role ID",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_user_role_delete'=>array(
		'doc'=>"Delete a user role. Parameters 1-2 (required): user ID, role ID",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_contact_role_list'=>array(
		'doc'=>"Returns a list of contact roles. Accepts no parameters",
		'signature'=>array(
			array("struct", "struct"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_contact_role_create'=>array(
		'doc'=>"Creates a contact role. Parameter 1 (required): name",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_user_contact_list'=>array(
		'doc'=>"Returns list of user contacts. Parameters 1 (required): user ID",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_user_contact_create'=>array(
		'doc'=>"Creates a user contact. Parameters 1-3 (required): user ID, contact name, contact role ID. Parameters 4-6 (optional): email, telephone, note",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
			array("struct", "struct", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_user_contact_update'=>array(
		'doc'=>"Updates a user contact. Parameter 1 (required): user contact ID. Parameters 2-6 (optional): name, role ID, email, telephone, note",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
			array("struct", "struct", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string"),
			array("struct", "struct", "string", "string", "string", "string", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
	'br_user_contact_delete'=>array(
		'doc'=>"Delete a user contact. Parameters 1-2 (required): user ID, contact ID",
		'signature'=>array(
			array("struct", "struct"),
			array("struct", "struct", "string"),
			array("struct", "struct", "string", "string"),
		),
		'forceFunction'=>true,
		'noPrincipal'=>true
	),
));

/**
 * Bridge API call to list censys principals.
 */
function api_br_principal_list() {
	global $api_instance;
	global $config;
	global $lDB;

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$clientDB = $config['db']['client'];

	$results = $lDB->get("
		SELECT
			cn_principal_id AS principal_id,
			cn_prn_name AS principal_name,
			cn_prn_name_short AS principal_shortname
		FROM
			$clientDB.cn_principal
	", 2);

	if(empty($results)) {
		return "No principals found";
	} else {
		return $results;
	}
}

function api_ac_get_principal($search="") {
	global $api_instance;
	global $config;
	global $lDB;

	$searchSQL = "";
	if(trim($search) != "") {
		$searchSQL = "
			AND (
				cn_principal.cn_prn_name LIKE '".addslashes(str_replace("*","%",$search))."'
				OR cn_principal.cn_principal_id LIKE '".addslashes(str_replace("*","%",$search))."'
			)
		";
	}

	$list = $lDB->get("
			SELECT
				cn_principal.cn_principal_id,
				cn_principal.cn_prn_name
			FROM
				".$config['db']['bridge'].".pr_user_link_principal
				INNER JOIN ".$config['db']['client'].".cn_principal ON cn_principal.cn_principal_id = pr_user_link_principal.cn_principal_id
				INNER JOIN ".$config['db']['bridge'].".pr_user_link ON pr_user_link.pr_user_link_id = pr_user_link_principal.pr_user_link_id
			WHERE
				pr_user_link.pr_user_id = '".$api_instance->User."'
				AND pr_user_link.pr_inactive_yn = 0
				AND pr_user_link_principal.pr_inactive_yn = 0
				AND pr_user_link_principal.cn_principal_id <> '*'
				AND pr_user_link_principal.pr_link_username <> '*'
				$searchSQL
	",2);

	$newList = array();
	foreach($list as $item) {
		array_push($newList,array('id'=>$item['cn_principal_id'],'name'=>$item['cn_prn_name']));
	}

	return $newList;	
}

function api_br_get_link($search="", $shortname=false) {
	global $api_instance;
	global $config;
	global $lDB;

	$userSQL = "";
	if(!empty($api_instance->Auth['user_id'])) {
		$userSQL = ' AND pr_user_link.pr_user_link_id = ' . $api_instance->Auth['user_id'] . ' ';
	}

	$searchSQL = "";
	if(trim($search) != "") {
		$searchSQL = "
			AND (
				cn_principal.cn_prn_name LIKE '".addslashes(str_replace("*","%",$search))."'
				OR cn_principal.cn_principal_id LIKE '".addslashes(str_replace("*","%",$search))."'
			)
		";
	}

	$list = $lDB->get("
			SELECT
				pr_user_link_principal.pr_user_link_principal_id,
				pr_user_link_principal.pr_link_username,
				pr_user_link_principal.cn_principal_id,
				cn_principal.cn_prn_name,
				cn_principal.cn_prn_name_short
			FROM
				".$config['db']['bridge'].".pr_user_link_principal
				LEFT JOIN ".$config['db']['client'].".cn_principal ON cn_principal.cn_principal_id = pr_user_link_principal.cn_principal_id
				INNER JOIN ".$config['db']['bridge'].".pr_user_link ON pr_user_link.pr_user_link_id = pr_user_link_principal.pr_user_link_id
			WHERE
				pr_user_link.pr_user_id = '".$api_instance->User."'
				$userSQL
				$searchSQL
	",2);

	$newList = array();
	foreach($list as $item) {
		if(empty($item['cn_prn_name'])) {
			$item['cn_prn_name'] = "";
		}
		$result = [
			'id'=>$item['pr_user_link_principal_id'],
			'principal_id'=>$item['cn_principal_id'],
			'principal_name'=>$item['cn_prn_name'],
			'principal_username'=>$item['pr_link_username']
		];
		if ($shortname) {
			$result['principal_shortname'] = $item['cn_prn_name_short'];
		}
		array_push($newList, $result);
	}

	return $newList;	
}

/**
 * Bridge API call to create a bridge account.
 */
function api_br_user_create($name, $bridgeUsername='', $bridgePassword='', $admin='0', $inactive='0', $bridgeNote='', $bridgeReference='') {
	global $api_instance;
	global $config;
	global $lDB;
	
	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}

	$bridgeDB = $config['db']['bridge'];
	$shortName = substr($name, 0, 23) . '_';

	if(empty($bridgeUsername)) {
		$bridgeUsername = "CONCAT('{$lDB->escape($shortName)}', UUID())";
	} else {
		// If a bridge_username was provided we need to ensure that it is unique within the database
		$result = $lDB->get("SELECT COUNT(*) FROM pr_user WHERE pr_username = '$bridgeUsername'", 4);
		if($result>0) {
			return $api_instance->Error("Username already exists",409);
		}
		$bridgeUsername = "'".$lDB->escape($bridgeUsername)."'";
	}

	if(empty($bridgePassword)) {
		$bridgePassword = "'".bin2hex(openssl_random_pseudo_bytes(32))."'"; 
	} else {
		$bridgePassword = "'".$lDB->escape($bridgePassword)."'";
	}

	$name = "'".$lDB->escape($name)."'";
	$admin = $admin == 1 ? 1 : 0;
	$inactive = $inactive == 1 ? 1 : 0;

	$sql = "INSERT INTO $bridgeDB.pr_user (
		pr_name,
		pr_username,
		pr_password,
		pr_admin_yn,
		pr_inactive_yn,
		ad_create_date,
		pr_user_note,
		pr_user_reference
	) VALUES (
		$name,
		$bridgeUsername,
		$bridgePassword,
		$admin,
		$inactive,
		NOW(),
		'" . $lDB->escape($bridgeNote) . "',
		'" . $lDB->escape($bridgeReference) . "'
	);";

	if($lDB->put($sql)) {
		$result = $lDB->get("
			SELECT
				pr_user_id AS user_id,
				pr_name AS bridge_name,
				pr_username AS bridge_username,
				pr_password AS bridge_password,
				pr_admin_yn AS admin_yn,
				pr_inactive_yn AS inactive_yn,
				pr_user_note AS bridge_note,
				pr_user_reference AS bridge_reference
			FROM
				$bridgeDB.pr_user
			WHERE
				pr_user_id = {$lDB->insert_id}
		", 6);

		$api_instance->auditCreate($lDB->insert_id, "Create user");
		$api_instance->auditAdd("", $result[0]['bridge_name'], "Name", $api_instance::AUDIT_CREATE);
		$api_instance->auditAdd("", $result[0]['bridge_username'], "Username", $api_instance::AUDIT_CREATE);
		$api_instance->auditAdd("", $admin, "Admin YN", $api_instance::AUDIT_CREATE);
		$api_instance->auditAdd("", $inactive, "Inactive YN", $api_instance::AUDIT_CREATE);
		if (!empty($bridgeNote)) {
			$api_instance->auditAdd("", $bridgeNote, "Note", $api_instance::AUDIT_CREATE);
		}
		if (!empty($bridgeReference)) {
			$api_instance->auditAdd("", $bridgeReference, "Reference", $api_instance::AUDIT_CREATE);
		}
		$api_instance->auditSave();

		return $result;
	} else {
		return $api_instance->Error("An error occurred with creating the user",500);
	}
}

/**
 * Bridge API call to create a bridge user.
 */
function api_br_user_link_create($userId='', $userName='', $firstName='', $lastName='') {
	global $api_instance;
	global $config;
	global $lDB;

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	if (empty($userId) && empty($userName)) {
		return $api_instance->Error("ID or Username not specified",400);
	}

	if (empty($userId)) {
		$userId = $lDB->get("SELECT pr_user_id FROM pr_user WHERE pr_username = '$userName'", 4);
		if(!$userId) {
			return $api_instance->Error("Username does not exist",404);
		}
	}

	$userId = $lDB->get("SELECT pr_user_id FROM pr_user WHERE pr_user_id = '$userId'", 4);
	if(!$userId) {
		return $api_instance->Error("User does not exist",404);
	}

	$firstName = empty($firstName) ? 'default_user' : $firstName;
	$lastName = empty($lastName) ? 'default_user' : $lastName;

	$sql = "INSERT INTO $bridgeDB.pr_user_link(pr_name_first, pr_name_last, pr_user_id, ad_create_date)
	VALUES('".$lDB->escape($firstName)."', '".$lDB->escape($lastName)."', ".$userId.", NOW());";

	if($lDB->put($sql)) {
		$result = [
			'user_link_id' => $lDB->insert_id,
			'user_id' => $userId,
			'user_name_first' => $firstName,
			'user_name_last' => $lastName
		];

		$api_instance->auditCreate($userId, "Create user link");
		$api_instance->auditAdd("", $userId, "User ID", $api_instance::AUDIT_CREATE);
		$api_instance->auditAdd("", $firstName, "First Name", $api_instance::AUDIT_CREATE);
		$api_instance->auditAdd("", $lastName, "Last Name", $api_instance::AUDIT_CREATE);
		$api_instance->auditSave();

		return $result;
	} else {
		return $api_instance->Error("An error occurred with creating the user link",500);
	}
}

/**
 * Bridge API call to list bridge users.
 */
function api_br_user_list($id='', $name='', $username='', $adminYn='', $inactiveYn='', $totalRecords='', $startRecord='') {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	$where = array();
	if (!empty($id)) {
		$where[] = "AND pr_user.pr_user_id = " . $id;
	}
	if (!empty($name)) {
		$where[] = "AND pr_user.pr_name LIKE '%" . $name . "%'";
	}
	if (!empty($username)) {
		$where[] = "AND pr_user.pr_username LIKE '%" . $username . "%'";
	}
	if ($adminYn != '') {
		$where[] = "AND pr_user.pr_admin_yn = " . intval($adminYn);
	}
	if ($inactiveYn != '') {
		$where[] = "AND pr_user.pr_inactive_yn = " . intval($inactiveYn);
	}

	$totalRecords = $totalRecords == '' ? 0 : intval($totalRecords);	// zero is no limit on number of results (default)
	$startRecord = $startRecord == '' ? 0 : intval($startRecord);

	$results = $lDB->get("
		SELECT
			pr_user.pr_user_id AS bridge_user_id,
			pr_user.pr_name AS bridge_user_name,
			pr_user.pr_username AS bridge_username,
			pr_user.pr_admin_yn AS admin_yn,
			pr_user.pr_inactive_yn AS inactive_yn,
			pr_user.pr_user_note AS bridge_note,
			pr_user.pr_user_reference AS bridge_reference,
			(
				SELECT
					COUNT(pr_user_link_principal.pr_user_link_principal_id)
				FROM
					pr_user_link
					LEFT JOIN pr_user_link_principal on pr_user_link_principal.pr_user_link_id =  pr_user_link.pr_user_link_id
				WHERE
					pr_user_link.pr_user_id = bridge_user_id
			) AS link_principal_count,
			(
				SELECT
					GROUP_CONCAT(sc_user_role.sc_user_role_name SEPARATOR ',')
				FROM
					pr_user_role
					LEFT JOIN sc_user_role on sc_user_role.sc_user_role_id = pr_user_role.sc_user_role_id
				WHERE
					pr_user_role.pr_user_id = bridge_user_id
			) AS user_roles,
			(
				SELECT
					GROUP_CONCAT(sc_user_role.sc_user_role_id SEPARATOR ',')
				FROM
					pr_user_role
					LEFT JOIN sc_user_role on sc_user_role.sc_user_role_id = pr_user_role.sc_user_role_id
				WHERE
					pr_user_role.pr_user_id = bridge_user_id
			) AS user_role_ids
		FROM
			$bridgeDB.pr_user
		WHERE
			1
			" . join(' ', $where) . ";
	", 2);
	$totalUsers = count($results);

	$record = 0;
	foreach ($results as $key => $result) {
		$results[$key]['record'] = $record . '';
		$results[$key]['total_records'] = $totalUsers . '';
		$record++;
	}

	if ($totalRecords > 0) {
		$results = array_slice($results, $startRecord, $totalRecords);
	}

	if(empty($results)) {
		return "No matching users found";
	} else {
		return $results;
	}
}

/**
 * Bridge API call to update a bridge user.
 */
function api_br_user_update($id, $username='', $name='', $bridgeUsername='', $bridgePassword='', $adminYn='0', $inactiveYn='0', $bridgeNote='', $bridgeReference='') {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	$where = array();
	if (!empty($id)) {
		$where[] = "AND pr_user_id = " . $id;
	}
	if (!empty($username)) {
		$where[] = "AND pr_username = '" . $username . "'";
	}

	if (empty($where)) {
		return $api_instance->Error("ID or Username not specified",400);
	}

	$record = $lDB->get("
		SELECT
			pr_user_id,
			pr_name,
			pr_username,
			pr_password,
			pr_admin_yn,
			pr_inactive_yn,
			pr_user_note,
			pr_user_reference
		FROM
			$bridgeDB.pr_user
		WHERE
			1
			" . join(' ', $where) . "
	", 1);
	if(empty($record)) {
		return $api_instance->Error("User does not exist",404);
	}

	$api_instance->auditCreate($record['pr_user_id'], "Update user");

	$changes = array();
	if (!empty($name) && $name != $record['pr_name']) {
		$changes[] = "pr_user.pr_name = '" . $name . "'";
		$api_instance->auditAdd($record['pr_name'], $name, "Name", $api_instance::AUDIT_EDIT);
	}
	if (!empty($bridgeUsername) && $bridgeUsername != $record['pr_username']) {
		$changes[] = "pr_user.pr_username = '" . $bridgeUsername . "'";
		$api_instance->auditAdd($record['pr_username'], $bridgeUsername, "Username", $api_instance::AUDIT_EDIT);
	}
	if (!empty($bridgePassword) && $bridgePassword != $record['pr_password']) {
		$changes[] = "pr_user.pr_password = '" . $bridgePassword . "'";
		$api_instance->auditAdd("********", "********", "Password", $api_instance::AUDIT_EDIT);
	}
	if ($adminYn != '' && intval($adminYn) != $record['pr_admin_yn']) {
		$changes[] = "pr_user.pr_admin_yn = '" . intval($adminYn) . "'";
		$api_instance->auditAdd($record['pr_admin_yn'], $adminYn, "Admin YN", $api_instance::AUDIT_EDIT);
	}
	if ($inactiveYn != '' && intval($inactiveYn) != $record['pr_inactive_yn']) {
		$changes[] = "pr_user.pr_inactive_yn = '" . intval($inactiveYn) . "'";
		$api_instance->auditAdd($record['pr_inactive_yn'], $inactiveYn, "Inactive YN", $api_instance::AUDIT_EDIT);
	}
	if (!empty($bridgeNote) && $bridgeNote != $record['pr_user_note']) {
		$changes[] = "pr_user.pr_user_note = '" . $bridgeNote . "'";
		$api_instance->auditAdd($record['pr_user_note'], $bridgeNote, "Note", $api_instance::AUDIT_EDIT);
	}
	if (!empty($bridgeReference) && $bridgeReference != $record['pr_user_reference']) {
		$changes[] = "pr_user.pr_user_reference = '" . $bridgeReference . "'";
		$api_instance->auditAdd($record['pr_user_reference'], $bridgeReference, "Reference", $api_instance::AUDIT_EDIT);
	}

	if (empty($changes)) {
		return $api_instance->Error("No changes specified",400);
	}

	$sql = "
		UPDATE
			$bridgeDB.pr_user
		SET
			" . join(', ', $changes) . "
		WHERE
			pr_user.pr_user_id = " . $record['pr_user_id'] . ";
	";

	if($lDB->put($sql)) {
		$api_instance->auditSave();
		$result = $lDB->get("
			SELECT
				pr_user_id AS user_id,
				pr_name AS bridge_name,
				pr_username AS bridge_username,
				pr_admin_yn AS admin_yn,
				pr_inactive_yn AS inactive_yn,
				pr_user_note AS bridge_note,
				pr_user_reference AS bridge_reference
			FROM
				$bridgeDB.pr_user
			WHERE
				pr_user_id = " . $record['pr_user_id'] . ";
		", 6);
		return $result;
	} else {
		return $api_instance->Error("An error occurred with editing the user",500);
	}
}

/**
 * Bridge API call to delete a bridge user.
 */
function api_br_user_delete($id, $username='', $permanent=0) {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';
	$permanent = intval($permanent) == 1 ? true : false;

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	$where = array();
	if (!empty($id)) {
		$where[] = "AND pr_user_id = " . $id;
	}
	if (!empty($username)) {
		$where[] = "AND pr_username = '" . $username . "'";
	}

	if (empty($where)) {
		return $api_instance->Error("ID or Username not specified",400);
	}

	$record = $lDB->get("
		SELECT
			pr_user_id,
			pr_name,
			pr_username,
			pr_inactive_yn
		FROM
			$bridgeDB.pr_user
		WHERE
			1
			" . join(' ', $where) . "
	", 1);
	if(empty($record)) {
		return $api_instance->Error("User does not exist",404);
	}
	if($record['pr_inactive_yn'] == 1 && !$permanent) {
		return $api_instance->Error("User already deactivated",409);
	}

	$userLink = $lDB->get("
		SELECT
			pr_name_first,
			pr_name_last,
			pr_inactive_yn
		FROM
			$bridgeDB.pr_user_link
		WHERE
			pr_user_link.pr_user_id = " . $record['pr_user_id'] . ";
	", 2);

	$userLinkPrincipal = $lDB->get("
		SELECT
			pr_user_link_principal.pr_link_username,
			pr_user_link_principal.cn_principal_id,
			pr_user_link_principal.pr_inactive_yn
		FROM
			$bridgeDB.pr_user_link
			LEFT JOIN $bridgeDB.pr_user_link_principal ON pr_user_link_principal.pr_user_link_id = pr_user_link.pr_user_link_id
		WHERE
			pr_user_link.pr_user_id = " . $record['pr_user_id'] . ";
	", 2);

	if ($permanent) {
		$sql = "
			DELETE
				pr_user,
				pr_user_link,
				pr_user_link_principal
			FROM
				$bridgeDB.pr_user
				LEFT JOIN $bridgeDB.pr_user_link ON pr_user_link.pr_user_id = pr_user.pr_user_id
				LEFT JOIN $bridgeDB.pr_user_link_principal ON pr_user_link_principal.pr_user_link_id = pr_user_link.pr_user_link_id
			WHERE
				pr_user.pr_user_id = " . $record['pr_user_id'] . ";
		";
		$api_instance->auditCreate($record['pr_user_id'], "Delete user");
		$deleteMessage = "User deleted permanently";
	} else {
		$sql = "
			UPDATE
				$bridgeDB.pr_user
				LEFT JOIN $bridgeDB.pr_user_link ON pr_user_link.pr_user_id = pr_user.pr_user_id
				LEFT JOIN $bridgeDB.pr_user_link_principal ON pr_user_link_principal.pr_user_link_id = pr_user_link.pr_user_link_id
			SET
				pr_user.pr_inactive_yn = 1,
				pr_user_link.pr_inactive_yn = 1,
				pr_user_link_principal.pr_inactive_yn = 1
			WHERE
				pr_user.pr_user_id = " . $record['pr_user_id'] . ";
		";
		$api_instance->auditCreate($record['pr_user_id'], "Deactivate user");
		$deleteMessage = "User deactivated";
	}

	if($lDB->put($sql)) {
		$api_instance->auditAdd($record['pr_name'], "", "Name", $api_instance::AUDIT_DELETE);
		$api_instance->auditAdd($record['pr_username'], "", "Username", $api_instance::AUDIT_DELETE);
		$api_instance->auditAdd($record['pr_user_id'], "", "User ID", $api_instance::AUDIT_DELETE);
		if (!$permanent) {
			$api_instance->auditAdd($record['pr_inactive_yn'], "1", "User inactive", $api_instance::AUDIT_DELETE);
		}

		foreach ($userLink as $link) {
			$api_instance->auditAdd($link['pr_name_first'], "", "User link first name", $api_instance::AUDIT_DELETE);
			$api_instance->auditAdd($link['pr_name_last'], "", "User link last name", $api_instance::AUDIT_DELETE);
			if (!$permanent) {
				$api_instance->auditAdd($link['pr_inactive_yn'], "1", "User link inactive", $api_instance::AUDIT_DELETE);
			}
		}

		foreach ($userLinkPrincipal as $principal) {
			$api_instance->auditAdd($principal['pr_link_username'], "", "User link principal username", $api_instance::AUDIT_DELETE);
			$api_instance->auditAdd($principal['cn_principal_id'], "", "User link principal id", $api_instance::AUDIT_DELETE);
			if (!$permanent) {
				$api_instance->auditAdd($link['pr_inactive_yn'], "1", "User link principal inactive", $api_instance::AUDIT_DELETE);
			}
		}
		$api_instance->auditSave();

		return $deleteMessage;
	} else {
		return $api_instance->Error("An error occurred with deleting the user",500);
	}
}

/**
 * Bridge API call to list bridge user links.
 */
function api_br_user_link_list($id, $username='', $pr_name_first='', $pr_name_last='', $totalRecords='', $startRecord='') {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	$where = array();
	if (!empty($id)) {
		$where[] = "AND pr_user.pr_user_id = " . $id;
	}
	if (!empty($username)) {
		$where[] = "AND pr_user.pr_username = '" . $username . "'";
	}

	if (empty($where)) {
		return $api_instance->Error("ID or Username not specified",400);
	}

	if (!empty($pr_name_first)) {
		$where[] = "AND pr_user_link.pr_name_first LIKE '%" . $pr_name_first . "%'";
	}
	if (!empty($pr_name_last)) {
		$where[] = "AND pr_user_link.pr_name_last LIKE '%" . $pr_name_last . "%'";
	}

	$totalRecords = $totalRecords == '' ? 0 : intval($totalRecords);	// zero is no limit on number of results (default)
	$startRecord = $startRecord == '' ? 0 : intval($startRecord);

	$results = $lDB->get("
		SELECT
			pr_user_link.pr_user_link_id AS link_id,
			pr_user_link.pr_name_first AS link_name,
			pr_user_link.pr_name_last AS link_username,
			pr_user_link.pr_inactive_yn AS inactive_yn
		FROM
			$bridgeDB.pr_user
			LEFT JOIN $bridgeDB.pr_user_link ON pr_user_link.pr_user_id = pr_user.pr_user_id
		WHERE
			1
			" . join(' ', $where) . "
	", 2);
	$totalUsers = count($results);

	$record = 0;
	foreach ($results as $key => $result) {
		$results[$key]['record'] = $record . '';
		$results[$key]['total_records'] = $totalUsers . '';
		$record++;
	}

	if ($totalRecords > 0) {
		$results = array_slice($results, $startRecord, $totalRecords);
	}

	if(empty($results)) {
		return "No matching links found";
	} else {
		return $results;
	}
}

/**
 * Bridge API call to update a bridge user link.
 */
function api_br_link_update($id, $pr_name_first='', $pr_name_last='') {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	$where = array();
	if (!empty($id)) {
		$where[] = "AND pr_user_link_id = " . $id;
	}

	if (empty($where)) {
		return $api_instance->Error("Link ID not specified",400);
	}

	$record = $lDB->get("
		SELECT
			pr_name_first,
			pr_name_last,
			pr_user_id
		FROM
			$bridgeDB.pr_user_link
		WHERE
			1
			" . join(' ', $where) . "
	", 1);
	if(empty($record)) {
		return $api_instance->Error("Link does not exist",404);
	}

	$api_instance->auditCreate($record['pr_user_id'], "Update link");

	$changes = array();
	if (!empty($pr_name_first) && $pr_name_first != $record['pr_name_first']) {
		$changes[] = "pr_user_link.pr_name_first = '" . $pr_name_first . "'";
		$api_instance->auditAdd($record['pr_name_first'], $pr_name_first, "First name", $api_instance::AUDIT_EDIT);
	}
	if (!empty($pr_name_last) && $pr_name_last != $record['pr_name_last']) {
		$changes[] = "pr_user_link.pr_name_last = '" . $pr_name_last . "'";
		$api_instance->auditAdd($record['pr_name_last'], $pr_name_last, "Last Name", $api_instance::AUDIT_EDIT);
	}

	if (empty($changes)) {
		return $api_instance->Error("No changes specified",400);
	}

	$sql = "
		UPDATE
			$bridgeDB.pr_user_link
		SET
			" . join(', ', $changes) . "
		WHERE
			pr_user_link.pr_user_link_id = " . $id . ";
	";

	if($lDB->put($sql)) {
		$api_instance->auditSave();
		$result = $lDB->get("
			SELECT
				pr_user_link.pr_user_link_id AS link_id,
				pr_user_link.pr_name_first AS link_first_name,
				pr_user_link.pr_name_last AS link_last_name
			FROM
				$bridgeDB.pr_user_link
			WHERE
				pr_user_link.pr_user_link_id = " . $id . ";
		", 1);
		return $result;
	} else {
		return $api_instance->Error("An error occurred with editing the link",500);
	}
}

/**
 * Bridge API call to delete a bridge user link.
 */
function api_br_link_delete($id, $permanent=0) {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';
	$permanent = intval($permanent) == 1 ? true : false;

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	$where = array();
	if (!empty($id)) {
		$where[] = "AND pr_user_link_id = " . $id;
	}

	if (empty($where)) {
		return $api_instance->Error("Link ID not specified",400);
	}

	$record = $lDB->get("
		SELECT
			pr_user_link_id,
			pr_name_first,
			pr_name_last,
			pr_user_id,
			pr_inactive_yn
		FROM
			$bridgeDB.pr_user_link
		WHERE
			1
			" . join(' ', $where) . "
	", 1);
	if(empty($record)) {
		return $api_instance->Error("Link does not exist",404);
	}
	if($record['pr_inactive_yn'] == 1 && !$permanent) {
		return $api_instance->Error("Link already deactivated",409);
	}

	$userLinkPrincipal = $lDB->get("
		SELECT
			pr_user_link_principal.pr_link_username,
			pr_user_link_principal.cn_principal_id,
			pr_user_link_principal.pr_inactive_yn
		FROM
			$bridgeDB.pr_user_link
			LEFT JOIN $bridgeDB.pr_user_link_principal ON pr_user_link_principal.pr_user_link_id = pr_user_link.pr_user_link_id
		WHERE
			pr_user_link.pr_user_link_id = " . $id . ";
	", 2);

	if ($permanent) {
		$sql = "
			DELETE
				pr_user_link,
				pr_user_link_principal
			FROM
				$bridgeDB.pr_user_link
				LEFT JOIN $bridgeDB.pr_user_link_principal ON pr_user_link_principal.pr_user_link_id = pr_user_link.pr_user_link_id
			WHERE
				pr_user_link.pr_user_link_id = " . $id . ";
		";
		$api_instance->auditCreate($record['pr_user_id'], "Delete user link");
		$deleteMessage = "User link deleted permanently";
	} else {
		$sql = "
			UPDATE
				$bridgeDB.pr_user_link
				LEFT JOIN $bridgeDB.pr_user_link_principal ON pr_user_link_principal.pr_user_link_id = pr_user_link.pr_user_link_id
			SET
				pr_user_link.pr_inactive_yn = 1,
				pr_user_link_principal.pr_inactive_yn = 1
			WHERE
				pr_user_link.pr_user_link_id = " . $id . ";
		";
		$api_instance->auditCreate($record['pr_user_id'], "Deactivate user link");
		$deleteMessage = "User link deactivated";
	}

	if($lDB->put($sql)) {
		$api_instance->auditAdd($record['pr_name_first'], "", "User link first name", $api_instance::AUDIT_DELETE);
		$api_instance->auditAdd($record['pr_name_last'], "", "User link last name", $api_instance::AUDIT_DELETE);
		if (!$permanent) {
			$api_instance->auditAdd($record['pr_inactive_yn'], "1", "User link inactive", $api_instance::AUDIT_DELETE);
		}

		foreach ($userLinkPrincipal as $principal) {
			$api_instance->auditAdd($principal['pr_link_username'], "", "User link principal username", $api_instance::AUDIT_DELETE);
			$api_instance->auditAdd($principal['cn_principal_id'], "", "User link principal id", $api_instance::AUDIT_DELETE);
			if (!$permanent) {
				$api_instance->auditAdd($link['pr_inactive_yn'], "1", "User link principal inactive", $api_instance::AUDIT_DELETE);
			}
		}
		$api_instance->auditSave();

		return $deleteMessage;
	} else {
		return $api_instance->Error("An error occurred with deleting the user link",500);
	}
}

/**
 * Bridge API call to list bridge user links.
 */
function api_br_user_link_principal_list($id, $username='', $cn_principal_id='', $pr_link_username='', $pr_cache_yn=0, $totalRecords='', $startRecord='') {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	$where = array();
	if (!empty($id)) {
		$where[] = "AND pr_user.pr_user_id = " . $id;
	}
	if (!empty($username)) {
		$where[] = "AND pr_user.pr_username = '" . $username . "'";
	}

	if (empty($where)) {
		return $api_instance->Error("ID or Username not specified",400);
	}

	if (!empty($cn_principal_id)) {
		$where[] = "AND pr_user_link_principal.cn_principal_id LIKE '%" . $cn_principal_id . "%'";
	}
	if (!empty($pr_link_username)) {
		$where[] = "AND pr_user_link_principal.pr_link_username LIKE '%" . $pr_link_username . "%'";
	}
	if ($pr_cache_yn != '') {
		$where[] = "AND pr_user_link_principal.pr_cache_yn = " . intval($pr_cache_yn);
	}

	$totalRecords = $totalRecords == '' ? 0 : intval(@$totalRecords);	// zero is no limit on number of results (default)
	$startRecord = $startRecord == '' ? 0 : intval($startRecord);

	$results = $lDB->get("
		SELECT
			pr_user_link_principal.pr_user_link_principal_id AS record_id,
			pr_user.pr_user_id AS user_id,
			pr_user_link_principal.cn_principal_id AS principal_id,
			pr_user_link_principal.pr_link_username AS principal_username,
			pr_user_link_principal.pr_cache_yn AS cache_yn,
			pr_user_link_principal.pr_inactive_yn AS inactive_yn
		FROM
			$bridgeDB.pr_user
			LEFT JOIN $bridgeDB.pr_user_link ON pr_user_link.pr_user_id = pr_user.pr_user_id
			LEFT JOIN $bridgeDB.pr_user_link_principal ON pr_user_link_principal.pr_user_link_id = pr_user_link.pr_user_link_id
		WHERE
			1
			" . join(' ', $where) . "
			AND pr_user_link_principal.cn_principal_id IS NOT NULL
	", 2);

	$totalUsers = count($results);

	$record = 0;
	foreach ($results as $key => $result) {
		$results[$key]['record'] = $record . '';
		$results[$key]['total_records'] = $totalUsers . '';
		$record++;
	}

	if ($totalRecords > 0) {
		$results = array_slice($results, $startRecord, $totalRecords);
	}

	if(empty($results)) {
		return "No matching link principals found";
	} else {
		return $results;
	}
}

/**
 * Bridge API call to list bridge user links.
 */
function api_br_user_link_principal_list_reverse($cn_principal_id='', $pr_link_username='') {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	$where = array();

	if (!empty($cn_principal_id)) {
		$where[] = "AND pr_user_link_principal.cn_principal_id = " . $cn_principal_id . "";
	}
	if (!empty($pr_link_username)) {
		$where[] = "AND pr_user_link_principal.pr_link_username = '" . $pr_link_username . "'";
	}

	if (empty($where)) {
		return $api_instance->Error("Principal ID or Username not specified",400);
	}

	$results = $lDB->get("
		SELECT
			pr_user_link_principal.pr_user_link_principal_id AS user_link_principal_id,
			pr_user.pr_user_id AS bridge_user_id,
			pr_user.pr_name AS bridge_user_name,
			pr_user_link_principal.cn_principal_id AS principal_id,
			pr_user_link_principal.pr_link_username AS principal_username,
			pr_user_link_principal.pr_cache_yn AS cache_yn,
			pr_user_link_principal.pr_inactive_yn AS inactive_yn,
			(
				SELECT
					GROUP_CONCAT(sc_user_role.sc_user_role_name SEPARATOR ',')
				FROM
					pr_user_role
					LEFT JOIN sc_user_role on sc_user_role.sc_user_role_id = pr_user_role.sc_user_role_id
				WHERE
					pr_user_role.pr_user_id = bridge_user_id
			) AS user_roles
		FROM
			$bridgeDB.pr_user
			LEFT JOIN $bridgeDB.pr_user_link ON pr_user_link.pr_user_id = pr_user.pr_user_id
			LEFT JOIN $bridgeDB.pr_user_link_principal ON pr_user_link_principal.pr_user_link_id = pr_user_link.pr_user_link_id
		WHERE
			1
			" . join(' ', $where) . "
			AND pr_user_link_principal.cn_principal_id IS NOT NULL
	", 2);

	if(empty($results)) {
		return "No matching link principals found";
	} else {
		return $results;
	}
}

/**
 * Bridge API call to create a bridge user link principal.
 */
function api_br_user_link_principal_create($id, $cn_principal_id='', $pr_link_username='', $pr_link_password='', $pr_cache_yn='0') {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';
	
	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	$pr_user_link_id = $lDB->get("SELECT pr_user_link_id FROM pr_user_link WHERE pr_user_link_id = " . $id, 4);
	if(!$pr_user_link_id) {
		return $api_instance->Error("Link does not exist",404);
	}
	$user_id = $lDB->get("SELECT pr_user_id FROM pr_user_link WHERE pr_user_link_id = " . $id, 4);

	if ($cn_principal_id == '' || $pr_link_username == '' || $pr_link_password == '') {
		return $api_instance->Error("Principal ID, username or password not specified",400);
	}

	$sql = "
		INSERT INTO $bridgeDB.pr_user_link_principal (
			cn_principal_id,
			pr_link_username,
			pr_link_password,
			pr_cache_yn,
			pr_user_link_id,
			pr_inactive_yn
		) VALUES (
			" . $lDB->escape(intval($cn_principal_id)) . ",
			'" . $lDB->escape($pr_link_username) . "',
			'" . $lDB->escape($pr_link_password) . "',
			" . intval($pr_cache_yn) . ",
			" . intval($pr_user_link_id) . ",
			0
		);
	";

	if($lDB->put($sql)) {
		$result = [
			'user_link_principal_id' => $lDB->insert_id,
			'cn_principal_id' => $cn_principal_id,
			'pr_link_username' => $pr_link_username,
			'pr_cache_yn' => $pr_cache_yn,
			'pr_user_link_id' => $pr_user_link_id
		];

		$api_instance->auditCreate($user_id, "Create user link principal");
		$api_instance->auditAdd("", $cn_principal_id, "Principal ID", $api_instance::AUDIT_CREATE);
		$api_instance->auditAdd("", $pr_link_username, "Principal username", $api_instance::AUDIT_CREATE);
		$api_instance->auditAdd("", $pr_cache_yn, "Cache YN", $api_instance::AUDIT_CREATE);
		$api_instance->auditSave();

		return $result;
	} else {
		return $api_instance->Error("An error occurred with creating the user link principal",500);
	}
}

/**
 * Bridge API call to update a bridge user link principal.
 */
function api_br_link_principal_update($id, $cn_principal_id='', $pr_link_username='', $pr_link_password='', $pr_cache_yn='0') {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	$where = array();
	if (!empty($id)) {
		$where[] = "AND pr_user_link_principal_id = " . $id;
	}

	if (empty($where)) {
		return $api_instance->Error("Link ID not specified",400);
	}

	$record = $lDB->get("
		SELECT
			pr_user_link_principal.cn_principal_id,
			pr_user_link_principal.pr_link_username,
			pr_user_link_principal.pr_link_password,
			pr_user_link_principal.pr_cache_yn,
			pr_user_link.pr_user_id
		FROM
			$bridgeDB.pr_user_link_principal
			LEFT JOIN $bridgeDB.pr_user_link ON pr_user_link.pr_user_link_id = pr_user_link_principal.pr_user_link_id
		WHERE
			1
			" . join(' ', $where) . "
	", 1);
	if(empty($record)) {
		return $api_instance->Error("Link principal does not exist",404);
	}

	$api_instance->auditCreate($record['pr_user_id'], "Update link principal");

	$changes = array();
	if (!empty($cn_principal_id) && $cn_principal_id != $record['cn_principal_id']) {
		$changes[] = "pr_user_link_principal.cn_principal_id = '" . $cn_principal_id . "'";
		$api_instance->auditAdd($record['cn_principal_id'], $cn_principal_id, "Principal ID", $api_instance::AUDIT_EDIT);
	}
	if (!empty($pr_link_username) && $pr_link_username != $record['pr_link_username']) {
		$changes[] = "pr_user_link_principal.pr_link_username = '" . $pr_link_username . "'";
		$api_instance->auditAdd($record['pr_link_username'], $pr_link_username, "Username", $api_instance::AUDIT_EDIT);
	}
	if (!empty($pr_link_password) && $pr_link_password != $record['pr_link_password']) {
		$changes[] = "pr_user_link_principal.pr_link_password = '" . $pr_link_password . "'";
		$api_instance->auditAdd("********", "********", "Password", $api_instance::AUDIT_EDIT);
	}
	if ($pr_cache_yn != '' && intval($pr_cache_yn) != $record['pr_cache_yn']) {
		$changes[] = "pr_user_link_principal.pr_cache_yn = '" . intval($pr_cache_yn) . "'";
		$api_instance->auditAdd($record['pr_cache_yn'], $pr_cache_yn, "Cache YN", $api_instance::AUDIT_EDIT);
	}

	if (empty($changes)) {
		return $api_instance->Error("No changes specified",400);
	}

	$sql = "
		UPDATE
			$bridgeDB.pr_user_link_principal
		SET
			" . join(', ', $changes) . "
		WHERE
			pr_user_link_principal.pr_user_link_principal_id = " . $id . ";
	";

	if($lDB->put($sql)) {
		$api_instance->auditSave();
		$result = $lDB->get("
			SELECT
				pr_user_link_principal.cn_principal_id AS principal_id,
				pr_user_link_principal.pr_link_username AS principal_username,
				pr_user_link_principal.pr_cache_yn AS cache_yn
			FROM
				$bridgeDB.pr_user_link_principal
			WHERE
				pr_user_link_principal.pr_user_link_principal_id = " . $id . ";
		", 1);
		return $result;
	} else {
		return $api_instance->Error("An error occurred with editing the link principal",500);
	}
}

/**
 * Bridge API call to delete a bridge user link principal.
 */
function api_br_link_principal_delete($id, $permanent=0) {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';
	$permanent = intval($permanent) == 1 ? true : false;

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	$where = array();
	if (!empty($id)) {
		$where[] = "AND pr_user_link_principal.pr_user_link_principal_id = " . $id;
	}

	if (empty($where)) {
		return $api_instance->Error("Link principal ID not specified",400);
	}

	$record = $lDB->get("
		SELECT
			pr_user_link_principal.pr_link_username,
			pr_user_link_principal.cn_principal_id,
			pr_user_link_principal.pr_inactive_yn,
			pr_user_link_principal.pr_cache_yn,
			pr_user_link.pr_user_id
		FROM
			$bridgeDB.pr_user_link_principal
			LEFT JOIN $bridgeDB.pr_user_link ON pr_user_link.pr_user_link_id = pr_user_link_principal.pr_user_link_id
		WHERE
			1
			" . join(' ', $where) . "
	", 1);

	if(empty($record)) {
		return $api_instance->Error("Link principal does not exist",404);
	}
	if($record['pr_inactive_yn'] == 1 && !$permanent) {
		return $api_instance->Error("Link principal already deactivated",409);
	}

	if ($permanent) {
		$sql = "
			DELETE
				pr_user_link_principal
			FROM
				$bridgeDB.pr_user_link_principal
			WHERE
				pr_user_link_principal.pr_user_link_principal_id = " . $id . ";
		";
		$api_instance->auditCreate($record['pr_user_id'], "Delete user link principal");
		$deleteMessage = "User link principal deleted permanently";
	} else {
		$sql = "
			UPDATE
				$bridgeDB.pr_user_link_principal
			SET
				pr_user_link_principal.pr_inactive_yn = 1
			WHERE
				pr_user_link_principal.pr_user_link_principal_id = " . $id . ";
		";
		$api_instance->auditCreate($record['pr_user_id'], "Deactivate user link principal");
		$deleteMessage = "User link principal deactivated";
	}

	if($lDB->put($sql)) {
		$api_instance->auditAdd($record['cn_principal_id'], "", "User link principal id", $api_instance::AUDIT_DELETE);
		$api_instance->auditAdd($record['pr_link_username'], "", "User link principal username", $api_instance::AUDIT_DELETE);
		$api_instance->auditAdd($record['pr_cache_yn'], "", "User link principal cache", $api_instance::AUDIT_DELETE);
		if (!$permanent) {
			$api_instance->auditAdd($record['pr_inactive_yn'], "1", "User link principal inactive", $api_instance::AUDIT_DELETE);
		}
		$api_instance->auditSave();

		return $deleteMessage;
	} else {
		return $api_instance->Error("An error occurred with deleting the user link principal",500);
	}
}

/**
 * Bridge API call to list bridge audit entries.
 */
function api_br_audit_list($date_start='', $date_end='', $pr_user_id='', $desc='', $fromValue='', $toValue='', $totalRecords='', $startRecord='') {
	global $api_instance;
	global $config;
	global $lDB;

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	$where = array();
	$whereOR = array();
	if (!empty($date_start)) {
		$where[] = "AND DATE(ad_user.ad_user_time) >= DATE('" . $date_start . "')";
	}
	if (!empty($date_end)) {
		$where[] = "AND DATE(ad_user.ad_user_time) <= DATE('" . $date_end . "')";
	}
	if (!empty($pr_user_id)) {
		$where[] = "AND ad_user.pr_user_id = " . $pr_user_id;
	}
	if (!empty($desc)) {
		$whereOR[] = "ad_user.ad_user_desc LIKE '%" . $desc . "%'";
	}
	if (!empty($fromValue)) {
		$whereOR[] = "ad_user_detail.ad_user_detail_value_from LIKE '%" . $fromValue . "%'";
	}
	if (!empty($toValue)) {
		$whereOR[] = "ad_user_detail.ad_user_detail_value_to LIKE '%" . $toValue . "%'";
	}

	if (!empty($whereOR)) {
		$where[] = "AND (" . join(' OR ', $whereOR) . ")";
	}

	$totalRecords = $totalRecords == '' ? 0 : intval($totalRecords);	// zero is no limit on number of results (default)
	$startRecord = $startRecord == '' ? 0 : intval($startRecord);

	$results = $lDB->get("
		SELECT DISTINCT
			ad_user.ad_user_id AS audit_id,
			ad_user.ad_user_desc AS audit_desc,
			ad_user.pr_user_id AS account_id,
			account.pr_name AS account_name,
			ad_user.ad_user_actioned_by_pr_user_id AS user_id,
			user.pr_name AS user_name,
			ad_user.ad_user_time AS audit_timestamp
		FROM
			$bridgeDB.ad_user
			LEFT JOIN $bridgeDB.pr_user account ON account.pr_user_id = ad_user.pr_user_id
			LEFT JOIN $bridgeDB.pr_user user ON user.pr_user_id = ad_user.ad_user_actioned_by_pr_user_id
			LEFT JOIN $bridgeDB.ad_user_detail ON ad_user_detail.ad_user_id = ad_user.ad_user_id
		WHERE
			1
			" . join(' ', $where) . "
		ORDER BY
			ad_user.ad_user_time DESC
	", 2);

	$totalUsers = count($results);

	$record = 0;
	foreach ($results as $key => $result) {
		$results[$key]['record'] = $record . '';
		$results[$key]['total_records'] = $totalUsers . '';
		$record++;
	}

	if ($totalRecords > 0) {
		$results = array_slice($results, $startRecord, $totalRecords);
	}

	if(empty($results)) {
		return "No matching audit entries found";
	} else {
		return $results;
	}
}

/**
 * Bridge API call to show bridge audit entry details.
 */
function api_br_audit_detail_list($id='') {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	if (empty($id)) {
		return $api_instance->Error("Audit ID not specified",400);
	}

	$result = $lDB->get("
		SELECT
			ad_user_detail_action_desc AS audit_desc,
			ad_user_detail_value_from AS audit_value_from,
			ad_user_detail_value_to AS audit_value_to,
			ad_user_detail_action_ind AS audit_action
		FROM
			$bridgeDB.ad_user_detail
		WHERE
			ad_user_id = " . $id . "
	", 2);

	if(empty($result)) {
		return $api_instance->Error("Audit detail does not exist",404);
	} else {
		return $result;
	}
}

/**
 * Bridge API call to list bridge roles.
 */
function api_br_role_list() {
	global $api_instance;
	global $config;
	global $lDB;

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	$results = $lDB->get("
		SELECT
			sc_user_role_id AS user_role_id,
			sc_user_role_name AS user_role_name
		FROM
			$bridgeDB.sc_user_role
	", 2);

	if(empty($results)) {
		return "No roles found";
	} else {
		return $results;
	}
}

/**
 * Bridge API call to create a bridge role.
 */
function api_br_role_create($name='') {
	global $api_instance;
	global $config;
	global $lDB;

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	if (trim($name) == '') {
		return $api_instance->Error("Role name not specified",400);
	}

	$result = $lDB->get("SELECT COUNT(*) FROM sc_user_role WHERE sc_user_role_name = '" . $name . "'", 4);
	if ($result > 0) {
		return $api_instance->Error("Role already exists",409);
	}

	$sql = "
		INSERT INTO $bridgeDB.sc_user_role (
			sc_user_role_name
		) VALUES (
			'" . $lDB->escape($name) . "'
		);
	";

	if($lDB->put($sql)) {
		$result = [
			'role_id' => $lDB->insert_id,
			'role_name' => $name
		];

		$api_instance->auditCreate("", "Create role");
		$api_instance->auditAdd("", $name, "Role name", $api_instance::AUDIT_CREATE);
		$api_instance->auditSave();

		return $result;
	} else {
		return $api_instance->Error("An error occurred with creating the role",500);
	}
}

/**
 * Bridge API call to list bridge user roles.
 */
function api_br_user_role_list($id) {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	if (empty($id)) {
		return $api_instance->Error("User ID not specified",400);
	}

	$results = $lDB->get("
		SELECT
			sc_user_role.sc_user_role_id AS user_role_id,
			sc_user_role.sc_user_role_name AS user_role_name
		FROM
			$bridgeDB.pr_user_role
			LEFT JOIN $bridgeDB.sc_user_role on sc_user_role.sc_user_role_id = pr_user_role.sc_user_role_id
		WHERE
			pr_user_role.pr_user_id = " . $id . "
	", 2);

	if(empty($results)) {
		return "No user roles found";
	} else {
		return $results;
	}
}

/**
 * Bridge API call to create a bridge user role.
 */
function api_br_user_role_create($id='', $roleId='') {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';
	$roleId = !empty($roleId) ? intval($roleId) : '';

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	if (empty($id) || empty($roleId)) {
		return $api_instance->Error("User ID or role ID not specified",400);
	}

	$result = $lDB->get("SELECT COUNT(*) FROM sc_user_role WHERE sc_user_role_id = " . $roleId, 4);
	if ($result == 0) {
		return $api_instance->Error("Role does not exist",409);
	}

	$result = $lDB->get("SELECT COUNT(*) FROM pr_user_role WHERE pr_user_id = " . $id . " AND sc_user_role_id = " . $roleId, 4);
	if ($result > 0) {
		return $api_instance->Error("User role already exists",409);
	}

	$sql = "
		INSERT INTO $bridgeDB.pr_user_role (
			pr_user_id,
			sc_user_role_id
		) VALUES (
			" . $lDB->escape($id) . ",
			" . $lDB->escape($roleId) . "
		);
	";

	if($lDB->put($sql)) {
		$user_role_name = $lDB->get("SELECT sc_user_role_name FROM sc_user_role WHERE sc_user_role_id = " . $roleId, 4);
		$result = [
			'user_role_name' => $user_role_name
		];

		$api_instance->auditCreate($id, "Create role");
		$api_instance->auditAdd("", $user_role_name, "Role name", $api_instance::AUDIT_CREATE);
		$api_instance->auditSave();

		return $result;
	} else {
		return $api_instance->Error("An error occurred with creating the user role",500);
	}
}

/**
 * Bridge API call to delete a bridge user role.
 */
function api_br_user_role_delete($id='', $roleId='') {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';
	$roleId = !empty($roleId) ? intval($roleId) : '';

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	if (empty($id) || empty($roleId)) {
		return $api_instance->Error("User ID or role ID not specified",400);
	}

	$result = $lDB->get("SELECT COUNT(*) FROM pr_user_role WHERE pr_user_id = " . $id . " AND sc_user_role_id = " . $roleId, 4);
	if ($result == 0) {
		return $api_instance->Error("User role does not exist",409);
	}

	$sql = "
		DELETE
			pr_user_role
		FROM
			$bridgeDB.pr_user_role
		WHERE
			pr_user_id = " . $id . "
			AND sc_user_role_id = " . $roleId . ";
	";

	if($lDB->put($sql)) {
		$user_role_name = $lDB->get("SELECT sc_user_role_name FROM sc_user_role WHERE sc_user_role_id = " . $roleId, 4);

		$api_instance->auditCreate($id, "Delete user role");
		$api_instance->auditAdd($user_role_name, "", "User role name", $api_instance::AUDIT_DELETE);
		$api_instance->auditSave();

		return "User role '" . $user_role_name . "' deleted";
	} else {
		return $api_instance->Error("An error occurred with deleting the user role",500);
	}
}

/**
 * Bridge API call to list bridge contact roles.
 */
function api_br_contact_role_list() {
	global $api_instance;
	global $config;
	global $lDB;

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	$results = $lDB->get("
		SELECT
			sc_contact_role_id AS contact_role_id,
			sc_contact_role_name AS contact_role_name
		FROM
			$bridgeDB.sc_contact_role
	", 2);

	if(empty($results)) {
		return "No contact roles found";
	} else {
		return $results;
	}
}

/**
 * Bridge API call to create a bridge contact role.
 */
function api_br_contact_role_create($name='') {
	global $api_instance;
	global $config;
	global $lDB;

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	if (trim($name) == '') {
		return $api_instance->Error("Contact Role name not specified",400);
	}

	$result = $lDB->get("SELECT COUNT(*) FROM sc_contact_role WHERE sc_contact_role_name = '" . $name . "'", 4);
	if ($result > 0) {
		return $api_instance->Error("Contact role already exists",409);
	}

	$sql = "
		INSERT INTO $bridgeDB.sc_contact_role (
			sc_contact_role_name
		) VALUES (
			'" . $lDB->escape($name) . "'
		);
	";

	if($lDB->put($sql)) {
		$result = [
			'contact_role_id' => $lDB->insert_id,
			'contact_role_name' => $name
		];

		$api_instance->auditCreate("", "Create contact role");
		$api_instance->auditAdd("", $name, "Contact role name", $api_instance::AUDIT_CREATE);
		$api_instance->auditSave();

		return $result;
	} else {
		return $api_instance->Error("An error occurred with creating the contact role",500);
	}
}

/**
 * Bridge API call to list bridge user contacts.
 */
function api_br_user_contact_list($id) {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	if (empty($id)) {
		return $api_instance->Error("User ID not specified",400);
	}

	$result = $lDB->get("SELECT COUNT(*) FROM pr_user WHERE pr_user_id = " . $id, 4);
	if ($result == 0) {
		return $api_instance->Error("User does not exist",404);
	}
	

	$results = $lDB->get("
		SELECT
			pr_user_contact.pr_user_id AS bridge_user_id,
			pr_user_contact.pr_user_contact_id AS contact_id,
			sc_contact_role.sc_contact_role_name AS contact_role,
			pr_user_contact.sc_contact_role_id AS contact_role_id,
			pr_user_contact.pr_user_contact_name AS contact_name,
			pr_user_contact.pr_user_contact_email AS contact_email,
			pr_user_contact.pr_user_contact_tel AS contact_tel,
			pr_user_contact.pr_user_contact_note AS contact_note
		FROM
			$bridgeDB.pr_user_contact
			LEFT JOIN $bridgeDB.sc_contact_role on sc_contact_role.sc_contact_role_id = pr_user_contact.sc_contact_role_id
		WHERE
			pr_user_contact.pr_user_id = " . $id . "
	", 2);

	if(empty($results)) {
		return "No user contacts found";
	} else {
		return $results;
	}
}

/**
 * Bridge API call to create a bridge user contact.
 */
function api_br_user_contact_create($id='', $name='', $roleId='', $email='', $tel='', $note='') {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';
	$roleId = !empty($roleId) ? intval($roleId) : '';

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	if (empty($id) || empty($name) || empty($roleId)) {
		return $api_instance->Error("User ID, name, or role ID not specified",400);
	}

	$result = $lDB->get("SELECT COUNT(*) FROM pr_user WHERE pr_user_id = " . $id, 4);
	if ($result == 0) {
		return $api_instance->Error("User does not exist",404);
	}

	$result = $lDB->get("SELECT COUNT(*) FROM sc_contact_role WHERE sc_contact_role_id = " . $roleId, 4);
	if ($result == 0) {
		return $api_instance->Error("Contact role does not exist",409);
	}

	$sql = "
		INSERT INTO $bridgeDB.pr_user_contact (
			pr_user_id,
			sc_contact_role_id,
			pr_user_contact_name,
			pr_user_contact_email,
			pr_user_contact_tel,
			pr_user_contact_note
		) VALUES (
			" . $lDB->escape($id) . ",
			" . $lDB->escape($roleId) . ",
			'" . $lDB->escape($name) . "',
			'" . $lDB->escape($email) . "',
			'" . $lDB->escape($tel) . "',
			'" . $lDB->escape($note) . "'
		);
	";

	if($lDB->put($sql)) {
		$contact_role_name = $lDB->get("SELECT sc_contact_role_name FROM sc_contact_role WHERE sc_contact_role_id = " . $roleId, 4);
		$result = [
			'contact_id' => $id,
			'contact_role' => $contact_role_name,
			'contact_name' => $name,
			'contact_email' => $email,
			'contact_tel' => $tel,
			'contact_note' => $note
		];

		$api_instance->auditCreate($id, "Create user contact");
		$api_instance->auditAdd("", $contact_role_name, "Role name", $api_instance::AUDIT_CREATE);
		$api_instance->auditAdd("", $name, "Name", $api_instance::AUDIT_CREATE);
		if (!empty($email)) {
			$api_instance->auditAdd("", $email, "Email", $api_instance::AUDIT_CREATE);
		}
		if (!empty($tel)) {
			$api_instance->auditAdd("", $tel, "Tel", $api_instance::AUDIT_CREATE);
		}
		if (!empty($note)) {
			$api_instance->auditAdd("", $note, "Note", $api_instance::AUDIT_CREATE);
		}
		$api_instance->auditSave();

		return $result;
	} else {
		return $api_instance->Error("An error occurred with creating the user contact",500);
	}
}

/**
 * Bridge API call to update a bridge user contact.
 */
function api_br_user_contact_update($id='', $name='', $roleId='', $email='', $tel='', $note='') {
	global $api_instance;
	global $config;
	global $lDB;

	$id = !empty($id) ? intval($id) : '';
	$roleId = !empty($roleId) ? intval($roleId) : '';

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	if (empty($id)) {
		return $api_instance->Error("Contact ID not specified",400);
	}

	$record = $lDB->get("
		SELECT
			pr_user_contact.pr_user_id,
			pr_user_contact.sc_contact_role_id,
			pr_user_contact.pr_user_contact_name,
			pr_user_contact.pr_user_contact_email,
			pr_user_contact.pr_user_contact_tel,
			pr_user_contact.pr_user_contact_note,
			sc_contact_role.sc_contact_role_name
		FROM
			$bridgeDB.pr_user_contact
			LEFT JOIN $bridgeDB.sc_contact_role on sc_contact_role.sc_contact_role_id = pr_user_contact.sc_contact_role_id
		WHERE
			pr_user_contact_id = " . $id . ";
	", 1);
	if(empty($record)) {
		return $api_instance->Error("User contact does not exist",404);
	}

	if (!empty($roleId)) {
		$role_name = $lDB->get("SELECT sc_contact_role_name FROM sc_contact_role WHERE sc_contact_role_id = " . $roleId, 4);
		if (empty($role_name)) {
			return $api_instance->Error("Contact role does not exist",409);
		}
	}

	$api_instance->auditCreate($record['pr_user_id'], "Update user contact");

	$changes = array();
	if (!empty($name) && $name != $record['pr_user_contact_name']) {
		$changes[] = "pr_user_contact.pr_user_contact_name = '" . $name . "'";
		$api_instance->auditAdd($record['pr_user_contact_name'], $name, "Name", $api_instance::AUDIT_EDIT);
	}
	if (!empty($roleId) && $roleId != $record['sc_contact_role_id']) {
		$changes[] = "pr_user_contact.sc_contact_role_id = '" . $roleId . "'";
		$api_instance->auditAdd($record['sc_contact_role_name'], $role_name, "Role", $api_instance::AUDIT_EDIT);
	}
	if (!empty($email) && $email != $record['pr_user_contact_email']) {
		$changes[] = "pr_user_contact.pr_user_contact_email = '" . $email . "'";
		$api_instance->auditAdd($record['pr_user_contact_email'], $email, "Email", $api_instance::AUDIT_EDIT);
	}
	if (!empty($tel) && $tel != $record['pr_user_contact_tel']) {
		$changes[] = "pr_user_contact.pr_user_contact_tel = '" . $tel . "'";
		$api_instance->auditAdd($record['pr_user_contact_tel'], $tel, "Tel", $api_instance::AUDIT_EDIT);
	}
	if (!empty($note) && $note != $record['pr_user_contact_note']) {
		$changes[] = "pr_user_contact.pr_user_contact_note = '" . $note . "'";
		$api_instance->auditAdd($record['pr_user_contact_note'], $note, "Note", $api_instance::AUDIT_EDIT);
	}

	if (empty($changes)) {
		return $api_instance->Error("No changes specified",400);
	}

	$sql = "
		UPDATE
			$bridgeDB.pr_user_contact
		SET
			" . join(', ', $changes) . "
		WHERE
			pr_user_contact.pr_user_contact_id = " . $id . ";
	";

	if($lDB->put($sql)) {
		$api_instance->auditSave();
		$result = $lDB->get("
			SELECT
				pr_user_contact.pr_user_contact_id AS contact_id,
				sc_contact_role.sc_contact_role_name AS contact_role,
				pr_user_contact.pr_user_contact_name AS contact_name,
				pr_user_contact.pr_user_contact_email AS contact_email,
				pr_user_contact.pr_user_contact_tel AS contact_tel,
				pr_user_contact.pr_user_contact_note AS contact_note
			FROM
				$bridgeDB.pr_user_contact
				LEFT JOIN $bridgeDB.sc_contact_role on sc_contact_role.sc_contact_role_id = pr_user_contact.sc_contact_role_id
			WHERE
				pr_user_contact.pr_user_contact_id = " . $id . "
		", 1);
		return $result;
	} else {
		return $api_instance->Error("An error occurred with editing the user contact",500);
	}
}

/**
 * Bridge API call to delete a bridge user contact.
 */
function api_br_user_contact_delete($userId='', $contactId='') {
	global $api_instance;
	global $config;
	global $lDB;

	$userId = !empty($userId) ? intval($userId) : '';
	$contactId = !empty($contactId) ? intval($contactId) : '';

	if(!$api_instance->Admin) {
		return $api_instance->Error("Access denied for non-admin user",403);
	}
	$bridgeDB = $config['db']['bridge'];

	if (empty($userId) || empty($contactId)) {
		return $api_instance->Error("User ID or contact ID not specified",400);
	}

	$result = $lDB->get("SELECT COUNT(*) FROM pr_user_contact WHERE pr_user_id = " . $userId . " AND pr_user_contact_id = " . $contactId, 4);
	if ($result == 0) {
		return $api_instance->Error("User contact does not exist",409);
	}

	$user_contact_name = $lDB->get("SELECT pr_user_contact_name FROM pr_user_contact WHERE pr_user_id = " . $userId . " AND pr_user_contact_id = " . $contactId, 4);

	$sql = "
		DELETE
			pr_user_contact
		FROM
			$bridgeDB.pr_user_contact
		WHERE
			pr_user_id = " . $userId . "
			AND pr_user_contact_id = " . $contactId . ";
	";

	if($lDB->put($sql)) {
		$api_instance->auditCreate($userId, "Delete user contact");
		$api_instance->auditAdd($user_contact_name, "", "User contact name", $api_instance::AUDIT_DELETE);
		$api_instance->auditSave();

		return "User contact deleted";
	} else {
		return $api_instance->Error("An error occurred with deleting the user contact",500);
	}
}