<?php

use Bridge\Multicall\MulticallQuery;
use Bridge\Multicall\MulticallComplexQuery;

/**
 * api.stock.php - API bridge stock caching
 */

$this->SetFunction('ac_get_stock',"api_bridge_ac_get_stock");

function api_bridge_ac_get_stock($ac_accomm_type_id, $fromDate='', $toDate='', $agentOverride='',$split='',$cacheOverride='') {
	global $api_instance;
	global $lDB;

	if(empty($cacheOverride)) {
		$cacheOverride = 0;
	}

	$cacheMem = true;
	$cacheDB = true;

	if(!empty($cacheOverride)) {
		if(!$api_instance->Admin) {
			return $api_instance->Error("Access denied for non-admin user.");
		}
		if($cacheOverride & 2) {
			$cacheMem = false;
		}
		if($cacheOverride & 4) {
			$cacheDB = false;
		}
	}

	$cacheOverride = strval($cacheOverride);

	// Ignore cache when additional parameters are used
	// or cache is disabled
	if(!empty($agentOverride) || !empty($split) || !$api_instance->Auth['cache'] || (!$cacheMem && !$cacheDB)) {
		if ($api_instance->isMulticall) {
			$request = $api_instance->generateRequest("ac_get_stock", array($ac_accomm_type_id, $fromDate,$toDate,$agentOverride,$split,$cacheOverride));
			return new MulticallQuery($request);
		} else {
			return $api_instance->Call("ac_get_stock", array($ac_accomm_type_id, $fromDate,$toDate,$agentOverride,$split,$cacheOverride));
		}
	}

	$today = date("Y-m-d");

	if(empty($fromDate)) {
		$now = explode("-",$today);
		$fromDate = date("Y-m-d",mktime(0,0,0,$now[1],$now[2],$now[0]));
		$toDate = date("Y-m-d",mktime(0,0,0,$now[1],$now[2]+6,$now[0]));
	}
	if(empty($toDate)) {
		$now = explode("-",$fromDate);
		$toDate = date("Y-m-d",mktime(0,0,0,$now[1],$now[2]+6,$now[0]));
	}

	$days = dateSubtract($toDate, $fromDate) + 1;
	$dateArray = explode("-", $fromDate);
	$stock = array();
	for($count = 0; $count < $days; $count++) {
		$thisDate = date("Y-m-d", mktime(0, 0, 0, $dateArray[1], $dateArray[2] + $count, $dateArray[0]));
		$stock[$thisDate] = false;
	}

	$prefix = '';

	if($cacheMem) {
		$cache = new Cache();
		$prefix = join("_", array(
			"stock",
			$today,
			$api_instance->Auth['ca_user_id'],
			$ac_accomm_type_id
		));
	}

	$missing = array();
	foreach($stock as $date=>&$amount) {
		if($cacheMem) {
			$key = $prefix . "_" . $date;

			// Get the stock amount from the cache
			$amount = $cache->get($key);
		} else {
			$amount = false;
		}

		// If the amount is not in the cache, try the database
		if($amount === false && $cacheDB) {
			$dbAmount = $lDB->get("
				SELECT
					ca_stock.ca_stock_amt
				FROM
					ca_stock
				WHERE
					ca_stock.ca_user_id = '".$lDB->escape($api_instance->Auth['ca_user_id'])."'
					AND ca_stock.ac_accomm_type_id = '".$lDB->escape($ac_accomm_type_id)."'
					AND ca_stock.ca_create_date = '".$lDB->escape($today)."'
					AND ca_stock.ca_stock_date = '".$lDB->escape($date)."'
			",4);
			if($dbAmount !== false) {
				$amount = $dbAmount;
				if($cacheMem) {
					// Store in the cache if found in the database
					$cache->set($key, $amount, "tomorrow");
				}
			}
		}

		// If the amount is not in the database either, add to list of missing dates
		if($amount === false) {
			$missing[] = $date;
		}
	}
	unset($amount);

	$ranges = dateRanges($missing);
	$requests = [];			// Requests to batch

	if(!empty($ranges)) {
		foreach($ranges as $range) {
			if ($api_instance->isMulticall) {
				$request = $api_instance->generateRequest("ac_get_stock", array($ac_accomm_type_id, $range['start_date'], $range['end_date'],'','',$cacheOverride));
				array_push($requests, $request);
				continue;
			}

			$remoteStock = $api_instance->Call("ac_get_stock", array($ac_accomm_type_id, $range['start_date'], $range['end_date'],'','',$cacheOverride));
			if(is_object($remoteStock) && isset($remoteStock->errno) && !empty($remoteStock->errno)) {
				return $api_instance->Error($remoteStock->errstr,$remoteStock->errno);
			}
			foreach($remoteStock as $date=>$amount) {
				$stock[$date] = $amount;
				if($cacheMem) {
					$cache->set($prefix . "_" . $date, $amount, "tomorrow");
				}
				if($cacheDB) {
					$lDB->put("
						INSERT INTO ca_stock (
							ca_user_id,
							ac_accomm_type_id,
							ca_create_date,
							ca_stock_date,
							ca_stock_amt
						) VALUES (
							'".$lDB->escape($api_instance->Auth['ca_user_id'])."',
							'".$lDB->escape($ac_accomm_type_id)."',
							'".$lDB->escape($today)."',
							'".$lDB->escape($date)."',
							'".$lDB->escape($amount)."'
						) ON DUPLICATE KEY UPDATE
							ca_stock_amt = VALUES(ca_stock_amt)
					");
				}
			}
		}
	}

	if ($api_instance->isMulticall && count($requests) > 0) {
		$data = [
			'stock' => &$stock,
			'ca_user_id' => $api_instance->Auth['ca_user_id'],
			'ac_accomm_type_id' => $ac_accomm_type_id,
			'prefix' => $prefix,
			'cache' => &$cache,
			'cacheMem' => $cacheMem,
			'cacheDB' => $cacheDB
		];

		return new MulticallComplexQuery($requests, 'bridge_ac_get_stock_callback', $data);
	}

	return $stock;
}

/**
 * Callback function for ac_get_stock call that was part of a multicall with caching.
 */
function bridge_ac_get_stock_callback($responses, $data)
{
	global $api_instance;
	global $lDB;

	$today = date("Y-m-d");

	foreach ($responses as $remoteStock) {
		if(is_object($remoteStock) && isset($remoteStock->errno) && !empty($remoteStock->errno)) {
			return $api_instance->Error($remoteStock->errstr,$remoteStock->errno);
		}

		foreach($remoteStock as $date=>$amount) {
			$data['stock'][$date] = $amount;
			if($data['cacheMem']) {
				$data['cache']->set($data['prefix'] . "_" . $date, $amount, "tomorrow");
			}
			if($data['cacheDB']) {
				$lDB->put("
					INSERT INTO ca_stock (
						ca_user_id,
						ac_accomm_type_id,
						ca_create_date,
						ca_stock_date,
						ca_stock_amt
					) VALUES (
						'".$lDB->escape($data['ca_user_id'])."',
						'".$lDB->escape($data['ac_accomm_type_id'])."',
						'".$lDB->escape($today)."',
						'".$lDB->escape($date)."',
						'".$lDB->escape($amount)."'
					) ON DUPLICATE KEY UPDATE
						ca_stock_amt = VALUES(ca_stock_amt)
				");
			}
		}
	}

	return new xmlrpcresp(php_xmlrpc_encode($data['stock']));
}