<?php

namespace Bridge\Multicall;

/**
 * Handles multicalls that are made to the bridge. Requests that are meant to
 * be sent to the same principal are grouped into a multicall and sent.
 */
class MulticallService
{
    private $api_instance;
    private $results;       // The results of processing each call

    /**
     * Create a new instance of the service.
     */
    public function __construct($api_instance)
    {
        $this->api_instance = $api_instance;
        $this->results = [];
    }

    /**
     * Add a result to the list of results.
     *
     * @param any|MulticallQuery|MulticallComplexQuery $result
     * @return void
     */
    public function addResult($result)
    {
        array_push($this->results, $result);
    }

    /**
     * Handle the multicall request that came in.
     *
     * @param object $server
     * @param object $message
     * @return xmlrpcresp
     */
    public function service($server, $message)
    {
        $this->executeCalls($server, $message);

        $totalRequests = count($this->results);
        $batchedQueries = [];           // Batched queries that need to be sent
        $intermediateResponses = [];      // Intermediate results from the 
        $intermediateIndex = 0;
        $intermediateIndices = [];      // The start positions in the intermediate result array
        $finalResults = [];
        $parameters = [];

        for ($i = 0; $i < $totalRequests; $i++) {
            $currentResult = $this->results[$i];

            if (
                $currentResult instanceof MulticallQuery
                || $currentResult instanceof MulticallComplexQuery
            ) {
                array_push($intermediateIndices, $intermediateIndex);

                if ($currentResult instanceof MulticallQuery) {
                    // Single request that needs to be made
                    if (!array_key_exists($currentResult->request->url, $batchedQueries)) {
                        $batchedQueries[$currentResult->request->url] = [
                            'params' => [],
                            'positions' => []         // Position in the intermediate result set.
                        ];
                    }

                    $params = array_slice($currentResult->request->data->params, 2, count($currentResult->request->data->params));
                    $parameter = [];
                    foreach ($params as $param) {
                      $parameter[] = array_values($param->me)[0] ?? "";
                    }
                    $parameters[] = [ $currentResult->request->data->methodname => $parameter ];
                    array_push(
                        $batchedQueries[$currentResult->request->url]['params'],
                        $currentResult->request->data
                    );
                    array_push(
                        $batchedQueries[$currentResult->request->url]['positions'],
                        $intermediateIndex
                    );

                    $intermediateIndex++;
                } else {
                    // One or more requests need to be made
                    foreach ($currentResult->requests as $request) {
                        $params = array_slice($request->data->params, 2, count($request->data->params));
                        $parameter = [];
                        foreach ($params as $param) {
                          $parameter[] = array_values($param->me)[0] ?? "";
                        }
                        $parameters[] = [ $request->data->methodname => $parameter ];
                        if (!array_key_exists($request->url, $batchedQueries)) {
                            $batchedQueries[$request->url] = [
                                'params' => [],
                                'positions' => []         // Position in the intermediate result set.
                            ];
                        }

                        array_push(
                            $batchedQueries[$request->url]['params'],
                            $request->data
                        );
                        array_push(
                            $batchedQueries[$request->url]['positions'],
                            $intermediateIndex
                        );

                        $intermediateIndex++;
                    }
                }

                array_push($finalResults, null);
            } else {
                // Normal xmlrpcresponse
                array_push($intermediateIndices, null);

                if ($currentResult->faultCode() != 0) {
                    $val = _xmlrpcs_multicall_error($currentResult);
                    array_push($finalResults, $val);
                } else {
                    $val = new \xmlrpcval(array($currentResult->value()), 'array');
                    array_push($finalResults, $val);
                }
            }
        }

        $bridgeUsername = $this->api_instance->Auth['bridge_username'] ?? "";
        $principalUsername = $this->api_instance->Auth['principal_username'] ?? "";
        $principalId = $this->api_instance->Auth['principal_id'] ?? "";
        $environmentCode = $this->api_instance->Auth['environment_code'] ?? "";
        $this->api_instance->log(
          $this->api_instance->uniqueId,
          $bridgeUsername,
          $principalUsername,
          $principalId,
          $environmentCode,
          "system.multicall",
          $this->api_instance::CALL_TYPE_REQUEST,
          $parameters
        );

        $responses = [];
        foreach ($batchedQueries as $url => $data) {
            $client = new \jsonrpc_client($url);
            $client->return_type = 'phpvals';
            $client->no_multicall = false;
            if (property_exists($this->api_instance, 'headers')) {
              $client->setHeaders($this->api_instance->headers);
            }
            $resp = $client->send($data['params']);

            $response = [];
            for ($i = 0; $i < count($data['positions']); $i++) {
                $intermediateResponses[$data['positions'][$i]] = $resp[$i];
                $methodName = $data['params'][$i]->methodname;
                $response[$methodName] = $resp[$i]->value();
            }
            $responses[] = $response;
        }

        $this->api_instance->log(
          $this->api_instance->uniqueId,
          $bridgeUsername,
          $principalUsername,
          $principalId,
          $environmentCode,
          "system.multicall",
          $this->api_instance::CALL_TYPE_RESPONSE,
          $responses,
          "RESPONSE 1"
        );

        for ($i = 0; $i < $totalRequests; $i++) {
            if ($intermediateIndices[$i] !== null) {
                $currentResult = $this->results[$i];
                $currentIndex = $intermediateIndices[$i];

                if ($currentResult instanceof MulticallQuery) {
                    $currentIntermediateResponse = $intermediateResponses[$currentIndex];

                    if ($currentIntermediateResponse->faultCode() != 0) {
                        $requestResult = _xmlrpcs_multicall_error($currentIntermediateResponse);
                    } else {
                        $requestResult = php_jsonrpc_encode(array($currentIntermediateResponse->value()), []); 
                    }

                    $finalResults[$i] = $requestResult;
                } else {
                    $nIntermediateRequests = count($this->results[$i]->requests);

                    $responses = array_slice(
                        $intermediateResponses,
                        $currentIndex,
                        $nIntermediateRequests
                    );

                    $parsedResponses = [];

                    foreach ($responses as $response) {
                        if($response->faultCode()) {
                            array_push($parsedResponses, $this->api_instance->Error($response->faultString(),$response->faultCode()));
                        } else {
                            array_push($parsedResponses, $response->value());
                        }	
                    }

                    $response = call_user_func_array(
                        $currentResult->callback,
                        [$parsedResponses, $currentResult->data]
                    );

                    if ($response->faultCode() != 0) {
                        $requestResult = _xmlrpcs_multicall_error($response);
                    } else {
                        $requestResult = php_jsonrpc_encode($response->value(), []); 
                    }

                    $finalResults[$i] = $requestResult;
                }
            }
        }

        $res = new \xmlrpcval($finalResults, 'array');
        $response = new \xmlrpcresp($res);
        return $response;
    }

    private function executeCalls($server, $message)
    {
        // Accept a plain list of php parameters, beside a single xmlrpc msg object
        if (is_object($message))
        {
            $calls = $message->getParam(0);
            $numCalls = $calls->arraysize();
            for($i = 0; $i < $numCalls; $i++)
            {
                $call = $calls->arraymem($i);
                _xmlrpcs_multicall_do_call($server, $call);
            }
        }
        else
        {
            $numCalls=count($message);
            for($i = 0; $i < $numCalls; $i++)
            {
                _xmlrpcs_multicall_do_call_phpvals($server, $message[$i]);
            }
        }
    }
}
