<?php

ob_start();

$versionDir = "../.."; // Override the location to look for the .ver file

require_once(__DIR__ . "/../../class.mysqldb.php");           // MySQL DB Class  # obvious
require_once(__DIR__ . "/../../functions.system.php");
require_once(__DIR__ . "/../../functions.php");
require_once(__DIR__ . "/../../inc.setup.php");               // Database Connection, customised setup and config file
require_once(__DIR__ . "/../../ac_logon.php");             // Access Control Results
require_once(__DIR__ . "/../../db.rv_reservation.php");
require_once(__DIR__ . "/../../db.rv_reservation_item.php");
require_once(__DIR__ . "/../../db.rv_extra.php");
require_once(__DIR__ . "/../../db.fn_folio.php");
require_once(__DIR__ . "/../../db.rv_res_item_group.php");
require_once(__DIR__ . "/../../db.rv_payment.php");
require_once(__DIR__ . "/../../db.rv_payment_item.php");
require_once(__DIR__ . "/../../functions.reservation.php");
require_once(__DIR__ . "/../../functions.financial.php");
require_once(__DIR__ . "/../../functions.rates.php");
require_once(__DIR__ . "/../../db.tc_sequence.php");

class ReservationImport {

    const CHECKS = array("use_key","regen","audit","use_agent");
    const SHEET_COLUMNS = array(
        "reservation_number"=>array("description"=>"Res Number","type"=>"VARCHAR(255)","example"=>"12345", "index"=>false, "sheet_heading"=>true),
        "arrival_date"=>array("description"=>"Arrive Date","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "departure_date"=>array("description"=>"Depature Date","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "created_by"=>array("description"=>"Created By","type"=>"VARCHAR(255)","example"=>"", "index"=>true, "sheet_heading"=>true),
        "agent"=>array("description"=>"Agent","type"=>"VARCHAR(255)","example"=>"", "index"=>true, "sheet_heading"=>true),
        "invoice_contact"=>array("description"=>"Invoice Contact","type"=>"VARCHAR(255)","example"=>"", "index"=>true, "sheet_heading"=>true),
        "billing_contact"=>array("description"=>"Billing Contact","type"=>"VARCHAR(255)","example"=>"", "index"=>true, "sheet_heading"=>true),
        "originator"=>array("description"=>"Originator","type"=>"VARCHAR(255)","example"=>"", "index"=>true, "sheet_heading"=>true),
        "internal_consultant"=>array("description"=>"Internal Consultant","type"=>"VARCHAR(255)","example"=>"", "index"=>true, "sheet_heading"=>true),
        "payment_plan"=>array("description"=>"Payment Plan","type"=>"VARCHAR(255)","example"=>"", "index"=>true, "sheet_heading"=>true),
        "reservation_name"=>array("description"=>"Reservation name","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "pax_adults"=>array("description"=>"Adults","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "pax_children"=>array("description"=>"Children","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "nights"=>array("description"=>"Nights","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "accommodation_total"=>array("description"=>"Accommodation Total Charge","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "override"=>array("description"=>"Override","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "property"=>array("description"=>"Property","type"=>"VARCHAR(255)","example"=>"", "index"=>true, "sheet_heading"=>true),
        "accommodation_type"=>array("description"=>"Room Name","type"=>"VARCHAR(255)","example"=>"", "index"=>true, "sheet_heading"=>true),
        "room_quantity"=>array("description"=>"Room QTY","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "external_rate"=>array("description"=>"Applied Rate","type"=>"VARCHAR(255)","example"=>"", "index"=>true, "sheet_heading"=>true),
        "agent_commission_amount"=>array("description"=>"Commission","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "invoice_currency"=>array("description"=>"Invoice Currency","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "nett"=>array("description"=>"Nett","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "voucher_ref"=>array("description"=>"Voucher Ref","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "source"=>array("description"=>"Source","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "booking_date"=>array("description"=>"Created Date","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "status"=>array("description"=>"Status","type"=>"VARCHAR(255)","example"=>"", "index"=>true, "sheet_heading"=>true),
        "provisional_expiry_date"=>array("description"=>"Provisional Expiry Date","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "payment"=>array("description"=>"Payment","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "paid_date"=>array("description"=>"Paid Date","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "payment_method"=>array("description"=>"Payment Method","type"=>"VARCHAR(255)","example"=>"", "index"=>true, "sheet_heading"=>true),
        "payment_note"=>array("description"=>"Payment Note","type"=>"VARCHAR(255)","example"=>"", "index"=>true, "sheet_heading"=>true),
        "bank_account"=>array("description"=>"Bank Account","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "reservation_notes"=>array("description"=>"Res Notes","type"=>"TEXT","example"=>"", "index"=>false, "sheet_heading"=>true),
        "internal_notes"=>array("description"=>"Int notes","type"=>"TEXT","example"=>"", "index"=>false, "sheet_heading"=>true),
        "guest_information"=>array("description"=>"Guest Information","type"=>"TEXT","example"=>"", "index"=>false, "sheet_heading"=>true),
        "nationality"=>array("description"=>"Nationality","type"=>"VARCHAR(255)","example"=>"", "index"=>true, "sheet_heading"=>true),
        "extra_service_date"=>array("description"=>"Extra Service Date","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "extra_description"=>array("description"=>"Extra Description","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "extra_property"=>array("description"=>"Property (Specific name)","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "extra_invoice_unit"=>array("description"=>"Invoicing unit (used when 'Not property specific')","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "extra_quantity"=>array("description"=>"Extra Qty","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "extra_tax"=>array("description"=>"Extra Tax","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "extra_tax_group"=>array("description"=>"Extra Tax Group","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "extra_charge"=>array("description"=>"Extra charge","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "extra_category"=>array("description"=>"Extra category","type"=>"VARCHAR(255)","example"=>"", "index"=>false, "sheet_heading"=>true),
        "rv_reservation_id"=>array("description"=>"","type"=>"VARCHAR(255)","example"=>"", "index"=>true, "sheet_heading"=>false), // This must always be the last column in this array
    );
    const INDEXES = array(
        "rt_rate_type"=>"rt_rate_type_desc",
        "rf_reservation_status"=>"rf_reservation_status_desc",
        "rf_source"=>"rf_source_desc",
        "pr_persona"=>"pr_name_last",
        "rf_currency"=>"rf_currency_symbol",
        "rf_country"=>"rf_country_name",
        "ac_accomm_type"=>"ac_accomm_desc",
        "rf_bank"=>"rf_currency_id",
        "rf_bank"=>"rf_bank_acc_name",
        "rf_mthd_pmnt"=>"rf_mthd_pmnt_desc"
    );
    protected $messages = "";
    protected $notifications = array();
    protected $html = "";
    protected $lDB = null;
    protected $action = "";
    protected $post = array();
    protected $files = array();
    protected $step = 1;
    protected $error = array();
    protected $filePrefix = "";
    protected $userId = "";
    protected $userStatusId = "";

    function __construct($_lDB, $_userId, $_userStatusId, $_post, $_files) {
        $this->log("Time started: ".date("Y-m-d H:i:s"));
        $this->userStatusId = $_userStatusId;
        $this->isAuthorised($this->userStatusId);
        $this->lDB = $_lDB;
        $this->lDB->isMaster = "1"; // Set to master
        $this->post = $_post;
        $this->files = $_files;
        $this->userId = $_userId;

        $this->action = !isset($_post['action'])?"":$_post['action'];
        $this->html = $this->getTemplate("reservation_import.htm");
        $this->filePrefix = uniqid("reservation_import_");
        $this->session($_post);
    }

    public function isAuthorised($userStatusId) {
        if($userStatusId < "2") {
            // User not authorised
            $this->redirect("/reservation.php");
            die();
        }
    }

    public function redirect($url){
        header("Location:".$url);
    }

    public function template($aValues,$aHTML) {
        $keys = array_keys($aValues);
        $newKeys = array();
        $newValues = array();
        foreach($keys as $item) {
            array_push($newKeys,"!$item!");
            if(!is_array($aValues[$item])) {
                array_push($newValues, $aValues[$item]);
            } else {
                array_push($newValues, print_r($aValues[$item],true));
            }
        }
        return str_replace($newKeys,$newValues,$aHTML);
    }

    public function getTemplate($gFilename) {
        $sectionArray = preg_split("/\|\|(\w*?)\|\|/",join("\n",file(__DIR__ . "/" . $gFilename)),-1,PREG_SPLIT_DELIM_CAPTURE);
        $htmlArray = array();
        for($count=0; $count < floor(sizeof($sectionArray)/2); $count++) {
            $htmlArray[$sectionArray[($count*2)+1]] = trim($sectionArray[$count*2]);
        }
        return $htmlArray;
    }

    public function output() {
        $tableList = $this->lDB->get("SHOW TABLES LIKE 'ai_%'",3);

        $tables = "";
        if (!empty($tableList) && !empty($_SESSION['reservation_import']['tables'])) {
            foreach($_SESSION['reservation_import']['tables'] as $tableName=>$tableInfo) {
                $table =& $_SESSION['reservation_import']['tables'][$tableName];
                $table['table'] = $tableName;
                if(isset($this->post[$tableName."_table"]) && $this->post[$tableName."_table"] != "") {
                    $table['importFrom'] = $this->post[$tableName."_table"];
                }
                $options = "";
                foreach($tableList as $item) {
                    $selected = "";
                    if(isset($table['importFrom']) && $table['importFrom'] == $item) {
                        $selected = "selected";
                    }
                    $options .= "<option $selected value=\"$item\">$item</option>";
                }
                $table['options'] = $options;
                $table['detail_show'] = "style=\"display:none;\"";
                if(isset($table['importFrom']) && $table['importFrom'] != "") {
                    $table['detail_show'] = "";
                    $fields = $this->lDB->get("DESC $table[importFrom]",2);
                    $details = "";
                    foreach($table['fields'] as $systemField=>$importField) {
                        $options = "";
                        foreach($fields as $field) {
                            $selected = "";
                            if($systemField == $field['Field'] && $importField == "") {
                                $table['fields'][$systemField] = $field['Field'];
                                $selected = "selected";
                            }
                            if(isset($this->post[$tableName."_".$systemField]) && $this->post[$tableName."_".$systemField] != "") {
                                $table['fields'][$systemField] = $this->post[$tableName."_".$systemField];
                                $importField = $this->post[$tableName."_".$systemField];
                            }
                            if($importField == $field['Field']) {
                                $selected = "selected";
                            }
                            $options .= "<option $selected value=\"$field[Field]\">$field[Field]</option>";
                        }
                        $details .= $this->template(array(
                            'id'=>$tableName."_".$systemField,
                            'name'=>$systemField,
                            'options'=>$options
                        ),$this->html['field_select']);
                    }
                    $table['detail_rows'] = $details;
                }

                $tables .= $this->template($table,$this->html['table_select']);
            }
        }

        $properties = $this->lDB->get("
            SELECT
                pr_business.pr_business_id,
                CONCAT(pr_persona.pr_name_last, ' (', pr_business.pr_bus_name_short, ')') AS pr_name_last
            FROM
                pr_business
                INNER JOIN pr_persona ON pr_persona.pr_persona_ix = pr_business.pr_business_id
            WHERE
                pr_business.pr_bus_inactive_yn = '0'
            ORDER BY
                pr_persona.pr_name_last
        ",6);
        $cmbPropertyOptions = "";
        foreach($properties as $property) {
            $selected = $_SESSION['reservation_import']['propertyId']!=""&&$_SESSION['reservation_import']['propertyId']?"selected":"";
            $cmbPropertyOptions .= "<option value='".$property['pr_business_id']."' $selected>".$property['pr_name_last']."</option>";
        }
        $this->tags['cmbPropertyOptions'] = $cmbPropertyOptions;

        $this->tags['importFileNameClean'] = "";
        if (!empty($_SESSION['reservation_import']['importFileNameClean'])) {
            $this->tags['importFileNameClean'] = "<div style='font-style:italic;font-size:15px;'><br/><br/>File imported: ".$_SESSION['reservation_import']['importFileNameClean']."</div>";
        }
        $this->tags['maxFileSize'] = ini_get("upload_max_filesize");
        $this->tags['tables'] = $tables;
        $this->tags['stepDescription'] = $this->step==1?"Start with step 1":"Current step";
        $this->tags['step'] = $this->step==1?"":$this->step;
        $this->tags['error'] = !empty($this->error)?"<h3>Errors</h3><ul><li>".join("</li><li>",$this->error)."</li></ul>":"";
        foreach ($this->error as $errorText) {
            $this->log($errorText, "ERROR");
        }
        $this->tags['messages'] = $this->messages;
        $this->tags['notifications'] = "";
        $this->tags['showTemplate'] = $this->step>1?"display:block;":"display:none;";
        if ($this->notifications) {
            foreach ($this->notifications as $notification) {
                $this->tags['notifications'] .= "notify('$notification');";
            }
        }
        $this->tags['use_key_checked'] = $_SESSION['reservation_import']['use_key'] == "1"?"checked":"";
        $this->tags['regen_checked'] = $_SESSION['reservation_import']['regen'] == "1"?"checked":"";
        $this->tags['audit_checked'] = $_SESSION['reservation_import']['audit'] == "1"?"checked":"";
        $this->tags['use_agent_checked'] = $_SESSION['reservation_import']['use_agent'] == "1"?"checked":"";
        $this->tags['hasErrors'] = !empty($this->tags['error'])?"true":"false";
        $this->tags['hasMessages'] = !empty($this->tags['messages'])?"true":"false";
        $output = $this->html['header'] . $this->html['table_options']. $this->html['footer']. "\n</body>\n</html>";
        $output = $this->template($this->tags,$output);
        echo $output;
        $this->log("Time completed: ".date("Y-m-d H:i:s"));
    }

    public function staging() {
        // Step 2
        $this->log("Importing to staging tables...");
        global $dbcode;

        $systemUserId = $this->lDB->get("SELECT DISTINCT pr_user_id FROM pr_user WHERE pr_user_name = 'system' ",4);

        // Clean up csv data
        $sql = "
            UPDATE
                ai_csv_reservation 
            SET";
        foreach(self::SHEET_COLUMNS as $columnName => $columnDetails) {
            $sqlSet[] = " $columnName = TRIM(TRIM(BOTH char(13) FROM $columnName)) "; // trim all whitespace
        }
        $sql = $sql . join(", ",$sqlSet);

        $this->sqlPut($sql);
        if(isset($_SESSION['reservation_import']['db_code']) && $_SESSION['reservation_import']['db_code'] != "") {
            $db_code = strtoupper($_SESSION['reservation_import']['db_code']);
        } else {
            $db_code = $dbcode;
        }
        foreach (self::INDEXES as $table => $index) {
            $this->sqlPut("CREATE INDEX ai_$index ON $table($index);");
        }

        // Transform data
        $this->log("Importing to ai_reservation...");
        $this->sqlPut("
            INSERT INTO ai_reservation
                (
                    `key`,
                    rv_reservation_db,
                    rv_agent_ref,
                    rv_amt_accomm_gross,
                    rv_amt_accomm_nett,
                    rv_amt_accomm_payable,
                    rv_amt_accomm_tax,
                    rv_amt_extra_comm,
                    rv_amt_extra_gross,
                    rv_amt_extra_nett,
                    rv_amt_extra_payable,
                    rv_amt_extra_tax,
                    rv_amt_paid,
                    rv_amt_travel_comm,
                    rv_amt_travel_gross,
                    rv_amt_travel_nett,
                    rv_amt_travel_payable,
                    rv_amt_travel_tax,
                    rv_commission_perc,
                    rv_date_recorded,
                    rv_note_general,
                    rv_note_guests,
                    rv_note_internal,
                    rv_pax_count,
                    rv_provision_expiry_date,
                    rv_res_name,
                    rv_reservation_type_ind,
                    rf_country_id,
                    rt_rate_type_id,
                    rf_reservation_status_id,
                    rf_confirmation_id,
                    rf_source_ix,
                    rv_agent_id,
                    rv_billing_persona_id,
                    rv_consultant_id,
                    rv_corr_persona_id,
                    rv_invoice_currency_id,
                    rv_invoice_persona_id,
                    rv_note_business_id,
                    rv_origin_agent_id,
                    pr_reservation_user_id,
                    rv_commission_deduct_yn
                )
                SELECT
                    reservation_number,
                    '$db_code',
                    voucher_ref,
                    SUM(accommodation_total),
                    SUM(accommodation_total),
                    SUM(accommodation_total) - SUM(payment),
                    '0',
                    '0',
                    SUM(extra_charge),
                    SUM(extra_charge),
                    '0',
                    '0',
                    SUM(payment),
                    '0',
                    SUM(accommodation_total),
                    SUM(accommodation_total),
                    SUM(accommodation_total - payment),
                    '0',
                    agent_commission_amount,
                    MIN(STR_TO_DATE(booking_date, '%Y-%m-%d %H:%i:%s')),
                    reservation_notes,
                    guest_information,
                    internal_notes,
                    SUM(pax_adults) + SUM(pax_children),
                    provisional_expiry_date,
                    reservation_name,
                    '0',
                    rf_country.rf_country_ix,
                    rt_rate_type.rt_rate_type_ix,
                    rf_reservation_status.rf_reservation_status_id,
                    NULL,
                    rf_source.rf_source_ix,
                    CASE
                        WHEN agent = 'Direct' THEN '0'
                        ELSE agent_persona.pr_persona_ix
                    END,
                    CASE
                        WHEN billing_contact = 'Direct' THEN '0'
                        ELSE billing_persona.pr_persona_ix
                    END,
                    internal_consultant,
                    NULL,
                    rf_currency.rf_currency_ix,
                    CASE
                        WHEN invoice_contact = 'Direct' THEN '0'
                        ELSE invoice_persona.pr_persona_ix
                    END,
                    NULL,
                    originator_persona.pr_persona_ix,
                    created_by,
                    CASE
                        WHEN nett = 'Nett' THEN '1'
                        WHEN nett = 'Gross' THEN '0'
                        ELSE '0'
                    END
                FROM
                    ai_csv_reservation
                    LEFT JOIN rt_rate_type ON rt_rate_type.rt_rate_type_desc = external_rate AND external_rate != ''
                    LEFT JOIN rf_reservation_status ON rf_reservation_status.rf_reservation_status_desc = status AND status != ''
                    LEFT JOIN rf_source ON rf_source.rf_source_desc = source AND source != ''
                    LEFT JOIN rf_country ON rf_country.rf_country_name = ai_csv_reservation.nationality AND ai_csv_reservation.nationality != ''
                    LEFT JOIN rf_currency ON rf_currency.rf_currency_symbol = invoice_currency AND invoice_currency != ''
                    LEFT JOIN pr_persona agent_persona ON agent_persona.pr_name_last = agent AND agent != ''
                    LEFT JOIN pr_persona invoice_persona ON invoice_persona.pr_name_last = invoice_contact AND invoice_contact != ''
                    LEFT JOIN pr_persona originator_persona ON originator_persona.pr_name_last = ai_csv_reservation.originator AND ai_csv_reservation.originator != ''
                    LEFT JOIN pr_persona billing_persona ON billing_persona.pr_name_last = billing_contact AND billing_contact != ''
                WHERE
                    ai_csv_reservation.reservation_number != ''
                    AND ai_csv_reservation.status != ''
                    AND ai_csv_reservation.arrival_date != ''
                    AND ai_csv_reservation.departure_date != ''
                GROUP BY reservation_number
                ORDER BY CAST(reservation_number AS INT) ASC
        ");
        $this->sqlPut("
            UPDATE
                ai_reservation
                LEFT JOIN pr_persona consultant_persona ON TRIM(CONCAT(consultant_persona.pr_name_first, ' ', consultant_persona.pr_name_last)) = ai_reservation.rv_consultant_id AND TRIM(CONCAT(consultant_persona.pr_name_first, ' ', consultant_persona.pr_name_last)) != ''
                LEFT JOIN pr_persona created_persona ON TRIM(CONCAT(created_persona.pr_name_first, ' ', created_persona.pr_name_last)) = ai_reservation.pr_reservation_user_id AND TRIM(CONCAT(created_persona.pr_name_first, ' ', created_persona.pr_name_last)) != ''
            SET
                ai_reservation.rv_consultant_id = consultant_persona.pr_persona_ix,
                ai_reservation.pr_reservation_user_id = created_persona.pr_persona_ix
        ");

        $sqlAiReservationWhere = "";
        $sqlAiReservationFrom = "";
        if (!empty($_SESSION['reservation_import']['propertyId'])) {
            $sqlAiReservationWhere = " AND ac_accomm_type.pr_business_id = '".$_SESSION['reservation_import']['propertyId']."'";
        } else {
            $sqlAiReservationFrom = " 
                INNER JOIN pr_persona ON pr_persona.pr_name_last = ai_csv_reservation.property
                INNER JOIN pr_business ON pr_business.pr_business_id = pr_persona.pr_persona_ix";
            $sqlAiReservationWhere = " AND ac_accomm_type.pr_business_id = pr_business.pr_business_id";
        }
        $this->log("Importing to ai_reservation_item...");
        $this->sqlPut("
            INSERT INTO ai_reservation_item
                (
                    `key`,
                    rv_reservation_item_db,
                    rv_item_accomm_count,
                    rv_item_amt_gross,
                    rv_item_amt_nett,
                    rv_item_amt_payable,
                    rv_item_amt_tax,
                    rv_item_overide_level_ind,
                    rv_item_overide_amt,
                    rv_item_date_added,
                    rv_item_date_arrive,
                    rv_item_date_depart,
                    rv_item_date_booked,
                    rv_item_adult_count,
                    rv_item_child_count,
                    rv_item_nights,
                    rt_rate_type_id,
                    ac_accomm_type_id,
                    pr_business_id,
                    rv_item_user_id
                )
                SELECT
                    reservation_number,
                    '$db_code',
                    CASE
                        WHEN room_quantity IS NULL THEN '1'
                        WHEN TRIM(room_quantity) = '' THEN '1'
                        ELSE room_quantity
                    END,
                    accommodation_total,
                    accommodation_total,
                    accommodation_total - payment,
                    '0',
                    CASE
                        WHEN override = 'Per Person' THEN 1
                        WHEN override = 'Per Unit' THEN 2
                        WHEN override = 'Per Stay' THEN 3
                        ELSE ''
                    END,
                    accommodation_total,
                    STR_TO_DATE(booking_date, '%Y-%m-%d %H:%i:%s'),
                    STR_TO_DATE(arrival_date, '%Y-%m-%d %H:%i:%s'),
                    STR_TO_DATE(departure_date, '%Y-%m-%d %H:%i:%s'),
                    STR_TO_DATE(booking_date, '%Y-%m-%d %H:%i:%s'),
                    pax_adults,
                    pax_children,
                    nights,
                    rt_rate_type.rt_rate_type_ix,
                    ac_accomm_type.ac_accomm_type_ix,
                    ac_accomm_type.pr_business_id,
                    IFNULL(consultant_persona.pr_persona_ix, '$systemUserId')
                FROM
                    ai_csv_reservation
                    INNER JOIN rt_rate_type ON rt_rate_type.rt_rate_type_desc = external_rate
                    INNER JOIN ac_accomm_type ON ac_accomm_type.ac_accomm_desc = accommodation_type
                    LEFT JOIN pr_persona consultant_persona ON CONCAT(consultant_persona.pr_name_first, ' ', consultant_persona.pr_name_second, ' ', consultant_persona.pr_name_last) = created_by
                    LEFT JOIN pr_user ON pr_user.pr_user_id = consultant_persona.pr_persona_ix
                    $sqlAiReservationFrom
                WHERE
                    ai_csv_reservation.reservation_number != ''
                    $sqlAiReservationWhere
        ");

        $this->log("Importing to ai_extra...");
        $this->sqlPut("
            INSERT INTO ai_extra
                (
                    `key`,
                    rv_extra_db,
                    ac_extra_id,
                    pr_business_id,
                    rv_extra_units,
                    rv_extra_date_serv,
                    rv_extra_charge,
                    rv_extra_discount,
                    rv_extra_comm_rec,
                    rv_extra_comm_pay,
                    rv_extra_note,
                    rv_extra_note_internal,
                    rv_res_item_group_id,
                    rf_currency_id,
                    rf_tax_id,
                    rv_extra_tax_ind,
                    rv_extra_inv_curr_id,
                    rv_extra_exch_rate,
                    rv_extra_exch_expiry,
                    pr_business_inv_id
                )
                SELECT
                    reservation_number,
                    '$db_code',
                    ac_extra.ac_extra_ix,
                    CASE 
                        WHEN TRIM(extra_property) = 'Not property specific' THEN '0'
                        ELSE extra_persona.pr_persona_ix
                    END,
                    extra_quantity,
                    STR_TO_DATE(extra_service_date, '%Y-%m-%d %H:%i:%s'),
                    extra_charge,
                    0,
                    0,
                    0,
                    NULL,
                    '
IMPORTED FROM EXTERNAL SYSTEM
Date: ".date('Y-m-d h:i:s a', time())."',
                    NULL,
                    rf_currency.rf_currency_ix,
                    CASE
                        WHEN TRIM(extra_tax_group) = '' OR extra_tax_group IS NULL THEN rf_tax_rate.rf_tax_rate_ix
                        ELSE rt_tax_group.rt_tax_group_ix
                    END,
                    CASE
                        WHEN TRIM(extra_tax_group) = '' OR extra_tax_group IS NULL THEN '10'
                        ELSE '20'
                    END,
                    rf_currency.rf_currency_ix,
                    0,
                    '0000-00-00',
                    CASE 
                        WHEN TRIM(extra_property) = 'Not property specific' THEN extra_persona_invoice_unit.pr_persona_ix
                        ELSE NULL
                    END
                FROM
                    ai_csv_reservation
                    INNER JOIN ac_extra ON ac_extra.ac_ext_desc = extra_description
                    INNER JOIN ac_extra_category ON ac_extra_category.ac_extra_category_ix = ac_extra.ac_extra_category_id AND extra_category != '' AND extra_category = ac_extra_category.ac_extra_cat_desc
                    LEFT JOIN pr_persona extra_persona ON extra_persona.pr_name_last = extra_property AND extra_property != ''
                    LEFT JOIN pr_persona extra_persona_invoice_unit ON extra_persona_invoice_unit.pr_name_last = extra_invoice_unit AND extra_invoice_unit != ''
                    LEFT JOIN pr_business ON pr_business.pr_business_id = extra_persona.pr_persona_ix
                    LEFT JOIN rt_tax_group ON rt_tax_group.rt_tax_group_desc = extra_tax_group
                    LEFT JOIN rt_tax_group_item ON rt_tax_group_item.rt_tax_group_id = rt_tax_group.rt_tax_group_ix
                    LEFT JOIN rf_tax_rate tax_group_rate ON tax_group_rate.rf_tax_rate_ix = rt_tax_group_item.rf_tax_rate_id AND tax_group_rate.rf_tax_rate_inactive_yn = 0
                    LEFT JOIN rf_tax_rate ON rf_tax_rate.rf_tax_rate_desc = extra_tax
                    INNER JOIN rf_currency ON rf_currency.rf_currency_symbol = invoice_currency
                WHERE
                    ai_csv_reservation.reservation_number != ''
                    AND extra_description != ''
                ORDER BY reservation_number, extra_service_date ASC
        ");
        // Set tax percentage for tax groups
        $this->sqlPut("
            UPDATE
                ai_extra
                INNER JOIN (
                  SELECT
                        SUM(tax_group_rate.rf_tax_rate_perc) AS tax_perc,
                        ai_extra.rf_tax_id
                  FROM
                        ai_extra
                        INNER JOIN rt_tax_group ON rt_tax_group.rt_tax_group_ix = ai_extra.rf_tax_id
                        INNER JOIN rt_tax_group_item ON rt_tax_group_item.rt_tax_group_id = rt_tax_group.rt_tax_group_ix
                        INNER JOIN rf_tax_rate tax_group_rate ON tax_group_rate.rf_tax_rate_ix = rt_tax_group_item.rf_tax_rate_id AND tax_group_rate.rf_tax_rate_inactive_yn = 0
                  WHERE
                      ai_extra.rv_extra_tax_ind = '20'
                  GROUP BY rt_tax_group.rt_tax_group_ix
                ) tax_perc_totals ON tax_perc_totals.rf_tax_id = ai_extra.rf_tax_id
            SET
                ai_extra.rv_extra_tax_perc = tax_perc_totals.tax_perc
            WHERE
                ai_extra.rv_extra_tax_ind = '20'
        ");
        // Set tax percentage for tax rates
        $this->sqlPut("
            UPDATE
                ai_extra
                INNER JOIN rf_tax_rate ON rf_tax_rate.rf_tax_rate_ix = ai_extra.rf_tax_id
            SET
                ai_extra.rv_extra_tax_perc = rf_tax_rate.rf_tax_rate_perc
            WHERE
                ai_extra.rv_extra_tax_ind = '10'
        ");

        $this->log("Importing to ai_payment...");
        $this->sqlPut("
            INSERT INTO ai_payment
                (
                    `key`,
                    rv_payment_db,
                    rv_pmnt_amount,
                    rv_pmnt_date,
                    rv_pmnt_date_created,
                    rv_pmnt_note,
                    rv_pmnt_ref,
                    pr_persona_id,
                    rf_bank_id,
                    rf_currency_id,
                    rf_mthd_pmnt_id,
                    rv_pmnt_user_id
                )
                SELECT
                    reservation_number,
                    '$db_code',
                    payment,
                    STR_TO_DATE(paid_date, '%Y-%m-%d %H:%i:%s'),
                    STR_TO_DATE(paid_date, '%Y-%m-%d %H:%i:%s'),
                    CASE
                        WHEN payment_note IS NOT NULL THEN payment_note
                        ELSE '
IMPORTED FROM EXTERNAL SYSTEM
Date: ".date('Y-m-d h:i:s a', time())."'
                    END,
                    NULL,
                    IFNULL(consultant_persona.pr_persona_ix, '$systemUserId'),
                    rf_bank.rf_bank_ix,
                    rf_currency.rf_currency_ix,
                    rf_mthd_pmnt.rf_mthd_pmnt_ix,
                    IFNULL(consultant_persona.pr_persona_ix, '$systemUserId')
                FROM
                    ai_csv_reservation
                    LEFT JOIN pr_persona consultant_persona ON CONCAT(consultant_persona.pr_name_first, ' ', consultant_persona.pr_name_second, ' ', consultant_persona.pr_name_last) = created_by
                    LEFT JOIN pr_user ON pr_user.pr_user_id = consultant_persona.pr_persona_ix
                    INNER JOIN rf_currency ON rf_currency.rf_currency_symbol = invoice_currency
                    INNER JOIN rf_bank ON rf_bank.rf_bank_acc_name = bank_account
                    LEFT JOIN rf_mthd_pmnt ON rf_mthd_pmnt.rf_mthd_pmnt_desc = payment_method
                WHERE
                    ai_csv_reservation.reservation_number != ''
                    AND payment != ''
                    AND paid_date != ''
                    AND payment_method != ''
                    AND bank_account != ''
        ");
        $this->log("Importing to staging tables done");
    }

    public function importData() {
        // Step 4
        global $dbcode;

        $rateGroupTemplate = array(
            array('rt_rate_group_ix'=>$this->lDB->get("SELECT rt_rate_group_ix FROM rt_rate_group WHERE rt_rate_group_class='0' AND rt_rate_group_sys_code = '1'",4)),
            array('rt_rate_group_ix'=>$this->lDB->get("SELECT rt_rate_group_ix FROM rt_rate_group WHERE rt_rate_group_class='1' AND rt_rate_group_sys_code = '2'",4)),
            array('rt_rate_group_ix'=>$this->lDB->get("SELECT rt_rate_group_ix FROM rt_rate_group WHERE rt_rate_group_class='2' AND rt_rate_group_sys_code = '3'",4))
        );
        if(isset($_SESSION['reservation_import']['db_code']) && $_SESSION['reservation_import']['db_code'] != "") {
            $db_code = strtoupper($_SESSION['reservation_import']['db_code']);
        } else {
            $db_code = false;
        }

        foreach($_SESSION['reservation_import']['tables'] as $tableName=>$tableInfo) {
            $table =& $_SESSION['reservation_import']['tables'][$tableName];
            $imports = $this->lDB->get("SELECT * FROM $table[table]",2);
            if($tableName == "ai_reservation") {
                $key = array();
            }
            foreach($imports as $import) {
                foreach($table['fields'] as $field) {
                    if(trim($field) == "") {
                        $import[$field] = false;
                    }
                }
                switch($tableName) {
                    case "ai_reservation":
                        $rv_reservation_id = false;
                        $this->log("Importing data from 'ai_reservation'");
                        if(trim($table['fields']['rv_reservation_db']) != "") {
                            $db_code = $import[$table['fields']['rv_reservation_db']];
                        }
                        if($_SESSION['reservation_import']['use_key'] == 1) {
                            $incrementSequence = $GLOBALS['incrementSequence'];
                            $GLOBALS['incrementSequence'] = "0";
                            $rv_reservation_id = $import[$table['fields']['key']];

                            $this->log("Overriding res number with: $rv_reservation_id");
                        } else {
                            if(trim($table['fields']['rv_reservation_id']) != "") {
                                $rv_reservation_id = $import[$table['fields']['rv_reservation_id']];
                            } else {
                                $rv_reservation_id = false;
                            }
                        }
                        $id = false;
                        if (!db_rv_reservation_exists($rv_reservation_id) || $rv_reservation_id === false) {
                            $rv_agent_id = trim($import[$table['fields']['rv_agent_id']]);
                            $rv_billing_persona_id = trim($import[$table['fields']['rv_billing_persona_id']]);
                            $rv_invoice_persona_id = trim($import[$table['fields']['rv_invoice_persona_id']]);
                            if($_SESSION['reservation_import']['use_agent'] == 1 && !empty($rv_agent_id)) {
                                if(empty($rv_billing_persona_id)) {
                                    $rv_billing_persona_id = $rv_agent_id;
                                }

                                if(empty($rv_invoice_persona_id)) {
                                    $rv_invoice_persona_id = $rv_agent_id;
                                }
                            }
                            $id = db_rv_reservation_insert(
                                $import[$table['fields']['rt_rate_type_id']],
                                $import[$table['fields']['rv_corr_persona_id']],
                                $rv_agent_id,
                                $import[$table['fields']['rv_commission_perc']],
                                $import[$table['fields']['rv_commission_deduct_yn']],
                                $import[$table['fields']['rv_invoice_currency_id']],
                                $rv_billing_persona_id,
                                $rv_invoice_persona_id,
                                $import[$table['fields']['rv_res_name']],
                                $import[$table['fields']['rv_agent_ref']],
                                $import[$table['fields']['rv_note_general']],
                                $import[$table['fields']['rv_note_guests']],
                                $import[$table['fields']['rv_note_internal']],
                                $import[$table['fields']['ac_pay_plan_id']],
                                $import[$table['fields']['rf_country_id']],
                                $import[$table['fields']['rf_source_ix']],
                                $import[$table['fields']['rf_reservation_status_id']],
                                $import[$table['fields']['rv_date_recorded']],
                                $import[$table['fields']['rv_provision_expiry_date']],
                                $import[$table['fields']['pr_reservation_user_id']],
                                $import[$table['fields']['rv_consultant_id']],
                                $import[$table['fields']['rv_date_recorded']],
                                $import[$table['fields']['rv_date_changed']],
                                $import[$table['fields']['rv_ref_num']],
                                $db_code,
                                $rv_reservation_id
                            );
                            if($_SESSION['reservation_import']['use_key'] == 1) {
                                $id = $import[$table['fields']['rv_reservation_db']] . $rv_reservation_id;
                                $GLOBALS['incrementSequence'] = $incrementSequence;
                            }
                        }
                        if($id === false || trim($id) == "") {
                            $this->error[] = "Not creating reservation ".$import[$table['fields']['key']];
                        } else {
                            $this->log("Created reservation '$id'");
                            $this->sqlPut("UPDATE ai_csv_reservation SET rv_reservation_id = '$id' WHERE reservation_number = '".$import[$table['fields']['key']]."'");
                            $this->sqlPut("UPDATE rv_reservation SET rv_origin_agent_id = '".$import[$table['fields']['rv_origin_agent_id']]."' WHERE rv_reservation_ix = '$id'");
                            if($_SESSION['reservation_import']['use_key'] == 1) {
                                $this->sqlPut("UPDATE rv_reservation SET rv_reservation_id = '".$import[$table['fields']['key']]."' WHERE rv_reservation_ix = '$id'");
                            }
                            if($_SESSION['reservation_import']['audit'] == "1" && !empty($id)) {
                                $rawDetails = $this->lDB->get("SELECT * FROM ai_csv_reservation WHERE reservation_number = '".$import[$table['fields']['key']]."'",6);
                                ammendReservation(
                                    $id,
                                    "Imported from external system
Date: ".date('Y-m-d h:i:s a', time())."
Data: ".$this->lDB->escape(json_encode($rawDetails))
                                );
                            }
                            $key[$import[$table['fields']['key']]] = $id;
                        }
                        break;
                    case "ai_reservation_item":
                        $this->log("Importing data from 'ai_reservation_item'");
                        if (isset($key[$import[$table['fields']['key']]])) {
                            $rateGroups = $rateGroupTemplate;
                            $rateGroups[0]['qty'] = $import[$table['fields']['rv_item_accomm_count']];
                            $rateGroups[1]['qty'] = $import[$table['fields']['rv_item_adult_count']];
                            $rateGroups[2]['qty'] = $import[$table['fields']['rv_item_child_count']];
                            $rv_item_overide_level_ind = $import[$table['fields']['rv_item_overide_level_ind']];
                            $rv_item_overide_amt = 0;
                            switch ($rv_item_overide_level_ind) {
                                case "1":
                                    $rv_item_overide_amt = $import[$table['fields']['rv_item_overide_amt']];
                                    $override = array(
                                        'level'=>1,
                                        'amounts'=>array(
                                            'rate_group'=>array(
                                                $rateGroups[1]['rt_rate_group_ix']=>$rv_item_overide_amt
                                            )
                                        )
                                    );
                                    break;
                                case "2":
                                    $rv_item_overide_amt = $import[$table['fields']['rv_item_overide_amt']];
                                    $override = array(
                                        'level'=>2,
                                        'amounts'=>array(
                                            'rate_group'=>array(
                                                $rateGroups[1]['rt_rate_group_ix']=>$rv_item_overide_amt
                                            )
                                        )
                                    );
                                    break;
                                case "3":
                                    $rv_item_overide_amt = $import[$table['fields']['rv_item_overide_amt']];
                                    $override = array(
                                        'level'=>3,
                                        'amounts'=>array('stay'=>$rv_item_overide_amt)
                                    );
                                    break;
                                default:
                                    $override = false;
                            }
                            $itemDataRaw = convertItemToRaw(
                              $import[$table['fields']['ac_accomm_type_id']],
                              $import[$table['fields']['rv_item_date_arrive']],
                              $import[$table['fields']['rv_item_date_depart']],
                              $rateGroups
                            );
                            $reservationItems = calCreateItem(
                              $key[$import[$table['fields']['key']]],
                              $itemDataRaw,
                              $import[$table['fields']['rt_rate_type_id']],
                              'defaultOnly',
                              $override,
                              $import[$table['fields']['rv_item_split_yn']],
                              "0"
                            );

                            if(empty($reservationItems)) {
                                $this->error[] = "Error adding itinerary to reservation " . $key[$import[$table['fields']['key']]] . " (".$import[$table['fields']['key']].")";
                            } else {
                              foreach ($reservationItems as $reservationItem) {
                                  $this->log("Created reservation item '$reservationItem'");
                                  db_rv_reservation_update_travel_dates($key[$import[$table['fields']['key']]]);
                                  $this->sqlPut("
                                    UPDATE 
                                      rv_reservation_item
                                    SET
                                      rv_item_user_id = '".$import[$table['fields']['rv_item_user_id']]."'
                                    WHERE rv_reservation_item_ix = '$reservationItem'");
                              }
                            }
                        }
                        break;
                    case "ai_extra":
                        $this->log("Importing data from 'ai_extra'");
                        if(array_key_exists($import[$table['fields']['key']],$key)) {
                            $billingUnit = db_pr_business_get_billing($import[$table['fields']['pr_business_id']]);
                            if (empty($billingUnit)) {
                              $billingUnit = $import[$table['fields']['pr_business_inv_id']];
                            }

                            // Ignore passed percentage, look up percentage from Tax Code Table
                            $taxPerc = taxPercentage($import[$table['fields']['rf_tax_id']], $import[$table['fields']['rv_extra_tax_ind']]);

                            $id = db_rv_extra_insert(
                                $key[$import[$table['fields']['key']]],
                                $import[$table['fields']['ac_extra_id']],
                                $import[$table['fields']['pr_business_id']],
                                $import[$table['fields']['rv_extra_units']],
                                $import[$table['fields']['rv_extra_date_serv']],
                                $import[$table['fields']['rv_extra_charge']],
                                $import[$table['fields']['rv_extra_discount']],
                                $taxPerc,
                                $import[$table['fields']['rv_extra_comm_rec']],
                                $import[$table['fields']['rv_extra_comm_pay']],
                                $import[$table['fields']['rv_extra_note']],
                                $import[$table['fields']['rv_extra_note_internal']],
                                $import[$table['fields']['rv_res_item_group_id']],
                                $import[$table['fields']['rf_currency_id']],
                                $import[$table['fields']['rv_extra_inv_curr_id']],
                                $import[$table['fields']['rv_extra_exch_rate']],
                                $import[$table['fields']['rv_extra_exch_expiry']],
                                $billingUnit
                            );
                            if($id === false || trim($id) == "") {
                                $this->error[] = "Error adding extra to " . $key[$import[$table['fields']['key']]] . "<br/>";
                            } else {
                                db_rv_extra_tax_insert($id, $import[$table['fields']['rv_extra_tax_ind']], $import[$table['fields']['rf_tax_id']]);
                                $this->log("Created extra '$id'");
                                $fn_folio_to_id = $this->lDB->get("SELECT fn_folio_to_id FROM fn_folio WHERE rv_reservation_id = '".$key[$import[$table['fields']['key']]]."' LIMIT 1",4);
                                $fn_folio_id = db_fn_folio_insert_extra($id,false,$fn_folio_to_id);
                                if(!empty($fn_folio_id) && $fn_folio_id != 2) {
                                    db_rv_extra_set_folio($id, $fn_folio_id);
                                }
                            }
                        }
                        break;
                    case "ai_payment":
                        $this->log("Importing data from 'ai_payment'");
                        if(array_key_exists($import[$table['fields']['key']],$key)) {
                            $fn_folio_id = $this->lDB->get("SELECT fn_folio_ix FROM fn_folio WHERE fn_folio.rv_reservation_id = '".$key[$import[$table['fields']['key']]]."' ORDER BY fn_folio.fn_folio_folio_num LIMIT 1",4);
                            if(!empty($fn_folio_id)) {
                                $rv_payment_id = db_rv_payment_insert(
                                    $import[$table['fields']['pr_persona_id']],
                                    $import[$table['fields']['rv_pmnt_date']],
                                    $import[$table['fields']['rf_bank_id']],
                                    $import[$table['fields']['rf_mthd_pmnt_id']],
                                    $import[$table['fields']['rv_pmnt_ref']],
                                    $import[$table['fields']['rv_pmnt_note']]
                                );
                                if(!empty($rv_payment_id)) {
                                    $this->log("Created payment '$id'");
                                    $rv_payment_item_id = db_rv_payment_item_insert(
                                        $rv_payment_id,
                                        $fn_folio_id,
                                        $import[$table['fields']['rv_pmnt_amount']],
                                        1 //TODO $import[$table['fields']['rv_pay_item_exch_rate']]
                                    );
                                    $this->log("Created payment item '$rv_payment_item_id'");
                                    $this->log("Recalculating totals for '".$key[$import[$table['fields']['key']]]."'");
                                    recalcTotals($key[$import[$table['fields']['key']]]);
                                } else {
                                    $this->error[] = "Error adding payment record to reservation " . $import[$table['fields']['key']] . "<br/>";
                                }
                            } else {
                                $this->error[] = "Error adding payment, missing folio on reservation " . $import[$table['fields']['key']] . "<br/>";
                            }
                        }
                        break;
                }
            }
        }
        if ($rv_reservation_id) {
          db_tc_sequence_increment('rv_reservation', $dbcode);
        }

        $this->messages .= "<h2>Import completed</h2>Imported " . sizeof($key) . " reservation(s).<br /><br />\n";
        $reservationsOmitted = $this->lDB->get("SELECT DISTINCT ai_csv_reservation.reservation_number FROM ai_csv_reservation WHERE rv_reservation_id IS NULL", 3);
        if (count($reservationsOmitted) > 0) {
            $this->messages .= "<h3>Reservations omitted</h3>";
            $this->messages .= join(", ",$reservationsOmitted)."<br/>";
        }

        if($_SESSION['reservation_import']['regen'] == "1") {
            $this->messages .= "<h3>Reservations regenerated</h3>";
            foreach($key as $rv_reservation_id) {
                $this->log("Regenerating reservation '$rv_reservation_id'");
                regenReservation($rv_reservation_id);
            }
        }

        $this->log("Generating payment plans...");
        $this->messages .= "<h3>Reservations created</h3>";
        foreach ($key as $extId=>$intId) {
            // Post processing
            $acPayPlanId = $this->lDB->get("
                SELECT
                    ac_pay_plan.ac_pay_plan_ix
                FROM
                    ac_pay_plan
                    INNER JOIN ai_csv_reservation ON ai_csv_reservation.payment_plan = ac_pay_plan.ac_pay_plan_desc
                WHERE
                    ai_csv_reservation.reservation_number = '".$extId."'
                    AND ai_csv_reservation.payment_plan IS NOT NULL
                    AND ai_csv_reservation.payment_plan != ''
                LIMIT 1",
            4);
            if (!empty($acPayPlanId)) {
                $this->log("Generating payment plan for '$intId'");
                genPaymentPlan($intId, $acPayPlanId);
                $this->log("Recalculating totals for '$intId'");
                recalcTotals($intId);
            }
            $this->messages .= "$intId ($extId)<br/>";
        }
        $this->notifications[] = "Import completed";
    }

    public function exportCSV() {
        $columns = array_search(true, array_column(self::SHEET_COLUMNS, 'sheet_heading'));
        $columns = "'".implode("','",$columns)."'";
        $importedValues = $this->lDB->get("SELECT $columns FROM ai_csv_reservation",6);
        $csv = fopen("php://output", 'w');
        $bFirstRowHeader = true;
        foreach ($importedValues as $row) {
            if ($bFirstRowHeader) {
                fputcsv($csv, array_keys($row));
                $bFirstRowHeader = false;
            }
            if ($headingsOnly) { break; }
            fputcsv($csv, array_values($row));
        }
        rewind($csv);
        $fileName = 'ai_csv_reservation.csv';
        $fstat = fstat($csv);
        $this->setHeader($filename, $fstat['size']);
        fpassthru($csv);
        fclose($csv);
    }

    public function createTables() {
        $this->log("Creating ai_csv_reservation...");
        // Step 1
        $this->sqlPut("DROP TABLE IF EXISTS ai_csv_reservation");
        $columnDefinitions = [];
        foreach (self::SHEET_COLUMNS as $columnName=>$columnDetails) {
            $columnDefinitions[] = "$columnName ".$columnDetails['type'] ." DEFAULT NULL";
        }
        $this->sqlPut("
            CREATE TABLE ai_csv_reservation
            (
                ".join(",", $columnDefinitions)."
            ) ENGINE=Aria DEFAULT CHARSET=utf8mb4 PAGE_CHECKSUM=1
        ");

        $this->log("Creating ai_reservation...");
        // create copy of rv_reservation table schema as ai_reservation
        $this->sqlPut("DROP TABLE IF EXISTS ai_reservation");
        $this->sqlPut("CREATE TABLE ai_reservation SELECT * FROM rv_reservation WHERE 1=0");
        $this->sqlPut("ALTER TABLE ai_reservation ADD `key` VARCHAR(255) FIRST");

        $this->log("Creating ai_reservation_item...");
        // create copy of rv_reservation_item table schema as ai_reservation_item
        $this->sqlPut("DROP TABLE IF EXISTS ai_reservation_item");
        $this->sqlPut("CREATE TABLE ai_reservation_item SELECT * FROM rv_reservation_item WHERE 1=0");
        $this->sqlPut("ALTER TABLE ai_reservation_item ADD `key` VARCHAR(255) FIRST");

        $this->log("Creating ai_extra...");
        // create copy of rv_extra table schema as ai_extra
        $this->sqlPut("DROP TABLE IF EXISTS ai_extra");
        $this->sqlPut("CREATE TABLE ai_extra SELECT * FROM rv_extra WHERE 1=0");
        $this->sqlPut("ALTER TABLE ai_extra ADD `key` VARCHAR(255) FIRST");

        $this->log("Creating ai_payment...");
        // create copy of rv_payment table schema as ai_payment
        $this->sqlPut("DROP TABLE IF EXISTS ai_payment");
        $this->sqlPut("CREATE TABLE ai_payment SELECT * FROM rv_payment WHERE 1=0");
        $this->sqlPut("ALTER TABLE ai_payment ADD `key` VARCHAR(255) FIRST");
        $this->notifications[] = "Tables created";
    }

    public function importFile($fileName) {
        $this->log("Running file import from CSV...");
        // Step 2
        if (!empty($fileName)) {
            $this->sqlPut("
                LOAD DATA INFILE '$fileName' REPLACE
                    INTO TABLE ai_csv_reservation
                    FIELDS TERMINATED BY ','
                        OPTIONALLY ENCLOSED BY '\"'
                        LINES TERMINATED BY '\\r\\n'
                    IGNORE 1 LINES
                    (
                        ".join(", ", array_keys(self::SHEET_COLUMNS))."
                    )
            ");
        }
        $this->log("File import from CSV done");
        foreach (self::SHEET_COLUMNS as $name => $details) {
            if (isset($details['index']) && $details['index'] === true) {
                $this->log("Creating index '$name'");
                $this->sqlPut("CREATE INDEX $name ON ai_csv_reservation($name);");
            }
        }
        $this->notifications[] = "File imported";
    }

    public function cleanup() {
        // Step 5
        $tableList = $this->lDB->get("SHOW TABLES LIKE 'ai_%'",3);
        foreach ($tableList as $table) {
            $this->sqlPut("DROP TABLE IF EXISTS $table");
        }
        foreach (self::INDEXES as $table => $index) {
            $this->sqlPut("DROP INDEX ai_$index ON $table;");
        }
        unset($_SESSION['reservation_import']);
        unset($_SESSION['step']);
        unset($_SESSION['ai_persona_table']);
        // Cleanup temp files
        $mask = '/tmp/reservation_import_*.*';
        array_map('unlink', glob($mask));
        $this->session();
        $this->notifications[] = "Cleanup complete";
    }

    public function step($step = 1) {
        if (!isset($_SESSION) && !isset($_SESSION['step'])) {
            $_SESSION['step'] = 1;
        } else {
            $_SESSION['step'] = $step;
        }
        $this->step = $_SESSION['step'];
        //TODO: Add step to verify imported sheet data and notify issues/anomalies to user
    }

    public function sqlPut($string) {
        global $dbcode;
        mysqli_real_query($this->lDB->conn, "CALL sp_set_environment('$dbcode');");
        mysqli_query($this->lDB->conn,$string);
        $error = mysqli_error($this->lDB->conn);
        if (!empty($error)) {
            $this->error[] = $error;
            $this->log($error. "' - WHEN EXECUTING: '$string'", "ERROR");
        }
    }

    public function setup() {
        global $dbcode;
        $GLOBALS['flagChngs'] = 0;
        if((!isset($_SESSION['reservation_import']) || empty($_SESSION['reservation_import']['tables']))) {
            $_SESSION['reservation_import']['tables'] = array(
                'ai_reservation'=>array(
                    'name'=>"Reservations (ai_reservation)",
                    'fields'=>array(
                        'key'=>'',
                        'rv_reservation_db'=>'',
                        'rv_reservation_id'=>'',
                        'rt_rate_type_id'=>'',
                        'rv_corr_persona_id'=>'',
                        'rv_agent_id'=>'',
                        'rv_commission_perc'=>'',
                        'rv_commission_deduct_yn'=>'',
                        'rv_invoice_currency_id'=>'',
                        'rv_billing_persona_id'=>'',
                        'rv_invoice_persona_id'=>'',
                        'rv_res_name'=>'',
                        'rv_agent_ref'=>'',
                        'rv_note_general'=>'',
                        'rv_note_guests'=>'',
                        'rv_note_internal'=>'',
                        'ac_pay_plan_id'=>'',
                        'rf_country_id'=>'',
                        'rf_source_ix'=>'',
                        'rf_reservation_status_id'=>'',
                        'rv_provision_expiry_date'=>'',
                        'rv_consultant_id'=>'',
                        'rv_date_recorded'=>'',
                        'rv_date_changed'=>'',
                        'rv_ref_num'=>'',
                        'rv_origin_agent_id'=>'',
                        'pr_reservation_user_id'=>''
                    )
                ),
                'ai_reservation_item'=>array(
                    'name'=>"Itinerary items (ai_reservation_item)",
                    'fields'=>array(
                        'key'=>'',
                        'ac_accomm_type_id'=>'',
                        'rv_item_date_arrive'=>'',
                        'rv_item_date_depart'=>'',
                        'rv_item_accomm_count'=>'',
                        'rv_item_adult_count'=>'',
                        'rv_item_child_count'=>'',
                        'rv_item_split_yn'=>'',
                        'rt_rate_type_id'=>'',
                        'rv_link_res_item_id'=>'',
                        'rv_item_user_id'=>'',
                        'rv_item_overide_level_ind'=>'',
                        'rv_item_overide_amt'=>''
                    )
                ),
                'ai_extra'=>array(
                    'name'=>"Extras (ai_extra)",
                    'fields'=>array(
                        'key'=>'',
                        'ac_extra_id'=>'',
                        'pr_business_id'=>'',
                        'rv_extra_units'=>'',
                        'rv_extra_date_serv'=>'',
                        'rv_extra_charge'=>'',
                        'rv_extra_discount'=>'',
                        'rv_extra_tax_perc'=>'',
                        'rv_extra_comm_rec'=>'',
                        'rv_extra_comm_pay'=>'',
                        'rv_extra_note'=>'',
                        'rv_extra_note_internal'=>'',
                        'rv_res_item_group_id'=>'',
                        'rf_currency_id'=>'',
                        'rf_tax_id'=>'',
                        'rv_extra_tax_ind'=>'',
                        'rv_extra_inv_curr_id'=>'',
                        'rv_extra_exch_rate'=>'',
                        'rv_extra_exch_expiry'=>'',
                        'pr_business_inv_id'=>''
                    )
                ),
                'ai_payment'=>array(
                    'name'=>"Payments (rv_payment)",
                    'fields'=>array(
                        'key'=>'',
                        'pr_persona_id'=>'',
                        'rv_pmnt_date'=>'',
                        'rf_bank_id'=>'',
                        'rf_mthd_pmnt_id'=>'',
                        'rv_pmnt_ref'=>'',
                        'rv_pmnt_note'=>'',
                        'rv_pmnt_amount'=>'',
                    )
                )
            );
        }
        $this->tags['title'] = "Reservation import";
        $this->tags['image'] = "";
        $this->tags['settings'] = 
            (isset($_SESSION['ai_persona_table']) ? $_SESSION['ai_persona_table'] : "") 
            . "|" .
            (isset($_SESSION['ai_agent_table']) ? $_SESSION['ai_agent_table'] : "");

        if(isset($this->post['db_code']) && $this->post['db_code']) {
            $_SESSION['reservation_import']['db_code'] = $this->post['db_code'];
        }
        $this->tags['db_code'] = isset($_SESSION['reservation_import']['db_code']) && !empty($_SESSION['reservation_import']['db_code']) ? $_SESSION['reservation_import']['db_code'] : $dbcode;
        $clientName = $this->lDB->get("
            SELECT
                rf_default.rf_site_title
            FROM
                rf_default
        ",4);
        $this->tags['clientName'] = $clientName?$clientName:"<i>[Client site title not set in defaults]</i>";
        $this->tags['codeVersion'] = getSystemVersion();
        $this->tags['dbVersion'] = $this->lDB->get("
            SELECT
                rf_database.rf_db_version_db
            FROM
                rf_database
            WHERE
                rf_database.rf_db_code = '$dbcode'
        ",4);

        foreach(self::CHECKS as $check) {
            if(isset($_SESSION['reservation_import'][$check]) && $_SESSION['reservation_import'][$check] == 1) {
                $this->tags[$check . '_checked'] = "checked";
            } else {
                $this->tags[$check . '_checked'] = "";
            }
        }
    }

    public function action() {
        $this->isAuthorised($this->userStatusId);
        set_time_limit(0);
        ini_set('memory_limit','4024M');
        ini_set('session.cookie_lifetime', 60 * 60 * 24 * 100);
        ini_set('session.gc_maxlifetime', 60 * 60 * 24 * 100);
        $this->setup();
        $this->session();
        switch ($this->action) {
            case 'settings':
                $this->log("Saving settings...");
                $this->step(2);
                $this->log("Done");
                break;
            case 'reset':
                $this->log("Resetting...");
                $this->cleanup();
                $this->log("Done");
                break;
            case 'template':
                $this->log("Getting template...");
                $this->csvTemplate();
                $this->log("Done");
                break;
            case 'importfile': // Step 1
                $this->log("Importing file...");
                if (!empty($this->files['filename'])) {
                    $fileName = "/tmp/$this->filePrefix".str_replace(" ", "", $this->files['filename']["name"]);
                    if (move_uploaded_file($this->files['filename']["tmp_name"], $fileName)) {
                        $this->createTables();
                        $this->importFile($fileName);
                        if (empty($this->error)) {
                            $this->staging();
                            $this->step(3);
                        } else {
                            $this->error[] = "Error importing file. ERROR: '".$this->error ."'";
                            $this->step(1);
                        }
                    } else {
                        $this->error[] = "Error uploading file";
                        $this->step(1);
                    }
                } else {
                    $this->error[] = "No filename specified";
                    $this->step(1);
                }
                $this->log("Done");
                break;
            case 'import': // Step 2
                $this->log("Importing data...");
                foreach(self::CHECKS as $check) {
                    if(isset($this->post[$check]) && $this->post[$check]) {
                        $_SESSION['reservation_import'][$check] = 1;
                    } else {
                        $_SESSION['reservation_import'][$check] = 0;
                    }
                }
                $this->importData();
                $this->step(3);
                $this->log("Done");
                break;
            case 'cleanup': // Step 3
                $this->log("Cleanup...");
                $this->cleanup();
                $this->log("Done");
                break;
            default:
                break;
        }
        return $this;
    }

    public function el($text, $type = "DEBUG") {
        if (!empty($text)) {
            error_log("RI $type: ".$text);
        }
    }

    public function session($post=array()) {
        $_SESSION['reservation_import'] = isset($_SESSION['reservation_import'])?$_SESSION['reservation_import']:array();
        if (empty($_SESSION['reservation_import'])) {
            $_SESSION['reservation_import']['tables'] = array();
            $_SESSION['reservation_import']['db_code'] = "";
            $_SESSION['reservation_import']['codeVersion'] = "";
            $_SESSION['reservation_import']['dbVersion'] = "";
            $_SESSION['reservation_import']['use_key'] = "1";
            $_SESSION['reservation_import']['use_agent'] = "";
            $_SESSION['reservation_import']['audit'] = "1";
            $_SESSION['reservation_import']['regen'] = "1";
            $_SESSION['reservation_import']['propertyId'] = "";
            $_SESSION['reservation_import']['importFileNameClean'] = "";
            $this->step();
        }
        $_SESSION['reservation_import']['propertyId'] = isset($post['cmbProperty'])&&!empty($post['cmbProperty'])?$post['cmbProperty']:$_SESSION['reservation_import']['propertyId'];
        $_SESSION['reservation_import']['importFileNameClean'] = isset($this->files['filename']["name"])?basename($this->files['filename']['name']):$_SESSION['reservation_import']['importFileNameClean'];
    }

    public function setHeader($filename, $filesize) {
        // disable caching
        $now = gmdate("D, d M Y H:i:s");
        header("Expires: Tue, 01 Jan 2001 00:00:01 GMT");
        header("Cache-Control: max-age=0, no-cache, must-revalidate, proxy-revalidate");
        header("Last-Modified: {$now} GMT");

        // force download  
        header("Content-Type: application/force-download");
        header("Content-Type: application/octet-stream");
        header("Content-Type: application/download");
        header('Content-Type: text/x-csv');

        // disposition / encoding on response body
        if (isset($filename) && strlen($filename) > 0)
            header("Content-Disposition: attachment;filename={$filename}");
        if (isset($filesize))
            header("Content-Length: ".$filesize);
        header("Content-Transfer-Encoding: binary");
        header("Connection: close");
    }

    public function csvTemplate() {
        $columnDescriptions = array();
        foreach (self::SHEET_COLUMNS as $name=>$values) {
            if ($values['sheet_heading']) {
                $columnDescriptions[] = self::SHEET_COLUMNS[$name]['description'];
            }
        }
        $csv = fopen("php://output", 'w');
        fputcsv($csv, $columnDescriptions);
        $fileName = 'template.csv';
        $fstat = fstat($csv);
        $this->setHeader($fileName, $fstat['size']);
        fpassthru($csv);
        fclose($csv);
        die();
    }

    public function log($message, $level="INFO") {
        $folder = __DIR__ . "/../../utils/reservation_import/log";
        if (!file_exists($folder)) {
            mkdir($folder, 0777, true);
        }
        $fileName = $folder."/reservation_import_" . date('Y-m-d') . '.log';
        file_put_contents($fileName, $level.": ".$message . "\n", FILE_APPEND);
    }
}
