<?php

namespace Resrequest\Authentication\Service;

use Zend\Authentication\AuthenticationService;
use DateTime;
use DateInterval;
use Resrequest\Authentication\Service\Password;
use ApigilityConsumer\Service\ClientService;

require_once(__DIR__ . "/../../../../../Application/src/Resrequest/legacy/functions.php");

/**
 * Authenticate with ResRequest Enterprise system
 */
class Authenticate
{
	private $config;

	protected $userid;
	protected $userName;
	protected $password;
	protected $context;
	protected $authenticated = false;
	protected $connection;
	protected $zendAuthentication;
	protected $passwordService;
	protected $identity;
  protected $legacyFolder;

    const AUTHENTICATE_CONTEXT_INTERNAL = 1;
    const AUTHENTICATE_CONTEXT_EXTERNAL = 2;
    const TOKEN_TYPE_ACCESS = 1;
    const TOKEN_TYPE_REFRESH = 2;

	public function __construct(
		$container,
		$entityManager,
		AuthenticationService $zendAuthentication,
		ClientService $clientService,
		Password $passwordService,
		$context = Authenticate::AUTHENTICATE_CONTEXT_INTERNAL)
	{
		$this->config = $container->get('Config');
		$this->connection = $entityManager->getConnection();
    $this->legacyFolder = getcwd() . $this->config['legacy_folder'];
		$this->zendAuthentication = $zendAuthentication;
		$this->clientService = $clientService;
		$this->passwordService = $passwordService;
        // TODO determine context based on API client_id = 'resrequest_enterprise'
		$this->context = $context;

		if (!empty($_SESSION['loggedIn']) && $_SESSION['loggedIn'] === true && !empty($_COOKIE['rrq_token'])) {
			$userDetails = $this->getUserById($_SESSION['userid']);
			$userName = $userDetails['pr_user_name'];
			$token = json_decode($_COOKIE['rrq_token'], true);
			$lifeTime = (int)$this->config['zf-oauth2']['options']['refresh_token_lifetime'];
			$this->updateRefreshTokenLifetime($token['refresh_token'], $userName, $lifeTime);
			$sc_group_id = $userDetails['sc_group_id'];
		}
		$this->clearExpiredTokens();
	}

	public function canAccessEnvironment($sc_group_id) {
		$scEnvGroupCount = $this->connection->fetchColumn('SELECT COUNT(*) FROM sc_env_group');
		if (!$scEnvGroupCount) {
			// If sc_env_group not populated allow access to all user groups.
			return true;
		}
		$environment_id = isset($GLOBALS['environment_id']) && !empty($GLOBALS['environment_id']) ? $GLOBALS['environment_id'] : "3";
		$canAccess = $this->connection->fetchColumn("
			SELECT
				COUNT(*)
			FROM
				sc_env_group
			WHERE
				sc_group_id = ?
				AND rf_database_id = ?
		", array($sc_group_id, $environment_id));
	    return $canAccess > 0 ? true : false;
	}

	public function login($userName, $password)
	{
		$this->username = $userName;
		$this->password = $password;

		$this->zendAuthentication->getAdapter()
			->setIdentity($userName)
			->setCredential($password);
		$authentication = $this->zendAuthentication->authenticate();
		$this->identity = $authentication->getIdentity();
		$this->authenticated = $authentication->isValid();
		if ($this->authenticated) {
			$this->setupContext($this->getUserByName($userName), $password);
			db_ad_user_log_insert();
			// set token cookie for client side access
			$this->setToken($this->getIdentity());
		} else {
			session_destroy();
		}

		return $this->authenticated;
	}

	public function logout($userName)
	{
		session_restart();
		$_SESSION = array();
		$params = session_get_cookie_params();
		setcookie(session_name(), '', time() - 42000,
			$params["path"], $params["domain"],
			$params["secure"], $params["httponly"]
		);
		setcookie('rrq_token', '', time() - 42000, "/", $params["domain"]);
		session_destroy();

		if (!empty($_COOKIE['rrq_token'])) {
			$token = json_decode($_COOKIE['rrq_token'], true);
			$this->revokeAccessToken($token['access_token'], $userName);
			$this->revokeRefreshToken($token['refresh_token'], $userName);
		}

		$this->clearExpiredTokens();

        $storage = $this->zendAuthentication->getStorage();
        $storage->clear();
        $this->setAuthenticated(false);
	}

	public function setupContext(array $userContext, $password=false)
	{
		$this->userid = isset($userContext['pr_user_id'])?$userContext['pr_user_id']:null;
		if ($this->userid) {
			switch ($this->context) {
                case $this::AUTHENTICATE_CONTEXT_INTERNAL:
					// legacy compatibility
					global $userid, $pr_sys_code,
						$wizResId, $pr_agent_link, $is_an_agent,
						$pr_business_link, $securitygroup,
						$sc_grp_desc, $defaultHomepages;

					require_once($this->legacyFolder . '/class.mysqldb.php');
					require_once($this->legacyFolder . '/functions.system.php');
					require_once($this->legacyFolder . '/inc.setup.php');
					require_once($this->legacyFolder . '/db.pf_field.php');
					require_once($this->legacyFolder . '/functions.reservation.itinerary.php');
					require_once($this->legacyFolder . '/db.ad_user_log.php');
					require_once($this->legacyFolder . '/functions.php');

					$_SESSION['calendarParms'] = isset($_SESSION['calendarParms'])? $_SESSION['calendarParms'] : "";
					$_SESSION['calPropList'] = isset($_SESSION['calPropList'])? $_SESSION['calPropList'] : "";
					$_SESSION['calAccommList'] = isset($_SESSION['calAccommList'])? $_SESSION['calAccommList'] : "";
					$_SESSION['calAgentId'] = isset($_SESSION['calAgentId'])? $_SESSION['calAgentId'] : "";
					$_SESSION['canCalWizard'] = isset($_SESSION['canCalWizard'])? $_SESSION['canCalWizard'] : "";
					$_SESSION['blockRateOverLap'] = isset($_SESSION['blockRateOverLap'])? $_SESSION['blockRateOverLap'] : "";
					$_SESSION['calRateList'] = isset($_SESSION['calRateList'])? $_SESSION['calRateList'] : "";
					$_SESSION['addedExpiry'] = isset($_SESSION['addedExpiry'])? $_SESSION['addedExpiry'] : "";
					$_SESSION['calStepNumber'] = isset($_SESSION['calStepNumber'])? $_SESSION['calStepNumber'] : "";
					$_SESSION['reportURL'] = isset($_SESSION['reportURL'])? $_SESSION['reportURL'] : "";
					$_SESSION['reportProfile'] = isset($_SESSION['reportProfile'])? $_SESSION['reportProfile'] : "";
					$_SESSION['calStartDate'] = isset($_SESSION['calStartDate'])? $_SESSION['calStartDate'] : "";
					$_SESSION['calEndDate'] = isset($_SESSION['calEndDate'])? $_SESSION['calEndDate'] : "";
					$_SESSION['userid'] = $this->userid;
					$_SESSION['wizResId'] = isset($_SESSION['wizResId'])? $_SESSION['wizResId'] : "";
					$_SESSION['userName'] =  $userContext['pr_user_name'];
					$_SESSION['loggedIn'] = true;
					unset($wizResId);

					$GLOBALS['userid'] = $this->userid;
					$userid = $GLOBALS['userid'];
					$GLOBALS['pr_name_first'] = $pr_name_first = $userContext['pr_name_first'];
					$GLOBALS['pr_name_last'] = $pr_name_last = $userContext['pr_name_last'];
					$GLOBALS['pr_sys_code'] = $userContext['pr_sys_code'];
					$GLOBALS['pr_business_link'] = $pr_business_link = idBusLink($this->userid);
					$GLOBALS['pr_agent_link'] = $pr_agent_link = idAgentLink($this->userid);
					$GLOBALS['is_an_agent'] = $is_an_agent = isAgent($this->userid);
					require_once($this->legacyFolder . '/ac_logon.php');
					$this->passwordService->setupContext($password, $userContext['pr_user_id']);
					$this->passwordService->finalise();
					$GLOBALS['defaultHomepages'] = $defaultHomepages = db_pf_option_get_defaults(db_pf_object_by_name("user_preferences"));

					cacheClear();
					clearTimedOutReservations();
					break;
                case $this::AUTHENTICATE_CONTEXT_EXTERNAL:
					break;
			}
		}
	}

	public function setToken($token)
	{
		$params = session_get_cookie_params();
		setcookie('rrq_token', json_encode($token), 0, "/", $params["domain"]);
	}

	/**
	 * Clears all expired Oauth access and refresh tokens
	 *
	 * @return void
	 */
  public function clearExpiredTokens() {
  	$environment_id = isset($GLOBALS['environment_id']) && !empty($GLOBALS['environment_id']) ? $GLOBALS['environment_id'] : "3";
    $timezone = $this->connection->fetchColumn("
        SELECT
          rf_db_time_zone
        FROM
          rf_database
        WHERE
          rf_database_id = ?
      ", [$environment_id]);
    if(empty($timezone) || !in_array($timezone,timezone_identifiers_list())) {
      $timezone = 'Africa/Johannesburg'; // Default timezone
    }
    set_timezone($timezone);
    if ($timezone) {
      // If a timezone is not set prevent tokens from being cleared
      $query = $this->connection->prepare("DELETE FROM oauth_refresh_tokens WHERE expires < '".date("Y-m-d H:i:s", time()-86400)."'");
      $query->execute();
      $query = $this->connection->prepare("DELETE FROM oauth_access_tokens WHERE expires < '".date("Y-m-d H:i:s", time()-86400)."'");
      $query->execute();
    }
  }

	public function revokeAccessToken($accessToken, $userName = false) {
		if ($userName === false) {
			$query = $this->connection->prepare("DELETE FROM oauth_access_tokens WHERE access_token = ?", $accessToken);
			$query->execute();
		} else {
			$query = $this->connection->executeQuery("DELETE FROM oauth_access_tokens WHERE access_token = :accessToken AND user_id = :userName", array(':accessToken' => $accessToken, ':userName' => $userName));
		}
	}

	public function revokeRefreshToken($refreshToken, $userName = false) {
		if ($userName === false) {
			$query = $this->connection->prepare("DELETE FROM oauth_refresh_tokens WHERE refresh_token = ?", $refreshToken);
			$query->execute();
		} else {
			$this->connection->executeQuery("DELETE FROM oauth_refresh_tokens WHERE refresh_token = :refreshToken AND user_id = :userName", [':refreshToken' => $refreshToken, ':userName' => $userName]);
		}
	}

	public function getRefreshTokenByUser($userName) {
		$refreshToken = $this->connection->fetchColumn("SELECT refresh_token FROM oauth_refresh_tokens WHERE user_id = ?" , [$userName]);
		if (empty($refreshToken)) {
			return false;
		} else {
			return $refreshToken;
		}
	}

	public function refreshTokenByCookie($cookie) {
		if (!empty($cookie)) {
			$token = json_decode($cookie, true);
			if (!empty($token['refresh_token'])) {
				$tokenData = $this->updateTokensExpiry($token);
				$this->setToken($tokenData);
				return $tokenData;
			} else {
				$this->setToken(false);
				return false;
			}
		} else {
			$this->setToken(false);
			return false;
		}
	}

	public function updateTokensExpiry($token) {
		$maxLifetime = ini_get("session.gc_maxlifetime");

		// Get the session expiry time
		$expiryDateTime = new DateTime('+ ' . $maxLifetime . ' seconds');
		$expiryDateTimeString = $expiryDateTime->format('Y-m-d H:i:s');

		$query = $this->connection->prepare("UPDATE oauth_access_tokens SET expires = :expiry WHERE access_token = :accessToken");
		$query->bindValue("accessToken", $token['access_token']);
		$query->bindValue("expiry", $expiryDateTimeString);
		$query->execute();

		$query = $this->connection->prepare("UPDATE oauth_refresh_tokens SET expires = :expiry WHERE refresh_token = :refreshToken");
		$query->bindValue("refreshToken", $token['refresh_token']);
		$query->bindValue("expiry", $expiryDateTimeString);
		$query->execute();

		return $token;
	}

	public function refreshToken($refreshToken) {
		$oauthClientId = $this->config['apigility-consumer']['oauth']['client_id'];
		$data = [
			'api-route-segment' => '/oauth',
			'form-request-method' => 'POST',

			'form-data' => [
				'grant_type' => 'refresh_token',
				'client_id' => $oauthClientId,
				'refresh_token' => $refreshToken,
			],

			'token_type' =>  'Bearer',
		];

		$clientResult = $this->clientService->callAPI($data, 100);

		if ($clientResult->success === true) {
			$tokenData = $clientResult->data;

			// Refresh token isn't returned from the API call
			// when always_issue_new_refresh_token is disabled.
			if (empty($tokenData['refresh_token'])) {
				$tokenData['refresh_token'] = $refreshToken; // Add refresh token to data
			}

			return $tokenData;
		} else {
			return false;
		}
	}

	function  updateRefreshTokenLifetime($refreshToken, $userName, $lifeTime) {
		if (!is_int($lifeTime)) {
			return false;
		}

		$now = new \DateTime();
		$now->modify('+' . $lifeTime . 'seconds');
		$lifeTime = $now->format('Y-m-d H:i:s');

		$query = $this->connection->executeQuery("
			UPDATE
				oauth_refresh_tokens
			SET
				expires = :lifeTime
			WHERE
				user_id = :userName
				AND refresh_token = :refreshToken",
			[
				':lifeTime' => $lifeTime,
				':userName' => $userName,
				':refreshToken' => $refreshToken
			]
		);
	}

	public function getUserByName($userName)
	{
		$user = $this->connection->fetchAssoc('
			SELECT
				pr_user.pr_user_id,
				pr_user.pr_user_name,
				pr_user.pr_user_password,
				pr_user.pr_user_inactive_yn,
				pr_persona.pr_name_first,
				pr_persona.pr_name_last,
				pr_persona.pr_sys_code,
                sc_user.sc_group_id
			FROM
				pr_user
				INNER JOIN pr_persona ON pr_user.pr_user_id = pr_persona.pr_persona_ix
                LEFT JOIN sc_user ON sc_user.pr_user_id = pr_user.pr_user_id
                LEFT JOIN sc_group ON sc_user.sc_group_id = sc_group.sc_group_id
			WHERE
                pr_user.pr_user_name = ?
                AND sc_group.sc_grp_inactive_yn=0',
			array($userName)
		);

		if (!$user) {
			return false;
		}

		return $user;
	}

	public function getUserById($id)
	{
		$user = $this->connection->fetchAssoc('
			SELECT
				pr_user.pr_user_id,
				pr_user.pr_user_name,
				pr_user.pr_user_password,
				pr_user.pr_user_inactive_yn,
				pr_persona.pr_name_first,
				pr_persona.pr_name_last,
				pr_persona.pr_sys_code,
                sc_user.sc_group_id
			FROM
				pr_user
				INNER JOIN pr_persona ON pr_user.pr_user_id = pr_persona.pr_persona_ix
                LEFT JOIN sc_user ON sc_user.pr_user_id = pr_user.pr_user_id
                LEFT JOIN sc_group ON sc_user.sc_group_id = sc_group.sc_group_id
			WHERE
                pr_user.pr_user_id = ?
                AND sc_group.sc_grp_inactive_yn=0',
			array($id)
		);

		if (!$user) {
			return false;
		}

		return $user;
	}

	public function getUserByToken($token, $tokenType)
	{
        $join = array();
        $where = array("sc_group.sc_grp_inactive_yn=0");
        switch ($tokenType) {
            case $this::TOKEN_TYPE_ACCESS:
                $join[] = "INNER JOIN oauth_access_tokens ON oauth_access_tokens.user_id = pr_user.pr_user_name";
                $where[] = "oauth_access_tokens.access_token = ?";
                break;
            case $this::TOKEN_TYPE_REFRESH:
                $join[] = "INNER JOIN oauth_refresh_tokens ON oauth_refresh_tokens.user_id = pr_user.pr_user_name";
                $where[] = "oauth_refresh_tokens.refresh_token = ?";
                break;
        }
		$user = $this->connection->fetchAssoc("
			SELECT
				pr_user.pr_user_id,
				pr_user.pr_user_name,
				pr_user.pr_user_password,
				pr_user.pr_user_inactive_yn,
				pr_persona.pr_name_first,
				pr_persona.pr_name_last,
				pr_persona.pr_sys_code,
                sc_user.sc_group_id
			FROM
				pr_user
				INNER JOIN pr_persona ON pr_user.pr_user_id = pr_persona.pr_persona_ix
                LEFT JOIN sc_user ON sc_user.pr_user_id = pr_user.pr_user_id
                LEFT JOIN sc_group ON sc_user.sc_group_id = sc_group.sc_group_id
                ".join("\r",$join)."
			WHERE
				".join("\rAND ",$where),
			array($token)
		);

		if (!$user) {
			return false;
		}

		return $user;
	}

	public function getContext()
	{
		return $this->context;
	}

	public function setAuthenticated($_authenticated)
	{
		$this->authenticated = $_authenticated;
		return $this->authenticated;
	}

	public function getAuthenticated()
	{
		return $this->authenticated;
	}

	public function getIdentity()
	{
		return $this->identity;
	}

	public function getAuthenticationService()
	{
		return $this->zendAuthentication;
	}

  public function setTimezone($timezone = null) {
    if (empty($timezone)) {
      $environment_id = isset($GLOBALS['environment_id']) && !empty($GLOBALS['environment_id']) ? $GLOBALS['environment_id'] : "3";
      $timezone = $this->connection->fetchColumn("
        SELECT
          rf_db_time_zone
        FROM
          rf_database
        WHERE
          rf_database_id = ?
      ", [$environment_id]);
    }
    if(empty($timezone) || !in_array($timezone,timezone_identifiers_list())) {
      $timezone = 'Africa/Johannesburg'; // Default timezone
    }
		$query = $this->connection->prepare("SET time_zone = '$timezone'");
		$query->execute();
    set_timezone($timezone);
  }
}
