<?php

namespace Resrequest\Authentication\Service;

use Zend\Authentication\AuthenticationService;
use DateTime;
use DateInterval;
use Resrequest\DB\Enterprise\Entity\PrUser;

/**
 * Password handling for ResRequest Enterprise system
 */
class Password
{
	protected $em;
	protected $connection;
	protected $databaseService;

	protected $password;
	protected $pr_user_id;
	protected $rules;

	public function __construct($entityManager, $databaseService)
	{
		$this->em = $entityManager;
		$this->databaseService = $databaseService;
		$this->connection = $this->em->getConnection();
		$this->rules = $this->connection->fetchAssoc("
			SELECT
				rf_auth_previous_password_yn,
				rf_auth_previous_password_limit,
				rf_auth_minimum_length_yn,
				rf_auth_minimum_length_limit,
				rf_auth_uppercase_yn,
				rf_auth_lowercase_yn,
				rf_auth_numeric_yn,
				rf_auth_special_yn,
				rf_auth_force_change_yn
			FROM
				rf_default
		");
	}

	public function setupContext($password, $pr_user_id=false)
	{
		$this->password = $password;
		$this->pr_user_id = $pr_user_id;
	}

	public function hash()
	{
		return password_hash($this->password, PASSWORD_DEFAULT);
	}

	public function verify($hash=false)
	{
		if($hash === false) {
			if(empty($this->pr_user_id)) {
				return false;
			}
			$user = $this->connection->fetchAssoc("
				SELECT
					pr_user.pr_user_password,
					pr_user.pr_auth_password_type_ind
				FROM
					pr_user
				WHERE
					pr_user.pr_user_id = ?
			",[
				$this->pr_user_id
			]);
			if($user === false) {
				return false;
			}

			if($user['pr_auth_password_type_ind'] == PrUser::PASSWORD_TYPE_INSECURE) {
				return (strtolower($user['pr_user_password']) == strtolower($this->password));
			} else {
				$hash = $user['pr_user_password'];
			}
		}
		return password_verify($this->password, $hash);
	}

	public function invalid()
	{
		$errors = [];
		if(!empty($this->rules['rf_auth_minimum_length_yn']) && mb_strlen($this->password) < $this->rules['rf_auth_minimum_length_limit']) {
			$errors[] = "Password too short. It must be at least " . $this->rules['rf_auth_minimum_length_limit'] . " characters.";
		}

		if(!empty($this->rules['rf_auth_lowercase_yn']) && !preg_match("/[a-z]/",$this->password)) {
			$errors[] = "Password must contain at least one lowercase letter.";
		}

		if(!empty($this->rules['rf_auth_uppercase_yn']) && !preg_match("/[A-Z]/",$this->password)) {
			$errors[] = "Password must contain at least one uppercase letter.";
		}

		if(!empty($this->rules['rf_auth_numeric_yn']) && !preg_match("/[0-9]/",$this->password)) {
			$errors[] = "Password must contain at least one number.";
		}

		if(!empty($this->rules['rf_auth_special_yn']) && !preg_match("/[^A-Za-z0-9]/",$this->password)) {
			$errors[] = "Password must contain at least one special character.";
		}

		if(!empty($this->rules['rf_auth_previous_password_yn']) && !empty($this->pr_user_id)) {
			$previous = $this->em->getRepository('Resrequest\DB\Enterprise\Entity\PrUserPassword')->findBy([
				'prUserId'=>$this->pr_user_id
			],[
				'adCreateDate'=>'DESC'
			],$this->rules['rf_auth_previous_password_limit']);
			foreach($previous as $item) {
				if($this->verify($item->getPrUserPassword())) {
					$errors[] = "Password may not match any of the previous " . $this->rules['rf_auth_previous_password_limit'] . " passwords used.";
					break;
				}
			}
		}

		if(sizeof($errors) > 0) {
			return $errors;
		}

		return false;
	}

	public function finalise()
	{
		if(empty($this->pr_user_id)) {
			return false;
		}

		$user = $this->em->getRepository('Resrequest\DB\Enterprise\Entity\PrUser')->find($this->pr_user_id);
		if(empty($user)) {
			return false;
		}

		$this->databaseService->masterOverride(true);
		// Force save of hashed password if in clear text
		if($user->getPrAuthPasswordTypeInd() == PrUser::PASSWORD_TYPE_INSECURE) {
			$user->setPrUserPassword($this->hash());
			$user->setPrAuthPasswordTypeInd(PrUser::PASSWORD_TYPE_NORMAL);
			$this->em->flush();
		}
		$this->databaseService->masterOverride(false);

		// If the username and password match or secure passwords are required
		// and the user's password is not secure force them to change it before
		// allowing access.
		// This check is only performed upon login, when we actually know what the password is.
		$_SESSION['require_password_change'] = false;
		if(
			!empty($this->password) &&
			!empty($this->rules['rf_auth_force_change_yn']) &&
			$this->invalid()
		) {
			$_SESSION['require_password_change'] = "invalid";
		}

		if($user->getPrUserName() == $this->password) {
			$_SESSION['require_password_change'] = "new";
		}

		return true;
	}
}
