<?php

namespace Resrequest\Authorisation\Service;

use Zend\EventManager\EventManagerAwareInterface;
use Zend\Permissions\Acl\Acl;
use Zend\Permissions\Acl\Role\GenericRole as Role;
use Zend\Permissions\Acl\Resource\GenericResource as Resource;
use Zend\EventManager\EventManagerAwareTrait;
use Resrequest\Authorisation\Service\AuthoriseService;

class AclService implements EventManagerAwareInterface {

    use EventManagerAwareTrait;

    protected $em;
    protected $acl;
    protected $globalConfig;
    protected $publicRoutes;
    protected $disabledRoutes;
    protected $authoriseService;

    const PUBLIC_ACCESS_GROUP = 4;
    const ARCHIVE_ACCESS_GROUP = -1;

    public function __construct(Acl $acl, $em, AuthoriseService $authoriseService, $globalConfig) {
        $this->globalConfig = $globalConfig;
        $this->publicRoutes = $globalConfig['enterprise']['public_routes']; // Public pages/popups
        $this->disabledRoutes = $globalConfig['enterprise']['disabled_routes']; // Disabled pages/popups
        $this->authoriseService = $authoriseService;
        $this->em = $em;
        $this->setAcl($acl);
        $this->importAcl();
    }

    /**
     * Imports the ACL from the session
     *
     * @return void
     */
    function importAcl() {
        if (empty($_SESSION['acl'])) {
            return false;
        } else {
            $acl = $_SESSION['acl'];
        }

        foreach ($acl as $accessGroup => $resources) {
            $this->allowResources(array($accessGroup => $resources['allow']));
            $this->denyResources(array($accessGroup => $resources['deny']));
        }
    }

    public function setAcl(Acl $acl) {
        $this->acl = $acl;
        return $this;
    }

    public function getAcl() {
        return $this->acl;
    }

    /**
     * Allow resources for a given role.
     *
     * @param array $config
     * @return void
     */
    public function allowResources(array $config) {
        $acl = $this->getAcl();

        foreach ($config as $role => $resources) {
            if (!$acl->hasRole($role)) {
                $acl->addRole(new Role($role));
                $this->aclPrepareRole($role);
            }

            $acl->removeAllow($role);

            foreach ($resources as $resource) {
                // Remove resource from 'deny' array
                $_SESSION['acl'][$role]['deny'] = array_diff($_SESSION['acl'][$role]['deny'], array($resource));
                if (!$acl->hasResource($resource)) {
                    $acl->addResource(new Resource($resource));
                }
                if (!in_array($resource, $_SESSION['acl'][$role]['allow'], true)) {
                    // Add role and resource to ACL session array
                    array_push($_SESSION['acl'][$role]['allow'], $resource);
                }
                $acl->allow($role, $resource);
            }
        }
    }

    /**
     * Deny resources for a given role.
     *
     * @param array $config
     * @return void
     */
    public function denyResources(array $config) {
        $acl = $this->getAcl();

        foreach ($config as $role => $resources) {
            if (!$acl->hasRole($role)) {
                $acl->addRole(new Role($role));
                $this->aclPrepareRole($role);
            }

            $acl->removeDeny($role);

            foreach ($resources as $resource) {
                // Remove resource from 'allow' array
                $_SESSION['acl'][$role]['allow'] = array_diff($_SESSION['acl'][$role]['allow'], array($resource));
                if (!$acl->hasResource($resource)) {
                    $acl->addResource(new Resource($resource));
                }
                if (!in_array($resource, $_SESSION['acl'][$role]['deny'], true)) {
                    // Add role and resource to ACL session array
                    array_push($_SESSION['acl'][$role]['deny'], $resource);
                }
                $acl->deny($role, $resource);
            }
        }
    }

    public function isJobAllowed($role, $resource) {
        if (in_array($resource, $this->disabledRoutes, true)) {
            return false;
        }
        if (in_array($resource, $this->publicRoutes, true)) {
            return true;
        }
        if ($this->getAcl()->hasResource($resource) && $this->getAcl()->hasRole($role)) {
            if ($this->acl->isAllowed($role, $resource) == true) {
                return true;
            } else {
                return false;
            }
        } else {
            return false;
        }
    }

    /**
     * Check if route is allowed. Allows wildcards in route in the form of an asterisk (*)
     * 
     * Route matching rules:
     *   1. /rooming = /rooming and only /rooming
     *   2. /rooming/ = /rooming/ and only /rooming/
     *   3. /rooming* = /rooming, /rooming/ and /rooming/something but not /roominghacker
     *   4. /rooming/* = /rooming/ and /rooming/something but not /rooming or /roominghacker
     *
     * @param mixed $role
     * @param mixed $resource
     * @return boolean
     */
    public function isRouteAllowed($role, $resource) {
        if (in_array($resource, $this->publicRoutes, true)) {
            return true;
        }
        $routeQuery = $this->em->createQueryBuilder();

        $routes = $routeQuery // Get routes
            ->select('route.scRouteName')
            ->from('Resrequest\DB\Enterprise\Entity\ScRoute', 'route')
            ->getQuery()
            ->getResult();

        $matchedRoute = false;

        foreach ($routes as $route) { // Match route
            $route = $route['scRouteName'];

            if (str_replace("*", '', $route) == $resource) { // Check literal match of route first
                $matchedRoute = $route;
                break;
            }
            if (fnmatch($route, $resource)) { // Route match
                $asteriskPos = strpos($route, '*');
                if ($asteriskPos === false) {
                    $matchedRoute = $route;
                    break;
                } else {
                    if (empty(strpos($route, '/*'))) { // Only do extra validation on route like '/rooming*', not 'rooming/*'
                        $tempRoute = substr_replace($route, "/", $asteriskPos, 0); // Add '/' before asterisk (*)

                        if (fnmatch($tempRoute, $resource)) { // Only allow route /rooming/*, not /roominghacker
                            $matchedRoute = $route;
                            break;
                        } else {
                            continue;
                        }
                    } else {
                        $matchedRoute = $route;
                        break;
                    }
                }
            }
        }

        if ($matchedRoute === false) { // If the route couldn't be matched
            return false;
        }

        if ($this->getAcl()->hasResource($matchedRoute) && $this->getAcl()->hasRole($role)) {
            if ($this->acl->isAllowed($role, $matchedRoute) == true) {
                return true;
            } else {
                return false;
            }
        } else {
            return false;
        }
    }

    /**
     * Prepares the ACL session array
     *
     * @param mixed $role
     * @return void
     */
    private function aclPrepareRole($role) {
        // Setup ACL session array
        if (!isset($_SESSION['acl'][$role])) {
            $_SESSION['acl'][$role] = array();
        }

        if (!isset($_SESSION['acl'][$role]['allow'])) {
            $_SESSION['acl'][$role]['allow'] = array();
        }
        if (!isset($_SESSION['acl'][$role]['deny'])) {
            $_SESSION['acl'][$role]['deny'] = array();
        }
    }

    /**
     * Adds an Enterpise user access group and it's resources to the ACL.
     * The ACL is stored in the session and is only updated every set amount of time.
     * This prevents multiple access queries on every request, which can be very expensive.
     *
     * @param integer $userGroupId Enterprise user access group to add to ACL
     * @param boolean $force Force update of user access regardless of last update time
     * @return void
     */
    public function addEnterpriseAccessGroup(int $userGroupId, bool $force = false) {
        $routes = [];
        $resources = [];
        $acl = $this->getAcl();

        $aclReloadInterval = $this->globalConfig['enterprise']['user_access']['acl_reload_interval']; // The amount of time between reloading of users ACL
        
        if ($force === false) {
            if (!empty($_SESSION['acl'][$userGroupId]['time_updated'])) {
                if ($_SESSION['acl'][$userGroupId]['time_updated'] + $aclReloadInterval >= time()) { // If the interval time has passed, reload user access for the user
                    return false; // Skip updating user group access
                }
            }
        }

        $jobs = array_keys($this->authoriseService->getAllowedJobs($userGroupId));
        $routes = array_flip($this->authoriseService->getAllowedRoutes($userGroupId));

        $resources =  array_merge($resources, $jobs, $routes);

        // Clear user group resources
        if ($acl->hasRole($userGroupId)) {
            $acl->removeRole($userGroupId);
        }

        $this->allowResources(array($userGroupId => $resources));

        $_SESSION['acl'][$userGroupId]['time_updated'] = time();

        return true;
    }
}
