<?php

namespace Resrequest\Authorisation\Service;

/**
 * Authorise
 */
class AuthoriseService {

    protected $entityManager;
    protected $connection;

    public function __construct($entityManager) {
        $this->entityManager = $entityManager;
        $this->connection = $this->entityManager->getConnection();
    }

    public function getAccess($accessRequest, $userGroupId = false)
    {
        $restrictionList = array();
        $accessQueryResult = array();

        if(empty($accessRequest)) {
            throw new \Exception("Empty request");
        } elseif(is_array($accessRequest)) {
            if(!empty(array_diff_key($accessRequest, array_flip(array('routes', 'functions', 'restrictions','environment'))))) {
                throw new \Exception("Bad request");
            }
        } else {
            throw new \Exception("Bad request");
        }

        if($userGroupId === false) {
            if(isset($_SESSION['userid'])) {
                $userId = $_SESSION['userid'];
            } else {
                throw new \Exception("Could not determine user ID");
            }

            $userGroupId = $this->getUserAccessGroup($_SESSION['userid']);

            if($userGroupId === false) {
                throw new \Exception("User access group could not be determined");
            }
        }

        if(array_key_exists('routes', $accessRequest)) { // Process routes
            $routeQueryResult = $this->getRouteAccessLevel($accessRequest['routes'], $userGroupId);
            $accessQueryResult['routes'] = $routeQueryResult;
        }
        if(array_key_exists('functions', $accessRequest)) { // Process functions
            $functionQueryResult = $this->getFunctionAccessLevel($accessRequest['functions'], $userGroupId);
            $accessQueryResult['functions'] = $functionQueryResult;
        }
        if(array_key_exists('restrictions', $accessRequest)) { // Process user group restrictions
            $restrictionQueryResult = $this->getRestrictionAccessLevel($accessRequest['restrictions'], $userGroupId);
            $accessQueryResult['restrictions'] = $restrictionQueryResult;
        }
        if(array_key_exists('environment', $accessRequest) && !empty($accessRequest['environment'])) { // Process environment
            $environmentQueryResult = $this->getEnvironmentAccessLevel($accessRequest['environment']);
            $accessQueryResult['environment'] = $environmentQueryResult;
        }

        $accessQueryResult = array_replace($accessRequest, $accessQueryResult); // Matches return key order to request

        return $accessQueryResult;
    }

    public function getRouteAccessLevel($routeRequest, $userGroupId = false)
    {
        $routeQuery = $this->entityManager->createQueryBuilder();
        $jobQuery = $this->entityManager->createQueryBuilder();
        $routeQueryParams = array();
        $jobQueryParams = array();
        $routeQueryResult = array();
        $jobQueryResult = array();

        $jobIdList = array(); // Stores job IDs to get access level of
        $routeList = array();

        if(empty($routeRequest)) {
            return $this->getAllowedRoutes($userGroupId);
        }

        if($userGroupId === false) {
            if(isset($_SESSION['userid'])) {
                $userId = $_SESSION['userid'];
            } else {
                throw new \Exception($_SESSION['userid']);
            }

            $userGroupId = $this->getUserAccessGroup($_SESSION['userid']);

            if($userGroupId === false) {
                throw new \Exception("User access group could not be determined");
            }
        }

        if(!is_array($routeRequest)) {
            $routeRequest = array($routeRequest);
        }

        foreach($routeRequest as $route) {
            if(preg_match("/^[0-9]+$/", $route)) { // Separate job IDs
                array_push($jobIdList, $route);
            } else {
                array_push($routeList, $route);
            }
        }

        if(!empty($jobIdList)) { // Query for jobs only
            $jobQueryParams['jobIdList'] = $jobIdList;

            $jobQuery = $jobQuery->select(array('funJob.scJobId', 'funJob.scFunctionId'))
                ->from('Resrequest\DB\Enterprise\Entity\ScFunJob', 'funJob')
                ->where('funJob.scJobId IN (:jobIdList)')
                ->setParameters($jobQueryParams)
                ->getQuery();
            foreach($jobQuery->getResult() as $index=>$result) { // Organize results into array where scJobId is the key, and scFunGrpLevel is the value
                $jobAccessLevel = $this->getFunctionAccessLevel($result['scFunctionId'], $userGroupId)[$result['scFunctionId']];
                $jobQueryResult[$result['scJobId']] = $jobAccessLevel;
            }
            foreach($jobIdList as $jobId) { // If the job doesn't exist, assume the group access level is 0
                if(!array_key_exists($jobId, $jobQueryResult)) {
                    $jobQueryResult[$jobId] = 0;
                }
            }
        }


        if(!empty($routeList)) { // Query for routes only
            $routeQueryParams['routeList'] = $routeList;

            $routeQuery = $routeQuery->select(array('route.scRouteName', 'funRoute.scFunctionId', 'route.scRouteLevel'))
                ->from('Resrequest\DB\Enterprise\Entity\ScRoute', 'route')
                ->innerJoin('Resrequest\DB\Enterprise\Entity\ScFunRoute', 'funRoute', "WITH", 'route.scRouteId = funRoute.scRouteId')
                ->where('route.scRouteName IN (:routeList)')
                ->setParameters($routeQueryParams)
                ->getQuery();

            foreach($routeQuery->getResult() as $index=>$result) {
                $routeAccessLevel = $this->getFunctionAccessLevel($result['scFunctionId'], $userGroupId)[$result['scFunctionId']];

                if($routeAccessLevel < $result['scRouteLevel']) {
                    $routeQueryResult[$result['scRouteName']] = 0;
                } else {
                    $routeQueryResult[$result['scRouteName']] = $routeAccessLevel;
                }
            }
            foreach($routeList as $route) { // If the route doesn't exist, set access level to 0
                if(!array_key_exists($route, $routeQueryResult)) {
                    $routeQueryResult[$route] = 0;
                }
            }
        }

        if(!empty($jobQueryResult) && !empty($routeQueryResult)) { // Merge job query result into route query result
            foreach($jobQueryResult as $jobId=>$accessLevel) {
                $routeQueryResult[$jobId] = $accessLevel;
            }
        } elseif(!empty($jobQueryResult)) {
            $routeQueryResult = $jobQueryResult;
        }

        $routeQueryResult = array_replace(array_flip($routeRequest), $routeQueryResult);

        return $routeQueryResult;
    }

    public function getFunctionAccessLevel($functionRequest, $userGroupId = false)
    {
        $functionIdList = array();

        if(empty($functionRequest)) {
            return $this->getAllowedFunctions($userGroupId);
        }

        if($userGroupId === false) {
            if(isset($_SESSION['userid'])) {
                $userId = $_SESSION['userid'];
            } else {
                throw new \Exception($_SESSION['userid']);
            }

            $userGroupId = $this->getUserAccessGroup($_SESSION['userid']);

            if($userGroupId === false) {
                throw new \Exception("User access group could not be determined");
            }
        }

        if(!is_array($functionRequest)) {
            $functionRequest = array($functionRequest);
        }
        foreach($functionRequest as $functionId) {
            if(preg_match("/^[0-9]+$/", $functionId)) { // Do a basic validation on the function IDs
                array_push($functionIdList, $functionId);
            } else {
                throw new \Exception("Invalid function passed");
            }
        }
        $functions = $this->getAllowedFunctions($userGroupId, $functionIdList);
            
        foreach($functionIdList as $functionId) { // If the function doesn't exist, assume group access level is 0
            if(!array_key_exists($functionId, $functions)) {
                $functions[$functionId] = 0;
            }
        }

        $functions = array_replace(array_flip($functionRequest), $functions);

        return $functions;
    }

    /**
     * Returns a list of functions and their access level for a given user access group
     * where the access level is above 0
     *
     * @param string $userGroupId The user access group to return functions for
     * @param boolean|array $functionFilter Function filter
     * @return array Key as function ID, value as access level
     */
    public function getAllowedFunctions($userGroupId, $functionFilter = false) {
        $functionResult = array();
        $functions = array();
        $functionQuery = $this->entityManager->createQueryBuilder();

        $functionQuery // Get access group resources
            ->select(array('DISTINCT funGroup.scFunctionId','funGroup.scFunGrpLevel', 'function.scFunSysCode'))
            ->from('Resrequest\DB\Enterprise\Entity\ScFunction', 'function')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\ScFunGroup', 'funGroup', "WITH", 'funGroup.scFunctionId = function.scFunctionId')
            ->where('funGroup.scGroupId = :userGroupId')
            ->setParameter('userGroupId', $userGroupId);
            
            
        if ($functionFilter !== false) {
            $functionQuery->andWhere('function.scFunctionId IN (:functionFilter)');
            $functionQuery->setParameter('functionFilter', $functionFilter);
        }

        $functionResult = $functionQuery->getQuery()->getArrayResult();

        // General access (functions that all users have access to) are not included in the above query. This is now added here
        $functionQueryGeneralAccess = $this->entityManager->createQueryBuilder();
        $functionQueryGeneralAccess // Get general access functions
            ->select(array('function.scFunctionId', '15 as scFunGrpLevel', 'function.scFunSysCode'))
            ->from('Resrequest\DB\Enterprise\Entity\ScFunction', 'function')
            ->where('function.scFunSysCode = :generalAccess')
            ->setParameter('generalAccess', 9);
        $functionGeneralAccessResult = $functionQueryGeneralAccess->getQuery()->getArrayResult();

        foreach ($functionGeneralAccessResult as $key => $function) {
            $functionResult[] = $function;
        }

        // Filter functions by sys code (sc_function.sc_fun_sys_code)
        // 1: No access for anyone.
        // 3: No access at property level.
        // 5: Read access at property level if normal access is read or higher.
        // 9: Management access for everyone.
        foreach ($functionResult as $function) {
            if (empty($function['scFunGrpLevel'])) {
                $function['scFunGrpLevel'] = 0;
            }
            if ($function['scFunSysCode'] == 1) {
                $function['scFunGrpLevel'] = 0;
            } elseif ($function['scFunSysCode'] == 3) {
                if ($GLOBALS['isPropServer']) {
                    $function['scFunGrpLevel'] = 0;
                }
            } elseif ($function['scFunSysCode'] == 5) {
                if ($GLOBALS['isPropServer']) {
                    if ($function['scFunGrpLevel'] > 5) {
                        $function['scFunGrpLevel'] = 5;
                    }
                }
            } elseif ($function['scFunSysCode'] == 9) {
                $function['scFunGrpLevel'] = 15;
            }

            if ($function['scFunGrpLevel'] != 0) {
                $functions[$function['scFunctionId']] = $function['scFunGrpLevel'];
            }
        }

        return $functions;
    }

    /**
     * Returns a list of allowed jobs and their access level for a given user access group
     * A job is determined as "allowed" if the access level to that job is 5 (read) or greater
     * 
     * @param string $userGroupId The user access group to return jobs for
     * @param boolean|array $jobFilter Job filter
     * @return array Key as job ID, value as access level
     */
    public function getAllowedJobs($userGroupId, $jobFilter = false) {
        $functionResult = array();
        $allowedJobs = array();
        $allowedJobsQuery = $this->entityManager->createQueryBuilder();

        $allowedFunctions = $this->getAllowedFunctions($userGroupId);

        if (empty($allowedFunctions)) {
            return $allowedJobs;
        }

        $allowedJobsQuery->select(['DISTINCT job.scJobId', 'funJob.scFunctionId'])
            ->from('Resrequest\DB\Enterprise\Entity\ScJob', 'job')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\ScFunJob', 'funJob', "WITH", 'funJob.scJobId = job.scJobId')
            ->where('funJob.scFunctionId IN (:allowedFunctions)')
            ->setParameter('allowedFunctions', array_keys($allowedFunctions));

        if ($jobFilter !== false) {
            $allowedJobsQuery->andWhere('job.scJobId IN (:jobFilter)');
            $allowedJobsQuery->setParameter('jobFilter', $jobFilter);
        }

        $allowedJobs = $allowedJobsQuery->getQuery()->getArrayResult();

        $tempJobs = array();
        foreach($allowedJobs as $job) {
            // Only return jobs with an access level of 5 and above
            if ($allowedFunctions[$job['scFunctionId']] >= 5) {
                $tempJobs[$job['scJobId']] = $allowedFunctions[$job['scFunctionId']];
            }
        }

        $allowedJobs = $tempJobs;
        unset($tempJobs);

        return $allowedJobs;
    }

    /**
     * Returns a list of allowed routes
     *
     * @param string $userGroupId
     * @return array Array of allowed routes
     */
    public function getAllowedRoutes($userGroupId) {
        $functionResult = array();
        $allowedRoutes = array();
        $allowedRoutesQuery = $this->entityManager->createQueryBuilder();

        $allowedFunctions = $this->getAllowedFunctions($userGroupId);

        if (empty($allowedFunctions)) {
            return $allowedRoutes;
        }

            $allowedRoutesQuery->select(['DISTINCT route.scRouteId', 'route.scRouteLevel', 'route.scRouteName', 'funRoute.scFunctionId'])
            ->from('Resrequest\DB\Enterprise\Entity\ScRoute', 'route')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\ScFunRoute', 'funRoute', "WITH", 'funRoute.scRouteId = route.scRouteId')
            ->where('funRoute.scFunctionId IN (:allowedFunctions)')
            ->setParameter('allowedFunctions', array_keys($allowedFunctions));


        $allowedRoutes = $allowedRoutesQuery->getQuery()->getArrayResult();

        $tempRoutes = array();
        foreach($allowedRoutes as $route) {
            // Only return routes with an access level of 5 and above
            if ($allowedFunctions[$route['scFunctionId']] >= $route['scRouteLevel']) {
                $tempRoutes[] = $route['scRouteName'];
            }
        }

        $allowedRoutes = $tempRoutes;
        unset($tempRoutes);

        return $allowedRoutes;
    }

    public function getRestrictionAccessLevel($restrictionRequest, $userGroupId = false)
    {
        $restrictionQuery = $this->entityManager->createQueryBuilder(); // Start queryBuilder
        $restrictionQuerySelect = array();
        $restrictionQueryResult = array();

        $restrictionList = array();

        // if(empty($restrictionRequest)) {
        //     throw new \Exception("Empty request");
        // }

        if($userGroupId === false) {
            if(isset($_SESSION['userid'])) {
                $userId = $_SESSION['userid'];
            } else {
                throw new \Exception($_SESSION['userid']);
            }

            $userGroupId = $this->getUserAccessGroup($_SESSION['userid']);

            if($userGroupId === false) {
                throw new \Exception("User access group could not be determined");
            }
        }

        $availableRestrictions = array(
            "scGrpAllocInd",
            "scGrpConsultantYn",
            "scGrpApiYn",
            "scGrpDefOptionalsYn",
            "scGrpArchiveYn",
            "scGrpForcePaymentYn",
            "scGrpAvailLimit",
            "scGrpAvailLimitYn",
            "scGrpAvailProvYn",
            "scGrpSysCode",
            "scGrpResStatusUnallocYn",
            "scGrpResStatusAllocYn",
            "scGrpCrossInvoiceYn",
            "scGrpResVoidInvoiceYn",
            "scGrpResVoidPaymentYn",
            "scGrpResOvrRateTypeYn",
            "scGrpResOvrAmountYn",
            "scGrpResOvrVarianceYn",
            "scGrpResOvrOverbookingYn",
            "scGrpResOvrSpecialsYn",
            "scGrpResOvrTotalPaxYn",
            "scGrpResSpApplyAutoYn",
            "scGrpResSpApplyManYn",
            "scGrpResLockYn",
            "scGrpResLockDays",
            "scGrpResLockInd"
        );

        if (empty($restrictionRequest)) {
            $restrictionList = $availableRestrictions;
        } else {
            if(!is_array($restrictionRequest)) {
                $restrictionRequest = array($restrictionRequest);
            }
            foreach($restrictionRequest as $restriction) { // Validate each restriction and convert to camel case for Doctrine
                $restrictionCamelCase = lcfirst(str_replace(' ', '', ucwords(str_replace('_', ' ', $restriction))));
                if(in_array($restriction, $availableRestrictions)) {
                    array_push($restrictionList, $restriction);
                } elseif(in_array($restrictionCamelCase, $availableRestrictions)) {
                    array_push($restrictionList, $restrictionCamelCase);
                } else {
                    throw new \Exception('Invalid access restriction ' . $restriction);
                }
            }
        }

        foreach($restrictionList as $restriction) {
            array_push($restrictionQuerySelect, 'accessGroup.' . $restriction);
        }

        $restrictionQuery->select($restrictionQuerySelect)
            ->from('Resrequest\DB\Enterprise\Entity\ScGroup', 'accessGroup')
            ->Where('accessGroup.scGroupId = :userGroupId')
            ->setParameter('userGroupId', $userGroupId);

        $restrictionQuery = $restrictionQuery->getQuery();
        $restrictionQueryResult = $restrictionQuery->getResult()[0];

        foreach($restrictionQueryResult as $restriction=>$value) {
            if(empty($value)) {
                $restrictionQueryResult[$restriction] = false;
            }
        }

        // $restrictionQueryResult = array_replace(array_flip($restrictionRequest), $restrictionQueryResult);

        return $restrictionQueryResult;
    }

    public function getEnvironmentAccessLevel($requests, $userGroupId = false)
    {
        $available = [
            'isMaster',
            'isProperty',
            'isOffline',
            'isWeb',
            'linkedProperties',
            'isPrimaryEnvironmentWeb'
        ];
        $camelRequests = array_map([$this,"camelCase"], $requests);
        foreach($camelRequests as $request) {
            if(!in_array($request, $available)) {
                throw new \Exception('Invalid request ' . $request);
            }
        }

        $typeInd = $this->entityManager->createQueryBuilder()
            ->select('rfDatabase.rfDbEnvTypeInd')
            ->from('Resrequest\DB\Enterprise\Entity\RfDatabase', 'rfDatabase')
            ->where('rfDatabase.rfDatabaseId = ?1')
            ->setParameter(1, $GLOBALS['environment_id'])
            ->getQuery()
            ->getSingleScalarResult();

        $dbCode = $GLOBALS['environment'];

        $results = [];
        foreach($camelRequests as $key=>$request) {
            $result = false;
            switch($request) {
            case 'isMaster':
                if($typeInd == 4) {
                    $result = true;
                } else {
                    $this->entityManager->getConnection()
                        ->exec("CALL sp_is_environment_master");
                    $isMaster = $this->entityManager->getConnection()
                        ->executeQuery("SELECT @is_env_master")
                        ->fetchColumn();
                    $result = ($isMaster == "1");
                }
                break;
            case 'isProperty':
                if($typeInd == 4) {
                    $result = true;
                }
                break;
            case 'isOffline':
                if($dbCode[0] != 'W') {
                    $result = true;
                }
                break;
            case 'isWeb':
                if($dbCode[0] == 'W') {
                    $result = true;
                }
                break;
            case 'linkedProperties':
                if($typeInd != 4) {
                    $result = [];
                } else {
                    $result = array_map(function($property) {
                        return $property['prBusinessId'];
                    },$this->entityManager->createQueryBuilder()
                        ->select('rfDbBusiness.prBusinessId')
                        ->from('Resrequest\DB\Enterprise\Entity\RfDbBusiness', 'rfDbBusiness')
                        ->where('rfDbBusiness.rfDbCode = ?1')
                        ->setParameter(1, $GLOBALS['environment'])
                        ->getQuery()
                        ->getArrayResult()
                    );
                }
                break;
            case 'isPrimaryEnvironmentWeb':
                if (empty($GLOBALS['primaryEnvironmentIsWeb'])) {
                    $result = false;
                } else {
                    $result = true;
                }
                break;
            }
            $results[$requests[$key]] = $result;
        }

        return $results;
    }

    function isRouteAllowed($userGroupid, $route) {
        $routeLevelQuery = $this->entityManager->createQueryBuilder();
        $routeAccessLevel = $this->getRouteAccessLevel($userGroupid, $route);

        $routeLevel = $routeLevelQuery
                ->select('route.scRouteLevel')
                ->from('Resrequest\DB\Enterprise\Entity\ScRoute', 'route')
                ->where('route.scRouteName = ?1')
                ->setParameter(1, $route)
                ->getQuery()
                ->getResult();

        if (empty($routeLevel) || $routeAccessLevel < $routeLevel) {
            return false;
        } else {
            return true;
        }
    }

    function getUserAccessGroup($userId) {
        $userGroupIdQuery = $this->entityManager->createQueryBuilder()
            ->select('user.scGroupId') // Get users access group ID from the user ID
            ->from('Resrequest\DB\Enterprise\Entity\ScUser', 'user')
            ->where('user.prUserId = :userId')
            ->setParameter('userId', $userId)
            ->getQuery()
            ->getResult();

        if(empty($userGroupIdQuery[0]['scGroupId'])) {
            return false; // User group couldn't be determined
        } else {
            $userGroupId = $userGroupIdQuery[0]['scGroupId'];
        }

        return $userGroupId;
    }

    public function getUserByName($userName)
	{
        $userId = $this->connection
            ->query('
                SELECT
                    pr_user.pr_user_id
                FROM
                    pr_user
                    LEFT JOIN sc_user ON sc_user.pr_user_id = pr_user.pr_user_id
                    LEFT JOIN sc_group ON sc_user.sc_group_id = sc_group.sc_group_id
                WHERE
                    pr_user.pr_user_name = "' . $userName . '"
                    AND sc_group.sc_grp_inactive_yn = 0'
            )
            ->fetchColumn();

		if (!$userId) {
			return false;
		}

		return $userId;
	}

    protected function camelCase($name) {
        return lcfirst(str_replace(' ', '', ucwords(str_replace('_', ' ', $name))));
    }
}
