<?php

namespace Resrequest\DB\Service;
use Zend\Db\Adapter\Adapter;

class EnterpriseDatabaseService
{
    public $databaseName;
    public $principalId;
    public $defaultEnvironment;

    private $scTables;
    private $sqlPath;
    private $migrationsPath;
    private $globalConfig;

    private $scTablesFile;
    private $triggersFile;
    private $triggerGenFile;
    private $tzFile;
    private $defaultsFile;
    private $versionFile;
    private $migrationsFile;
    private $templateFile;
    private $functionsFile;
    private $proceduresFile;

    protected $em;
    protected $eConn; // Enterprise database connection
    protected $cConn; // Censys database connection

    public function __construct($sm, $em = false) {
        $this->em = $em;

        if ($this->em !== false) {
            $this->eConn = $this->em->getConnection();
            $this->databaseName = $this->em->getConnection()->getDatabase();
        } else {
            $this->eConn = false;
            $this->databaseName = false;
        }


        $this->globalConfig = $sm->get('config');
        $this->defaultEnvironment = $this->globalConfig['enterprise']['enterprise_database']['default_environment'];
        $this->scTables = $this->globalConfig['enterprise']['enterprise_database']['sc_tables'];
        $this->sqlPath = realpath(__DIR__ . '/../Enterprise/Sql');
        $this->migrationsPath = realpath(__DIR__ . '/../Enterprise/Migrations');
        
        $this->triggersFile = $this->sqlPath . '/triggers.sql';
        $this->triggerGenFile = $this->sqlPath . '/trigger_gen.sql';
        $this->scTablesFile = $this->sqlPath . '/sc_tables.sql';
        $this->tzFile = $this->sqlPath . '/tz.sql';
        $this->defaultsFile = $this->sqlPath . '/defaults.sql';
        $this->versionFile = $this->sqlPath . '/version.sql';
        $this->migrationsFile = $this->sqlPath . '/migrations.sql';
        $this->templateFile = $this->sqlPath . '/template.sql';
        $this->functionsFile = $this->sqlPath . '/functions.sql';
        $this->proceduresFile = $this->sqlPath . '/procedures.sql';
    }

    public function setEnterpriseConnection($databaseName) {
        if ($this->databaseExists($databaseName) === false) {
            die("Database '$databaseName' does not exist\n");
        }

        $connectionParams = array(
            'dbname' => $databaseName,
            'user' => 'root',
            'password' => '',
            'host' => 'localhost',
            'driver' => 'pdo_mysql',
        );
        $this->databaseName = $databaseName;
        $this->eConn = \Doctrine\DBAL\DriverManager::getConnection($connectionParams, new \Doctrine\DBAL\Configuration());

        return true;
    }

    public function setCensysConnection() {
        $connectionParams = array(
            'dbname' => 'censys',
            'user' => 'root',
            'password' => '',
            'host' => 'localhost',
            'driver' => 'pdo_mysql',
        );
        $this->cConn = \Doctrine\DBAL\DriverManager::getConnection($connectionParams, new \Doctrine\DBAL\Configuration());

        return true;
    }

    public function databaseExists($databaseName) {
        $connection = @mysqli_connect('localhost', 'root', '', $databaseName);
        if($connection == true)  {
            mysqli_close($connection);
            return true;
        } else {
            return false;
        }
    }

    public function padPrincipalId($principalId) {
        $principalId = str_pad($principalId, 4, "0", STR_PAD_LEFT);
        return $principalId;
    }
    
    public function upgrade($backup = true, $displayMigrationsOutput = false) {
        $codeVersion = $this->getCodeVersion();
        $databaseVersion = $this->getDatabaseVersion();

        if (!empty($codeVersion) && $codeVersion != 'HEAD') {
            echo("Current code version is v$codeVersion\n");
        }
        
        if (empty($databaseVersion )) {
            die("Database version could not be determined. Aborting\n");
        } else {
            echo("Current database version is v$databaseVersion\n");
        }

        if (version_compare($databaseVersion, '7.19.0', '<')) {
            die("This functionality can only be used on database versions 7.19.x and up\n");
        }

        putenv("DB_NAME=$this->databaseName");
        exec('php public/index.php migrations:status', $migrationsStatus);
        $newMigrations = substr($migrationsStatus[19], 59); // Extract migrations to run from console output
        
        if ($newMigrations == 0) {
            echo("Database already upgraded\n");
            return false;
        }

        if ($backup === true) {
                $this->backup();
        } else {
            echo("Database backup skipped\n");
        }

        echo("Starting upgrade on database '$this->databaseName'\n");

        putenv("DB_NAME=$this->databaseName");

        exec('php public/index.php migrations:migrate --no-interaction', $output);
        
        if ($displayMigrationsOutput !== true) {
            $output = array_splice($output, -5);
        }

        foreach ($output as $line) {
            echo($line . PHP_EOL);
        }

        return true;
    }

    public function offlineUpgrade($backup = true, $displayMigrationsOutput = false) {
        $this->setCensysConnection();

        $principalIds = $this->cConn->query('SELECT DISTINCT cn_principal_id FROM cn_principal')->fetchAll();
        
        if (empty($principalIds)) {
            die("No principal entries, unable to determine database. Aborting\n");
        }

        if (sizeof($principalIds) > 1) {
            die("Multiple principal entries, unable to determine database. Aborting\n");
        } else {
            $principalId = $principalIds[0]['cn_principal_id'];
            $principalId = $this->padPrincipalId($principalId);
        }

        $databaseName = 'cn_live_' . $principalId;

        $this->setEnterpriseConnection($databaseName);

        $this->upgrade($backup, $displayMigrationsOutput);
    }

    public function backup() {
        $backupLocation = '../backups';
        $databaseVersion = $this->getDatabaseVersion();

        if($databaseVersion === false) {
            $fileName = date("Y-m-d_H-i-s") . '_' . $this->databaseName . '.sql.bz2';
        } else {
            $fileName = date("Y-m-d_H-i-s") . '_' . $this->databaseName . '_' . $databaseVersion . '.sql.bz2';
        }

        if (!is_dir($backupLocation)){
            mkdir($backupLocation);
        }
        
        $backupLocation = realpath($backupLocation);

        echo("Backing up database to '{$backupLocation}/{$fileName}'\n");
        exec("mysqldump --opt --add-drop-table --routines -u root {$this->databaseName} | bzip2 -c > {$backupLocation}/{$fileName}");
       
        return true;
    }

    public function build($versionNumber) {
        if (!preg_match('/^\d+\.\d+\.\d+$/', $versionNumber)) {
            die("Incorrect version number specified\n");
        }

        echo("Starting build of v$versionNumber\n");

        $databaseName = 'zzz_upgrade_' . str_replace('.', '_', $versionNumber);

        // Remove all zzz_upgrade_* databases
        echo("Cleaning previous upgrade databases\n");
        exec('mysql -u root  -e "SHOW DATABASES" | grep zzz_upgrade_* | xargs -I "@@" mysql -u root -e "DROP database \`@@\`"');
        echo("Creating database '$databaseName'\n");

        $this->mysqlExecute(false, "CREATE DATABASE $databaseName;");
        echo("Importing database template\n");
        $this->mysqlExecute($databaseName, false, $this->templateFile); // Load template into database

        $this->setEnterpriseConnection($databaseName);

        $this->insertFunctions(); // Insert procedures from functions.sql
        $this->insertProcedures(); // Insert procedures from procedures.sql
        $this->insertScTables(); // Insert procedures from sc_tables.sql
        $this->insertDefaults(); // Insert procedures from defaults.sql
        $this->insertMigrationsTable(); // Insert migrations table from migrations.sql

        $this->upgrade(false); // Run migrations

        $this->generateProceduresMigration();
        $this->generateTriggers(); // Generates trigger migration
        $this->generateFunctionsMigration();
        $this->generateMigrationsTable(); // Updates migrations.sql file
        $this->generateScTables(); // Generates SC tables migration
        $this->generateTimezoneInfo(); // Updates tz.sql file
        $this->generateVersion($versionNumber); // Generates version migration
        
        $this->insertVersion();

        $this->generateTemplate();

        return true;
    }

    /**
     * Creates a new Enterprise ready database from scratch
     *
     * @param integer $principalId
     * @return void
     */
    public function template($principalId) {
        $principalId = $this->padPrincipalId($principalId);
        $databaseName = 'cn_live_' . $principalId;

        $this->mysqlExecute(false, "CREATE DATABASE $databaseName;");
        $this->mysqlExecute($databaseName, false, $this->templateFile); // Load template into database

        $this->setEnterpriseConnection($databaseName);

        $this->insertFunctions(); // Insert procedures from functions.sql
        $this->insertProcedures(); // Insert procedures from procedures.sql
        $this->insertScTables(); // Insert procedures from sc_tables.sql
        $this->insertDefaults(); // Insert procedures from defaults.sql
        $this->insertMigrationsTable(); // Insert migrations table from migrations.sql

        $this->upgrade(); // Run migrations

        $this->insertVersion();

        return true;
    }

    public function generateTemplate() {
        if (file_exists($this->templateFile)) {
            unlink($this->templateFile);
        }
        echo("Generating database template\n");
        exec('mysqldump ' . $this->databaseName  . ' -u root --no-data --skip-trigger > ' . $this->templateFile);
        


        return true;
    }

    public function generateTriggers() {
        echo("Generating database triggers\n");
        $triggerGenSql = file_get_contents($this->triggerGenFile);

        if (file_exists($this->triggersFile)) {
            unlink($this->triggersFile);
        }

        $triggerGenSql = substr($triggerGenSql, 14); // Remove 'DELIMITER $$'
        $triggerGenSql = substr($triggerGenSql, 0, -3); // Remove final '$$'
        $statements = explode('//', $triggerGenSql); // Seperate each statement into an array
        foreach($statements as $statement) {
            $triggerGenQuery = $this->eConn->query($statement);
        }

        $triggersSql = $triggerGenQuery->fetchColumn();
        file_put_contents($this->triggersFile, $triggersSql);
        $this->generateTriggersMigration();
        echo("Database triggers generated\n");

        return true;
    }

    private function mysqlExecute($databaseName = false, $sql = false, $file = false, $force = false) {
        if ($databaseName === false) {
            $databaseName = '';
        }

        if ($force === true) {
            $force = '-f';
        } else {
            $force = '';
        }

        if ($file === false) {
            $command = 'mysql ' . $force . '-u root "' . $databaseName . '" -e "' . $sql . '" ';
        } else {
            $command = 'mysql ' . $force . '-u root "' . $databaseName . '" < ' . $file . ' ';
        }

        exec($command, $output);
        
        return true;
    }

    public function insertTriggers() {
        echo("Inserting database triggers\n");
        if (!file_exists($this->triggersFile)) {
            echo("Triggers file not found. Generating triggers\n");
            $this->generateTriggers();
        }

        $triggersSql = file_get_contents($this->triggersFile);

        $triggersSql = substr($triggersSql, 14); // Remove 'DELIMITER $$'
        $triggersSql = substr($triggersSql, 0, -4); // Remove final '$$'
        $triggers = explode('$$', $triggersSql); // Seperate each statement into an array
        foreach($triggers as $trigger) {
            $triggerGenQuery = $this->eConn->query($trigger);
        }
        echo("Database triggers inserted\n");

        return true;
    }

    public function generateScTables() {
        $scTablesList = implode(' ', $this->scTables);
        echo("Generating sc tables\n");

        exec('mysqldump ' . $this->databaseName  . ' -u root --skip-trigger --tables ' . $scTablesList, $scTablesSql);

        if (file_exists($this->scTablesFile)) {
            unlink($this->scTablesFile);
        }
        $scTablesSql = implode(PHP_EOL, $scTablesSql);
        file_put_contents($this->scTablesFile, $scTablesSql);
        echo("Sc tables generated\n");

        return true;
    }

    public function insertScTables() {
        if (!file_exists($this->scTablesFile)) {
            die("Sc tables file not found. Aborting\n");
        }
        
        $scTablesSql = file_get_contents($this->scTablesFile);

        $this->eConn->query($scTablesSql);
        echo("Sc tables inserted\n");
        return true;
    }

    public function generateTimezoneInfo() {
        echo("Generating MySQL timezone info\n");

        exec('mysql_tzinfo_to_sql /usr/share/zoneinfo', $timezoneInfo);

        if (file_exists($this->tzFile)) {
            unlink($this->tzFile);
        }

        $timezoneInfo = implode(PHP_EOL, $timezoneInfo);
        file_put_contents($this->tzFile, $timezoneInfo);
        echo("MySQL timezone info generated\n");

        return true;
    }

    public function insertTimezoneInfo() {
        if (!file_exists($this->tzFile)) {
            die("MySQL timezone info file not found. Aborting\n");
        }
        echo("Inserting MySQL timezone info\n");
        $timezoneInfoSql = file_get_contents($this->tzFile);
        $timezoneInfoSql = explode(';', $timezoneInfoSql);
        $mysqlDatabase = new \Zend\Db\Adapter\Adapter(
            array(
                'driver' => 'Mysqli',
                'database' => 'mysql',
                'username' => 'root',
                'password' => ''
            )
        );
        
        foreach($timezoneInfoSql as $statement) {
            if (empty($statement)) {
                continue;
            }
            $mysqlDatabase->query($statement, \Zend\Db\Adapter\Adapter::QUERY_MODE_EXECUTE);
        }
        echo("MySQL timezone info inserted\n");

        return true;
    }

    public function insertDefaults() {
        if (!file_exists($this->defaultsFile)) {
            echo("Defaults file not found. Aborting\n");
        }
        echo("Inserting defaults\n");
        $defaultsSql = file_get_contents($this->defaultsFile);
        
        $this->eConn->executeQuery($defaultsSql);
        
        echo("Defaults inserted\n");

        return true;
    }

    public function generateVersion($versionNumber) {
        if (!preg_match('/^\d+\.\d+\.\d+$/', $versionNumber)) {
            die("Incorrect version number specified\n");
        }
        if (file_exists($this->versionFile)) {
            unlink($this->versionFile);
        }

        $versionSql = "UPDATE rf_database SET rf_db_version_db = '$versionNumber';";

        file_put_contents($this->versionFile, $versionSql);
        $this->generateVersionMigration();
        echo("Version file v" . $versionNumber . " generated\n");

        return true;
    }

    public function insertVersion() {
        if (!file_exists($this->versionFile)) {
            echo("Version file not found. Aborting\n");
        }
        echo("Setting version\n");
        $versionSql = file_get_contents($this->versionFile);

        $this->setEnvironment();
        $this->masterOverride(true);
        $this->eConn->query($versionSql);
        $this->masterOverride(false);
        
        echo("Version set\n");

        return true;
    }

    public function getDatabaseVersion() {
        $databaseVersion = $this->eConn->query("SELECT DISTINCT rf_db_version_db FROM rf_database")->fetchColumn();
        
        if (empty($databaseVersion)) {
            return false;
        }

        return $databaseVersion;
    }

    public function setEnvironment($environment = false) {
        if ($environment === false) {
            $environment = $this->defaultEnvironment;
        }
        $setEnvironmentSql = 'CALL sp_set_environment("' . $environment . '");';

        $this->eConn->exec($setEnvironmentSql);
    }

    public function generateMigrationsTable() {
        echo("Generating migrations file\n");
        $migrationsTable = $this->globalConfig['doctrine']['migrations_configuration']['orm_enterprise']['table'];
        exec('mysqldump ' . $this->databaseName  . ' -u root --skip-trigger --tables ' . $migrationsTable, $migrationsSql);

        if (file_exists($this->migrationsFile)) {
            unlink($this->migrationsFile);
        }
        $migrationsSql = implode(PHP_EOL, $migrationsSql);
        file_put_contents($this->migrationsFile, $migrationsSql);
        echo("Migrations file generated\n");

        return true;
    }

    public function insertMigrationsTable() {
        if (!file_exists($this->migrationsFile)) {
            die("Migrations file not found. Aborting\n");
        }
        echo("Inserting migrations\n");
        $migrationsSql = file_get_contents($this->migrationsFile);

        $this->eConn->query($migrationsSql);
        echo("Migrations table inserted\n");

        return true;
    }

    public function insertProcedures() {
        echo("Inserting database stored procedures\n");
        if (!file_exists($this->functionsFile)) {
            echo("Procedures file not found. Aborting\n");
        }

        $proceduresSql = file_get_contents($this->proceduresFile);

        $proceduresSql = substr($proceduresSql, 14); // Remove 'DELIMITER //'
        $proceduresSql = substr($proceduresSql, 0, -3); // Remove final '//'
        $procedures = explode('//', $proceduresSql); // Seperate each statement into an array
        foreach($procedures as $procedure) {
            $this->eConn->query($procedure);
        }
        echo("Database stored procedures inserted\n");

        return true;
    }

    public function insertFunctions() {
        echo("Inserting database functions\n");
        if (!file_exists($this->functionsFile)) {
            echo("Functions file not found. Aborting\n");
        }

        $functionsSql = file_get_contents($this->functionsFile);

        $functionsSql = substr($functionsSql, 14); // Remove 'DELIMITER //'
        $functionsSql = substr($functionsSql, 0, -3); // Remove final '//'
        $functions = explode('//', $functionsSql); // Seperate each statement into an array
        foreach($functions as $function) {
            $this->eConn->query($function);
        }
        echo("Database functions inserted\n");

        return true;
    }

    public function createMigration($generateBoilerplate = true) {
        // Multiple migrations can be generated too quickly, causing them to have the same name,
        // overwriting the previously generated migration instead of creating a new one.
        $date = new \DateTime("now", new \DateTimeZone('UTC') );
        $date = $date->format('YmdHis');
        $migrationFile = $this->migrationsPath . '/Version' . $date . '.php';

        if (file_exists($migrationFile)) {
                echo("Migrations file already exists, sleeping 1 second before generating new migration\n");
                sleep(1);
        }

        exec("php public/index.php migrations:generate", $output);
        $migrationFile = substr($output[1], 34, -1);

        $migration = file_get_contents($migrationFile);

        if ($generateBoilerplate === true) {
            $templateCode  = '$this->addSql("CALL sp_set_environment(\'RS\')");' . PHP_EOL . '        ';
            $templateCode .= '$this->addSql("CALL sp_transfer_flagging(FALSE)");' . PHP_EOL . '        ';
            $templateCode .= '$this->addSql("CALL sp_master_override(TRUE)");' . PHP_EOL . '        ';
            $templateCode .= '// template' . PHP_EOL . '        ';
            $templateCode .= '$this->addSql("CALL sp_transfer_flagging(TRUE)");' . PHP_EOL . '        ';
            $templateCode .= '$this->addSql("CALL sp_master_override(FALSE)");' . PHP_EOL . '        ';
        } else {
            $templateCode = '// template' . PHP_EOL . '        ';
        }

        $migration = str_replace("// this up() migration is auto-generated, please modify it to your needs", $templateCode, $migration);
        file_put_contents($migrationFile, $migration);

        return $migrationFile;
    }

    public function generateProceduresMigration() {
        $proceduresMigrationFile = $this->createMigration(false);
        $proceduresMigration = file_get_contents($proceduresMigrationFile);

        $procedureCode = '';
        $procedureSql = file_get_contents($this->proceduresFile);
        $procedureSql = substr($procedureSql, 14); // Remove "DELIMITER //"
        $procedureSql = substr($procedureSql, 0, -4); // Remove final "//"
        $procedureSql = explode("//", $procedureSql); // Seperate each statement into an array
        foreach($procedureSql as $procedure) {
            $procedureCode .= '$this->addSql("' . addcslashes($procedure, '"') . '");' . PHP_EOL . '        ';
        }

        $proceduresMigration = str_replace("// template", $procedureCode, $proceduresMigration);
        file_put_contents($proceduresMigrationFile, $proceduresMigration);
    }

    public function generateFunctionsMigration() {
        $functionsMigrationFile = $this->createMigration(false);
        $functionsMigration = file_get_contents($functionsMigrationFile);

        $functionCode = '';
        $functionsSql = file_get_contents($this->functionsFile);
        $functionsSql = substr($functionsSql, 14); // Remove "DELIMITER //"
        $functionsSql = substr($functionsSql, 0, -4); // Remove final "//"
        $functionsSql = explode("//", $functionsSql); // Seperate each statement into an array
        foreach($functionsSql as $function) {
            $functionCode .= '$this->addSql("' . addcslashes($function, '"') . '");' . PHP_EOL . '        ';
        }

        $functionsMigration = str_replace("// template", $functionCode, $functionsMigration);
        file_put_contents($functionsMigrationFile, $functionsMigration);
    }

    public function generateTriggersMigration() {
        $triggersMigrationFile = $this->createMigration(false);
        $triggersMigration = file_get_contents($triggersMigrationFile);

        $triggerCode = '';
        $triggersSql = file_get_contents($this->triggersFile);
        $triggersSql = substr($triggersSql, 14); // Remove "DELIMITER $$"
        $triggersSql = substr($triggersSql, 0, -4); // Remove final "$$"
        $triggersSql = explode("$$", $triggersSql); // Seperate each statement into an array
        foreach($triggersSql as $trigger) {
            $triggerCode .= '$this->addSql("' . addcslashes($trigger, '"') . '");' . PHP_EOL . '        ';
        }

        $triggersMigration = str_replace("// template", $triggerCode, $triggersMigration);
        file_put_contents($triggersMigrationFile, $triggersMigration);
    }

    public function generateScTablesMigration() {
        $scTablesMigrationFile = $this->createMigration(false);
        $scTablesMigration = file_get_contents($scTablesMigrationFile);

        $scTablesSql = file_get_contents($this->scTablesFile);
        $scTablesCode = '$this->addSql("' . $scTablesSql . '");';
        $scTablesMigration = str_replace("// template", $scTablesCode, $scTablesMigration);
        file_put_contents($scTablesMigrationFile, $scTablesMigration);
    }

    public function generateVersionMigration() {
        $versionMigrationFile = $this->createMigration();
        $versionMigration = file_get_contents($versionMigrationFile);

        $versionSql = file_get_contents($this->versionFile);
        $versionCode = '$this->addSql("' . addcslashes($versionSql, '"') . '");';

        $versionMigration = str_replace("// template", $versionCode, $versionMigration);
        file_put_contents($versionMigrationFile, $versionMigration);
    }

    /**
     * Gets the code version from the version file
     *
     * @return void
     */
    public function getCodeVersion() {
        $versionDirectory = 'module/Resrequest/Application/src/Resrequest/legacy/';
        if (is_dir($versionDirectory)) {
            if ($dh = opendir($versionDirectory)) {
                while (($file = readdir($dh)) !== false) {
                    if (strpos($file,".ver")) {
                        $ver = $file;
                        break;
                    }
                }
                closedir($dh);
            }
        }
        $codeVersion = substr($ver, 0, -4);

        if (empty($codeVersion)) {
            $codeVersion = 'HEAD';
        }
        
        return $codeVersion;
    }

    public function masterOverride(bool $enableOverride) {
        if($enableOverride === true) {
            $this->eConn->exec("CALL sp_master_override(TRUE)");
        } else {
            $this->eConn->exec("CALL sp_master_override(FALSE)");
        }
    }
}
