<?php

namespace Resrequest\Setup\Service;

use Zend\Db\Adapter\Adapter;
use Zend\Console\Request as ConsoleRequest;
use Zend\Console\Console;
use Zend\Session\SessionManager;
use Zend\Session\Container;
use Resrequest\Authentication\Service\Authenticate;

class Enterprise
{
    public $principalId;
    public $archive = false; // True if the system is archived
    public $loggedIn = false;
    public $config;
    public $userId; // ID of the logged in user
    public $userName; // Username of logged in user
    public $accessGroupId; // ID of the logged in user's access group
    public $userStatusId; // 3 = System (special), 2 = Internal, 1 = External/Agent, 0 = Basic/Public
    public $passwords = array();
    public $systemUser;

    private $em;
    private $sm;
    private $request;
    private $session;
    private $identity;
    private $connection;
    private $aclService;
    private $authoriseService;
    private $authenticateService;

    public function __construct($sm)
    {
        if (Console::isConsole()) {
            return;
        }
        
        $this->sm = $sm;
        $this->em = $this->sm->get("EnterpriseEntityManager");
        $this->connection = $this->em->getConnection();
        $this->authoriseService = $this->sm->get('Resrequest\Authorisation\Service\AuthoriseService');
        $this->request = $this->sm->get('Request');
        $globalConfig = $this->sm->get('config');
        $this->config = $globalConfig['enterprise'];

        $this->archive = $this->config['archive'];

        $this->passwords = array(
            "notifications" => md5($this->config['shortName'] . "ydeivcmny9g6" . gmdate("Y-m-d")),
            "systeminfo" => md5($this->config['shortName'] . "xtzj57bhcmhp65pc" . date("Y-m-d")),
            "systemdiagnostics" => md5($this->config['shortName'] . "b3wokrqc5gbar9f9" . date("Y-m-d")),
            'resvegaSetAccess' => md5($this->config['shortName'] . "rrdmzzgh9x0s5qiw" . date("Y-m-d")),
            'systemscripts' => md5($this->config['shortName'] . "qz3kx1exfowp56y2" . gmdate("Y-m-d")),
            'systemcommand' => md5($this->config['shortName'] . "hv8makjkimtw4nte" . gmdate("Y-m-d"))
        );

        // Start the session
        $this->startSession();

        if (isset($_SESSION['userStatusId'])) {
            $this->userStatusId = $_SESSION['userStatusId'];
        } else {
            $this->userStatusId = 0;
        }

        if (!empty($_SESSION['loggedIn']) && $_SESSION['loggedIn'] === true) {
            $this->userId = $_SESSION['userid'];
            $this->userName = $_SESSION['userName'];
            $this->accessGroupId = $this->authoriseService->getUserAccessGroup($this->userId);
            $this->loggedIn = true;
        }

        $this->systemUser = array(
          'userid' => 'RS1',
          'userStatusId' => 3
        );

        $this->setupAuthentication();

        // Get the ACL service after the session has started
        $this->aclService = $this->sm->get('AclService');

        $this->updateTokens();

        $this->setupAcl();
    }

    /**
     * Called on login from the login form
     *
     * @param string $userName
     * @return void
     */
    public function onLogin() {
        if (!empty($_SESSION['loggedIn']) && $_SESSION['loggedIn'] === true) {
            $this->userId = $_SESSION['userid'];
            $this->accessGroupId = $this->authoriseService->getUserAccessGroup($this->userId);
            $this->aclService->addEnterpriseAccessGroup($this->accessGroupId);
            $this->loggedIn = true;
        }
    }

    public function setUser($username) {
        $userId = $this->authoriseService->getUserByName($username);

        if (!empty($userId)) {
            $this->userName = $username;
            $this->userId = $userId;
            $this->accessGroupId = $this->authoriseService->getUserAccessGroup($this->userId);
            $this->loggedIn = true;
        }
    }

    /**
     * Adds the necessary access groups to the ACL
     *
     * @return void
     */
    public function setupAcl() {
        // Setup user if already logged in
        if (!empty($this->accessGroupId)) {
            $this->aclService->addEnterpriseAccessGroup($this->accessGroupId);
        }

        if ($this->archive === true) {
            // Add special archive user group to ACL
            $this->aclService->addEnterpriseAccessGroup($this->aclService::ARCHIVE_ACCESS_GROUP);
        }
    }

    public function setupAuthentication()
    {
        $this->authenticateService = $this->sm->get('Resrequest\Authentication\Service\Authenticate');
    }

    public function updateTokens() {
        $router = $this->sm->get('Router');
        $routerMatch = $router->match($this->request);
        if (!empty($routerMatch)) {
            $matchedRouteName = $routerMatch->getMatchedRouteName();
            // Only automatically update tokens for specified routes
            if (in_array($matchedRouteName, $this->config['token_refresh_routes'])) {
                if ($this->loggedIn) {
                    if (!empty($_COOKIE['rrq_token'])) {
                        $this->authenticateService->refreshTokenByCookie($_COOKIE['rrq_token']);
                    }
                } else {
                    $this->authenticateService->setToken(false);
                }
            }
        }
    }

    public function startSession()
    {
        $this->session = $this->sm->get('Zend\Session\SessionManager');
        $this->session->start();

        $container = new Container('Enterprise');
        if (!isset($container->init)) {
            $this->session->regenerateId(true);
            $container->init = 1;
            $container->remoteAddr = $this->request->getServer()->get('REMOTE_ADDR');
            $container->httpUserAgent = $this->request->getServer()->get('HTTP_USER_AGENT');

            $config = $this->sm->get('Config')['enterprise'];
            if (!isset($config['session'])) {
                return;
            }

            $sessionConfig = $config['session'];
            if (isset($sessionConfig['validators'])) {
                $chain = $this->session->getValidatorChain();

                foreach ($sessionConfig['validators'] as $validator) {
                    switch ($validator) {
                        case 'Zend\Session\Validator\HttpUserAgent':
                            $validator = new $validator($container->httpUserAgent);
                            break;
                        case 'Zend\Session\Validator\RemoteAddr':
                            $validator = new $validator($container->remoteAddr);
                            break;
                        default:
                            $validator = new $validator();
                    }

                    $chain->attach('session.validate', array($validator, 'isValid'));
                }
            }
        }
    }

    /**
     * Returns the public users user access group
     *
     * @return int
     */
    public function publicUserAccessGroup() {
        $publicAccessGroupQuery = $this->em->createQueryBuilder();
        $publicAccessGroup = $publicAccessGroupQuery
            ->select(
                [
                    'accessGroup.scGroupId',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\ScGroup', 'accessGroup')
            ->where('accessGroup.scGrpInactiveYn = 0')
            ->andWhere('accessGroup.scGrpSysCode = 5')
            ->getQuery()
            ->getSingleScalarResult();

        return $publicAccessGroup;
    }

    public static function setup() {
        // Legacy code copied from inc.setup, used to determine the environment and DB
        // This needs to be done before the OAuth2 adapter starts as the OAuth tables are in the DB
        $modulesConfig = include(__DIR__ . '/../../../../../Application/config/module.config.php');
        $legacyPath = $modulesConfig['legacy_folder'];
        $legacyPath = __DIR__ . '/../../../../../../..' . $legacyPath;

        require_once($legacyPath . '/inc.config.php');
        require_once($legacyPath . '/class.mysqldb.php');

        $tld = "";
        $dbCodeOverride = false;
        $serverConfig = [
            'prefixes' => ['www'], // Prefixes will be removed
            'suffixes' => ['com', 'co.za', 'net'],	// Suffixes are retained but must be
                                                    // ignored when detecting URL format
            'local' =>['live', 'upgrade', 'demo', 'test', 'training']
                // Local client names are an alias for the first entry in
                // censys.cn_principal but use a database with the local name instead
                // of live. Eg. training.resrequest would use cn_training_xxxx
        ];

        // Get the full hostname
        $domain = $_SERVER['SERVER_NAME'] ?? "";

        // Remove common prefixes and suffixes
        $serverName = $domain;
        foreach($serverConfig['prefixes'] as $prefix) {
            $serverName = preg_replace("/^$prefix\\./","",$serverName);
        }

        $cookieDomain = $serverName;

        foreach($serverConfig['suffixes'] as $suffix) {
            if (preg_match("/\\.$suffix$/",$serverName)) {
                $serverName = preg_replace("/\\.$suffix/","",$serverName);
                $tld = $suffix;
                break;
            }
        }

        // Split into components
        $serverName = explode(".",$serverName);
        // Store and remove any archive component
        $archive = false;
        $archiveId = false;
        if (preg_match("/archive-\d{2}/",$serverName[0])) {
            $archiveId = explode("-", array_shift($serverName));
            $archiveId = ltrim($archiveId[1],"0");
        }

        // Remove base domain and store along with suffix
        $dotcom = ($tld == "") ? "" : ".";
        $server = array_pop($serverName) .$dotcom. $tld;

        // Connect to the censys database
        global $dbHost, $db, $user, $pass;
        $cDB = new \MySQLDB($db,$user,$pass,$dbHost);

        # Check to see if we are local or remote
        # And get server, client (and system - live/demo/upgrade) if local
        switch(sizeof($serverName)) {
        case 1: // Client name only
            if ($tld == "" && in_array($serverName[0], $serverConfig['local'])) {
                # This is a client's local system
                $client = "local";
                $system = $serverName[0];
            } else {
                $client = $serverName[0];
                $system = "live";
            }
            break;

        case 2: // Client name and either environment or country code: pa.mala or mala.za
            if ($cDB->get("SELECT COUNT(*) FROM cn_principal WHERE cn_prn_name_short = '".$cDB->escape($serverName[1])."'",4) > 0) {
                $dbCodeOverride = $serverName[0];
                $client = $serverName[1];
                $system = "live";
            } elseif ($cDB->get("SELECT COUNT(*) FROM cn_principal WHERE cn_prn_name_short = '".$cDB->escape($serverName[0])."'",4) > 0) {
                $client = $serverName[0];
                $country = $serverName[1];
                $system = "live";
            }
            break;

        case 3: // Environment code, client name and country code: pa.mala.za
            $dbCodeOverride = $serverName[0];
            $client = $serverName[1];
            $country = $serverName[2];
            $system = "live";
            break;
        }

        if (!empty($dbCodeOverride)) { // Remove environment from cookie domain
            $cookieDomain = preg_replace("/^$dbCodeOverride\\.$client\\./", $client . ".", $cookieDomain);
        }
        if (!empty($country)) { // Remove country from cookie domain
            $cookieDomain = preg_replace("/^$client\\.$country\\./", $client . ".", $cookieDomain);
        }

        $cookieDomain = '.' . $cookieDomain;

        $rf_sys_db_me_id = $cDB->get("select cn_prn_db_me_id from cn_principal where cn_prn_name_short = '".$cDB->escape($client)."'",4);
        if ($rf_sys_db_me_id == "" || $rf_sys_db_me_id == "0" || $rf_sys_db_me_id == 0)
        {
            $rf_sys_db_me_id = $cDB->get("select cn_sys_db_me_id from cn_system",4);
        }

        # If we are local then there is only one entry in the Censys database
        # table cn_principal
        $sql = "SELECT cn_principal_id, cn_prn_name_short, cn_prn_custom_yn, cn_prn_data_synch_mthd FROM cn_principal";
        if ($client != "local") {
            $sql .= " WHERE cn_prn_name_short = '".$cDB->escape($client)."'";
        }
        $temp = $cDB->get($sql,1);

        # Get this client's database name
        $principalId = (string)$temp['cn_principal_id'];
        $principalId = str_pad($principalId,4,"0",STR_PAD_LEFT);
        $db2 = "cn_".$system."_".$principalId;
        if ($archiveId !== false) {
            $archive = true;
            $db2 = "cn_archive_".$principalId."_".str_pad($archiveId,2,"0",STR_PAD_LEFT);
        }

        $lDB = new \MySQLDB($db2,$user,$pass,$dbHost);

        // Apply dbcode override
        $primaryEnvironmentIsWeb = $lDB->get("
            SELECT rf_db_code FROM rf_database WHERE rf_database_id = '" . $lDB->escape($rf_sys_db_me_id) . "'
        ",4)[0] == "W";
        if ($dbCodeOverride !== false) {
            if (!$primaryEnvironmentIsWeb) {
                die("Environment may not be overridden on an offline server. Contact your system administrator for more information.");
            }
            $override_db_me_id = $lDB->get("
                SELECT
                    rf_database_id
                FROM
                    rf_database
                WHERE
                    rf_db_code = '".$lDB->escape(strtoupper($dbCodeOverride)) . "'
                    AND rf_db_env_type_web_yn = '1'
            ",4);
            if (!empty($override_db_me_id)) {
                $rf_sys_db_me_id = $override_db_me_id;
                $envWebOverride = true;
            } else {
                die("This environment is not marked for online use and may not be used online. Contact your system administrator for more information.");
            }
        }

        $environment = $lDB->get("SELECT rf_db_code FROM rf_database WHERE rf_database_id = '$rf_sys_db_me_id'",4);

        $GLOBALS['isPropServer'] = false;

        $rf_db_env_type_ind = $lDB->get("select rf_db_env_type_ind from rf_database where rf_db_code = '".$environment."'",4);
        $GLOBALS['rf_db_env_type_ind'] = $rf_db_env_type_ind;
        if ($rf_db_env_type_ind == "4") $GLOBALS['isPropServer'] = true;

        $databaseName = $db2;
        $GLOBALS['dbName'] = $databaseName;
        $GLOBALS['environment'] = $environment;
        $GLOBALS['principal_id'] = $principalId;
        $GLOBALS['environment_id'] = $rf_sys_db_me_id;

        $codeVersion = "";
        if (is_dir($legacyPath)) {
            if ($dh = opendir($legacyPath)) {
                while (($file = readdir($dh)) !== false) {
                    if(strpos($file,".ver")) {
                        $ver = $file;
                        break;
                    }
                }
                closedir($dh);
            }
            $codeVersion = substr($ver,0,strrpos($ver,"."));
        }

        $https_enabled = (isset($_SERVER['HTTPS']) && !empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] != 'off');
        if (Console::isConsole()) {
          $https_enabled = false;
        }
        $http = ($https_enabled ? "https://" : "http://");
        $images_dir = ($client == "local" ? $server."/images" : $domain."/client_images");
        $images_dir_on_disk = __DIR__ . "/../../../../../../../images";
        $online_images_dir = $domain.'/client_images';
        $config = array(
            'enterprise' => array(
                'databaseName' => $databaseName,
                'principalId' => $principalId,
                'environment' => $environment,
                'environment_id' => $rf_sys_db_me_id,
                'shortName' => $client,
                'codeVersion' => $codeVersion,
                'archive' => $archive,
                'images' => array(
                  'online_images_dir' => $online_images_dir,
                  'images_dir_on_disk' => $images_dir_on_disk,
                  'images_dir' => $images_dir
                ),
                'http' => $http, 
                'session' => array(
                    'config' => array(
                        'options' => array(
                            'cookie_domain' => $cookieDomain,
                        ),
                    ),
                ),
            ),
        );

        return $config;
    }

    /**
     * Fetches the Azure AD key for the Tourism.Today Power BI reports.
     * 
     * The key file is only present on web servers and is removed on
     * deployment for offlines.
     *
     * @return string|false Key string. Returns false if the key is not available
     */
    public function getTourismTodayKey() {
        $keyFile = __DIR__ . '/TourismToday.key';
        if (file_exists($keyFile)) {
            return file_get_contents($keyFile);
        } else {
            return false;
        }
    }
}
