import {
  Component,
  OnInit,
  Input,
  ViewChildren,
  QueryList,
  ViewChild,
  ElementRef,
  OnDestroy,
  TemplateRef,
  AfterViewChecked,
  AfterViewInit
} from '@angular/core';

import { GridsterConfig, GridsterItem } from 'angular-gridster2';

import { ChartComponent } from '../chart/chart.component';
import { ChartResponse } from '../core/chart-response.model';
import { ChartContainerService } from '../services/chart-container.service';
import { BehaviorSubject, Observable, Subscription, interval } from 'rxjs';
import { filter } from 'rxjs/operators';
import { ChartContainerManager } from '../core/chart-container-manager';
import { ChartContainerOptionService } from '../services/chart-container-option.service';
import { OptionService } from '../services/option.service';

@Component({
  selector: 'app-chart-container',
  templateUrl: './chart-container.component.html',
  styleUrls: ['./chart-container.component.scss'],
  providers: [
    OptionService,
    ChartContainerOptionService
  ]
})
export class ChartContainerComponent implements OnInit, AfterViewInit, OnDestroy {
  @Input() id = '';
  @Input() active = true;
  @ViewChildren('chart') chartComponents: QueryList<ChartComponent>;
  @ViewChild('grid', { read: ElementRef }) grid: ElementRef<any>;
  @ViewChild('portal') portal: TemplateRef<any>;

  name: string;
  description: string;
  charts: {[key: string]: ChartResponse};
  config: GridsterConfig;
  dashboard: Array<GridsterItem>;
  loading = true;
  allowed = true;
  chartContainerManager: ChartContainerManager;
  hasConfigurableOptions$: Observable<boolean>;

  /**
   * Calculations of height to provide gridster with the height of
   * its content so that the page scrolls rather than the gridster
   * view.
   */
  private displayHeightSource: BehaviorSubject<number|null>
  displayHeight$: Observable<number>;
  displayHeight: number;
  private subscription: Subscription;

  constructor(
    private chartContainerService: ChartContainerService,
    private optionsService: ChartContainerOptionService
  ) {
    this.chartContainerManager = new ChartContainerManager();
    this.displayHeightSource = new BehaviorSubject<number|null>(null);
    this.displayHeight$ = this.displayHeightSource.asObservable()
      .pipe(filter(h => h !== null));
    this.displayHeight = -1;
    this.hasConfigurableOptions$ = this.chartContainerManager.hasConfigurableOptions$;
  }

  ngOnInit() {
    this.chartContainerService.getChartContainerById(this.id).subscribe(chartContainer => {
      this.name = chartContainer.name;
      this.description = chartContainer.description;

      if (chartContainer.config !== false) {
        this.config = chartContainer.config.dashboard.configuration;
        this.charts = chartContainer.charts;
        this.dashboard = chartContainer.config.dashboard.items;
        this.chartContainerManager.setChartContainer(chartContainer);
      } else {
        this.allowed = false;
      }

      this.loading = false;
    });

    this.subscription = interval(1000).subscribe(() => {
      if (this.grid && this.active) {
        const scrollHeight = this.grid.nativeElement.scrollHeight;

        if (scrollHeight !== this.displayHeight) {
          this.displayHeight = scrollHeight;
          this.displayHeightSource.next(this.displayHeight);
        }
      }
    });
  }

  ngAfterViewInit() {
    this.chartComponents.changes.subscribe((chartComponents: QueryList<ChartComponent>) => {
      const chartManagers = [];
      chartComponents.forEach(chartComponent => {
        chartManagers.push(chartComponent.getChartManager());
      });
      this.chartContainerManager.setChartManagers(chartManagers);
    });
  }

  ngOnDestroy() {
    if (this.subscription && !this.subscription.closed) {
      this.subscription.unsubscribe();
    }
  }

  public resize() {
    if (this.config) {
      this.config.api.resize();
    }
  }

  showDashboardOptions() {
    const sidebar = {
      id: 'dashboard-options',
      template: this.portal
    };
    this.optionsService.onShowOptions(this.chartContainerManager, sidebar);
  }
}
