import { Component, OnInit, Input, OnDestroy, ViewChild, TemplateRef } from '@angular/core';
import { ChartService } from '../services/chart.service';
import { Chart } from '../core/chart.model';
import { ChartManager } from '../core/chart-manager';
import { SubSink } from 'subsink';
import { Observable, BehaviorSubject } from 'rxjs';
import { ChartSummary } from '../core/chart-summary';
import { OptionService } from '../services/option.service';
import { filter } from 'rxjs/operators';
import { trigger, transition, style, animate } from '@angular/animations';

@Component({
  selector: 'app-chart',
  templateUrl: './chart.component.html',
  styleUrls: ['./chart.component.scss'],
  animations: [
    trigger(
      'fadeOutAnimation',
      [
        transition(
          ':leave',
          [
            style({ opacity: 1 }),
            animate('500ms ease-in',
              style({ opacity: 0 }))
          ]
        )
      ]
    )
  ]
})
export class ChartComponent implements OnInit, OnDestroy {
  @Input() id: string;
  @Input() chart: Chart;
  @Input() data: any;
  @ViewChild('portal') portal: TemplateRef<any>;

  width = '100%';
  height = '100%';

  dataFormat = 'json';
  chartObj: any;

  chartManager: ChartManager;
  private subsink: SubSink;
  chartSummary$: Observable<ChartSummary>;
  hasConfigurableOptions$: Observable<boolean>;

  /**
   * The source for loading$.
   */
  private loadingSource: BehaviorSubject<boolean>;

  /**
   * Whether the chart is loading or not.
   */
  loading$: Observable<boolean>;

  constructor(
    private chartService: ChartService,
    private optionsService: OptionService
  ) {
    this.loadingSource = new BehaviorSubject<boolean>(false);
    this.loading$ = this.loadingSource.asObservable();

    this.chartManager = new ChartManager(this.chartService);
    this.chartSummary$ = this.chartManager.summary$.pipe(
      filter(summary => summary !== null)
    );

    this.hasConfigurableOptions$ = this.chartManager.hasConfigurableOptions$;
    this.subsink = new SubSink();
  }

  ngOnInit() {
    this.subsink.sink = this.chartManager.updated$.subscribe(data => {
      if (data !== null) {
        this.data = data;
        this.loadChart();
      }
    });

    if (this.chart) {
      if (this.data === null) {
        this.getChart();
      } else {
        this.chartManager.setChart(this.chart);
      }
    } else {
      this.getChart();
    }
  }

  ngOnDestroy() {
    this.subsink.unsubscribe();
  }

  loadChart() {
    if (this.chartObj && this.chartObj.disposed !== true) {
      this.chartObj.setJSONData(this.data);
    }
  }

  onChartInitialised(event) {
    this.chartObj = event.chart;
    this.chartObj.setChartAttribute('theme', 'fusion');
  }

  showOptions(): void {
    const sidebar = {
      id: 'chart-options',
      template: this.portal
    };
    this.optionsService.onShowOptions(this.chartManager, sidebar);
  }

  getChart(): void {
    this.loadingSource.next(true);
    this.chartService.getChartById(this.id).subscribe(response => {
      this.chart = response.chart;
      this.data = response.data;
      this.chartManager.setChart(this.chart);
      this.loadingSource.next(false);
    });
  }

  getChartManager(): ChartManager {
    return this.chartManager;
  }

  refresh(): void {
    this.chartManager.refresh();
  }
}
