import { Chart } from './chart.model';
import { Option } from './option.model';
import { ChartService } from '../services/chart.service';
import { Subject, Observable, BehaviorSubject } from 'rxjs';
import { ChartSummary } from './chart-summary';
import { ChartContainer } from './chart-container.model';
import { ChartManager } from './chart-manager';

export class ChartContainerManager {
  private options: { [key: string]: Option };

  chartContainer: ChartContainer;

  /**
   * The source for chartManagers$.
   */
  private chartManagersSource: Subject<ChartManager[]>;

  /**
   *  Chart managers that belong to the chart container.
   */
  chartManagers$: Observable<ChartManager[]>;

  /**
   * The source for updated$.
   */
  private updatedSource: Subject<any>;

  /**
   * The new data when a chart has been updated.
   */
  updated$: Observable<any>;

  /**
   * The source for hasConfigurableOptions$.
   */
  private hasConfigurableOptionsSource: BehaviorSubject<boolean>;

  /**
   * Whether the chart has configurable options.
   */
  hasConfigurableOptions$: Observable<boolean>;

  /**
   * The source for refreshing$.
   */
  private refreshingSource: BehaviorSubject<boolean>;

  /**
   * Whether the chart is refreshing or not.
   */
  refreshing$: Observable<boolean>;

  constructor() {
    this.updatedSource = new Subject<any>();
    this.updated$ = this.updatedSource.asObservable();
    this.hasConfigurableOptionsSource = new BehaviorSubject<boolean>(false);
    this.hasConfigurableOptions$ = this.hasConfigurableOptionsSource.asObservable();
    this.refreshingSource = new BehaviorSubject<boolean>(false);
    this.refreshing$ = this.refreshingSource.asObservable();
    this.chartManagersSource = new BehaviorSubject<ChartManager[]>([]);
    this.chartManagers$ = this.chartManagersSource.asObservable();
  }

  public setChartContainer(chartContainer: ChartContainer): void {
    if (chartContainer.config !== false) {
      this.chartContainer = chartContainer;
    }
    this.buildOptions();
    this.hasConfigurableOptionsSource.next(this.hasConfigurableOptions());
  }

  public setChartManagers(managers: ChartManager[]) {
    this.chartManagersSource.next(managers);
  }

  private buildOptions(): void {
    this.options = {};
    if (this.chartContainer.config === false) {
      return;
    }

    this.chartContainer.config.options.forEach(option => {
      const id = option.config.name;

      if (id) {
        this.options[id] = option.config;
      }
    });

    this.chartContainer.config.converters.forEach(converter => {
      const id = converter.id;

      if (id) {
        this.options[id] = converter;
      }
    });
  }

  update(options) {
    this.refresh(options);
  }

  getOption(id: string): Option {
    if (id in this.options) {
      return this.options[id];
    } else {
      return null;
    }
  }

  getOptions(): Option[] {
    return Object.keys(this.options).map(key => this.options[key]);
  }

  /**
   * Uses the current chart configuration to update the chart.
   */
  refresh(optionValues = []): void {
    this.refreshingSource.next(true);
  }

  /**
   * Checks whether there is a configured UI for the options.
   */
  private hasConfigurableOptions() {
    if (this.chartContainer.config && this.chartContainer.config.ui) {
      return this.chartContainer.config.ui.menu.length > 0;
    } else {
      return false;
    }
  }
}
