import { Chart } from './chart.model';
import { Option } from './option.model';
import { ChartService } from '../services/chart.service';
import { Subject, Observable, BehaviorSubject } from 'rxjs';
import { ChartSummary } from './chart-summary';

export class ChartManager {
  private options: { [key: string]: Option };

  chart: Chart;

  /**
   * The source for updated$.
   */
  private updatedSource: Subject<any>;

  /**
   * The new data when a chart has been updated.
   */
  updated$: Observable<any>;

  /**
   * The source for summary$.
   */
  private summarySource: BehaviorSubject<ChartSummary>;

  /**
   * The summary of the chart currently being managed.
   */
  summary$: Observable<ChartSummary>;

  /**
   * The source for hasConfigurableOptions$.
   */
  private hasConfigurableOptionsSource: BehaviorSubject<boolean>;

  /**
   * Whether the chart has configurable options.
   */
  hasConfigurableOptions$: Observable<boolean>;

  /**
   * The source for refreshing$.
   */
  private refreshingSource: BehaviorSubject<boolean>;

  /**
   * Whether the chart is refreshing or not.
   */
  refreshing$: Observable<boolean>;

  /**
   * The source for options$.
   */
  private optionsSource: BehaviorSubject<Option[]>;

  /**
   * The chart's options.
   */
  options$: Observable<Option[]>;

  constructor(private chartService: ChartService) {
    this.updatedSource = new Subject<any>();
    this.updated$ = this.updatedSource.asObservable();
    this.summarySource = new BehaviorSubject<ChartSummary>(null);
    this.summary$ = this.summarySource.asObservable();
    this.hasConfigurableOptionsSource = new BehaviorSubject<boolean>(false);
    this.hasConfigurableOptions$ = this.hasConfigurableOptionsSource.asObservable();
    this.refreshingSource = new BehaviorSubject<boolean>(false);
    this.refreshing$ = this.refreshingSource.asObservable();
    this.optionsSource = new BehaviorSubject<Option[]>(null);
    this.options$ = this.optionsSource.asObservable();
  }

  public setChart(chart: Chart): void {
    this.chart = chart;
    this.summarySource.next({
      id: chart.id,
      name: chart.name,
      description: chart.description,
      reportInfo: chart.reportInfo,
      library: chart.library,
      type: chart.type,
      chartClass: chart.chartClass
    });
    this.buildOptions();
    this.hasConfigurableOptionsSource.next(this.hasConfigurableOptions());
  }

  private buildOptions(): void {
    this.options = {};
    this.chart.config.datasets.forEach(dataset => {
      dataset.config.options.forEach(option => {
        const id = option.config.id;

        if (id) {
          this.options[id] = option.config;
        }
      });

      dataset.config.converters.forEach(converter => {
        const id = converter.id;

        if (id) {
          this.options[id] = converter;
        }
      });
    });
    this.optionsSource.next(this.getOptions());
  }

  update(options) {
    this.refresh(options);
  }

  getOption(id: string): Option {
    if (id in this.options) {
      return this.options[id];
    } else {
      return null;
    }
  }

  getOptions(): Option[] {
    return Object.keys(this.options).map(key => this.options[key]);
  }

  /**
   * Uses the current chart configuration to update the chart.
   */
  refresh(optionValues = []): void {
    this.refreshingSource.next(true);

    this.chartService
      .buildChart(this.chart, optionValues)
      .subscribe(response => {
        this.refreshingSource.next(false);

        if (response) {
          this.setChart(response.chart);
          this.updatedSource.next(response.data);
        } else {
          this.updatedSource.next(null);
        }
      });
  }

  /**
   * Checks whether there is a configured UI for the options.
   */
  private hasConfigurableOptions() {
    if (this.chart) {
      return this.chart.config.ui.menu.length > 0;
    } else {
      return false;
    }
  }
}
