import { Component, OnInit, Input, Output, EventEmitter } from '@angular/core';
import { InputChanged } from '../../core/input-changed.model';

interface CurrencyValue {
  currencyTo: string;
  rates: RateConfig;
}

interface CurrencyConfig {
  supportedCurrencies: string[];
}

interface RateConfig {
  [currencyTo: string]: {
    [currencyFrom: string]: Rate
  };
}

interface Rate {
  exchangeRate: number;
}

@Component({
  selector: 'app-currency-converter',
  templateUrl: './currency-converter.component.html',
  styleUrls: ['./currency-converter.component.scss']
})
export class CurrencyConverterComponent implements OnInit {
  @Input() config: CurrencyConfig;
  @Input() value: CurrencyValue;

  @Output() valueChanged: EventEmitter<InputChanged>;

  currentValue: CurrencyValue;
  initialValue: CurrencyValue;
  exchangeRates: { [currencyFrom: string]: number };

  constructor() {
    this.valueChanged = new EventEmitter();
    this.exchangeRates = {};
  }

  ngOnInit() {
    if (typeof this.value.rates !== 'object') {
      this.value.rates = {};
    }
    this.initialValue = JSON.parse(JSON.stringify(this.value));
    this.currentValue = JSON.parse(JSON.stringify(this.value));

    setTimeout(() => {
      this.populateExchangeRate();
    });
  }

  onValueChanged() {
    let valueChanged = true;

    const value = {
      currencyTo: this.currentValue.currencyTo,
      rates: this.currentValue.rates
    };

    if (JSON.stringify(this.initialValue) === JSON.stringify(value)) {
      valueChanged = false;
    }

    const inputChanged: InputChanged = {
      value,
      valueChanged
    };

    this.valueChanged.emit(inputChanged);
  }

  populateExchangeRate() {
    this.exchangeRates = {};

    this.config.supportedCurrencies.forEach(currency => {
      this.exchangeRates[currency] = this.getExchangeRate(currency);
    });

    // this.updateExchangeRates();
  }

  updateExchangeRates() {
    const rates = this.currentValue.rates[this.currentValue.currencyTo] = {};

    Object.keys(this.exchangeRates).forEach(currency => {
      const exchangeRate = this.exchangeRates[currency];
      if (exchangeRate > 0) {
        rates[currency] = {
          exchangeRate: this.exchangeRates[currency]
        };
      } else {
        delete rates[currency];
      }
    });

    if (Object.keys(rates).length === 0) {
      delete this.currentValue.rates[this.currentValue.currencyTo];
    }
  }

  getExchangeRate(currencyFrom) {
    const currencyTo = this.currentValue.currencyTo;
    const rates = this.currentValue.rates[currencyTo];

    if (rates) {
      const rate = rates[currencyFrom];

      if (rate) {
        return rate.exchangeRate;
      }
    }

    // Exchange rate does not exist.
    return 0;
  }

  onCurrencyToChanged(): void {
    this.populateExchangeRate();
    this.onValueChanged();
  }

  onRateChange(): void {
    this.updateExchangeRates();
    this.onValueChanged();
  }
}
