import { Component, OnInit, Input, Output, EventEmitter, ViewChild, ElementRef } from '@angular/core';
import { BehaviorSubject, Observable, combineLatest } from 'rxjs';
import { map, first } from 'rxjs/operators';
import { InputChanged } from '../../core/input-changed.model';
import { DataSource } from '@angular/cdk/table';

interface MultiSelectConfig {
  options: Option[];
  enableSearch: boolean;
  searchLabel: string;
  headerLabel: string;
}

interface Option {
  name: string;
  value: string;
  inactive: boolean;
}

interface OptionState {
  index: number;
  label: string;
  value: string;
  inactive: boolean;
  selected: boolean;
  initial: boolean;
}

class SelectSource extends DataSource<OptionState> {
  constructor(private visibleOptions$: Observable<OptionState[]>) {
    super();
  }

  connect() {
    return this.visibleOptions$;
  }

  disconnect() {

  }
}

@Component({
  selector: 'app-multi-select',
  templateUrl: './multi-select.component.html',
  styleUrls: ['./multi-select.component.scss']
})
export class MultiSelectComponent implements OnInit {
  @Input() config: MultiSelectConfig;
  @Input() value: string[];
  @Input() showInactive: boolean;
  private selectionState$: BehaviorSubject<OptionState[]>;
  masterSelected$: Observable<boolean>;
  masterIndeterminate$: Observable<boolean>;
  searchTerm$: BehaviorSubject<string>;
  changedOptions: number[];
  visibleOptions$: Observable<OptionState[]>;
  optionsSource: SelectSource;
  @Output() valueChanged: EventEmitter<InputChanged>;
  @ViewChild('input', { read: ElementRef }) input: ElementRef<any>;
  showSearchBorder: boolean;

  constructor() {
    this.value = [];
    this.showInactive = false;
    this.selectionState$ = new BehaviorSubject<OptionState[]>([]);
    this.searchTerm$ = new BehaviorSubject<string>('');
    this.valueChanged = new EventEmitter<InputChanged>();
    this.changedOptions = [];
    this.showSearchBorder = false;
  }

  ngOnInit() {
    this.initialiseState();
    this.masterSelected$ = this.checkMasterSelected();
    this.masterIndeterminate$ = this.checkMasterIndeterminate();
    this.visibleOptions$ = combineLatest([this.selectionState$, this.searchTerm$]).pipe(
      map(([options, searchTerm]) => {
        const term = searchTerm.toLocaleLowerCase();
        options = options.filter(option => this.showInactive || !option.inactive);
        options = options.filter(option => {
          const label = option.label.toLocaleLowerCase();

          return !term || label.includes(term);
        });

        return options;
      })
    );
    this.optionsSource = new SelectSource(this.visibleOptions$);
  }

  private initialiseState(): void {
    const states = this.config.options.map((option, index) => {
      const selected = this.value.includes(option.value);

      return {
        index,
        label: option.name,
        value: option.value,
        inactive: option.inactive,
        initial: selected,
        selected
      };
    });

    this.selectionState$.next(states);
  }

  private checkMasterSelected(): Observable<boolean> {
    return combineLatest([this.selectionState$, this.searchTerm$]).pipe(
      map(([states, searchTerm]) => {
        if (states.length === 0) {
          return false;
        }

        const search = searchTerm.toLocaleLowerCase();

        return states
          .filter(state => this.showInactive || !state.inactive)
          .filter(state => {
            const label = state.label.toLocaleLowerCase();
            return !search || label.includes(search);
          })
          .map(state => state.selected)
          .reduce((previous, current) => previous && current, true);
      })
    );
  }

  private checkMasterIndeterminate(): Observable<boolean> {
    return combineLatest([this.selectionState$, this.searchTerm$]).pipe(
      map(([states, searchTerm]) => {
        if (states.length === 0) {
          return false;
        }

        const search = searchTerm.toLocaleLowerCase();
        const filteredStates = states
          .filter(state => this.showInactive || !state.inactive)
          .filter(state => {
            const label = state.label.toLocaleLowerCase();
            return !search || label.includes(search);
          });
        const hasSelected = filteredStates.filter(state => state.selected).length > 0;
        const hasUnselected = filteredStates.filter(state => !state.selected).length > 0;

        return hasSelected && hasUnselected;
      })
    );
  }

  onMasterChange(value: boolean): void {
    combineLatest([this.selectionState$, this.searchTerm$])
      .pipe(first())
      .subscribe(([states, searchTerm]) => {
        const search = searchTerm.toLocaleLowerCase();

        states
          .filter(state => this.showInactive || !state.inactive)
          .filter(state => {
            const label = state.label.toLocaleLowerCase();
            return !search || label.includes(search);
          })
          .forEach((state, index) => {
            // Only update visible checkboxes.
            state.selected = value;

            if (state.initial !== value) {
              // Ensure is in array
              const optionInArray = this.changedOptions.find(i => i === index);

              if (typeof optionInArray === 'undefined') {
                this.changedOptions.push(index);
              }
            } else {
              // Remove from array if it exists
              const optionInArray = this.changedOptions.find(i => i === index);

              if (typeof optionInArray !== 'undefined') {
                this.changedOptions.splice(this.changedOptions.indexOf(index), 1);
              }
            }
          });

        this.selectionState$.next(states);
        this.onValueChanged();
      });
  }

  showOption(index: number): Observable<boolean> {
    return combineLatest([this.selectionState$, this.searchTerm$]).pipe(
      map(([states, searchTerm]) => {
        const state = states[index];

        if (!this.showInactive && state.inactive) {
          return false;
        } else if (!searchTerm) {
          return true;
        } else {
          const label = state.label.toLocaleLowerCase();
          const search = searchTerm.toLocaleLowerCase();
          return label.includes(search);
        }
      })
    );
  }

  isOptionSelected(index: number): Observable<boolean> {
    return this.selectionState$.pipe(
      map(states => {
        return states[index].selected;
      })
    );
  }

  onOptionChange(index: number, value: boolean): void {
    this.selectionState$.pipe(first()).subscribe(states => {
      const state = states[index];
      state.selected = value;

      if (state.initial !== value) {
        // Ensure is in array
        const optionInArray = this.changedOptions.find(i => i === index);

        if (typeof optionInArray === 'undefined') {

          this.changedOptions.push(index);
        }
      } else {
        // Remove from array if it exists
        const optionInArray = this.changedOptions.find(i => i === index);

        if (typeof optionInArray !== 'undefined') {
          this.changedOptions.splice(this.changedOptions.indexOf(index), 1);
        }
      }

      this.selectionState$.next(states);
      this.onValueChanged();
    });
  }

  onValueChanged(): void {
    this.selectionState$.pipe(first()).subscribe(states => {
      const selected = states
        .filter(state => state.selected)
        .map(state => state.value);

      let valueChanged = false;

      if (this.changedOptions.length !== 0) {
        valueChanged = true;
      }

      const inputChanged: InputChanged = {
        value: selected,
        valueChanged
      };

      this.valueChanged.emit(inputChanged);
    });
  }

  onSearchTermChanged(searchTerm: string): void {
    this.searchTerm$.next(searchTerm);
  }

  inputFocused(focused: boolean): void {
    this.showSearchBorder = focused;
  }
}
