import { Component, OnInit, Input, ElementRef, ViewChild, AfterViewInit, OnDestroy } from '@angular/core';
import { ResvegaService } from 'src/app/dashboard/tourism-today/resvega.service';
import * as pbi from 'powerbi-client';
import { combineLatest, BehaviorSubject, Observable } from 'rxjs';
import { SubSink } from 'subsink';
import { AuthorisationService } from 'src/app/shared/services/authorisation/authorisation.service';
import { InfoService } from 'src/app/shared/services/info/info.service';

@Component({
  selector: 'app-power-bi',
  templateUrl: './power-bi.component.html',
  styleUrls: ['./power-bi.component.scss']
})
export class PowerBiComponent implements OnInit, AfterViewInit, OnDestroy {
  @ViewChild('reportContainer') reportContainer: ElementRef;

  @Input() id: string;

  isPrimaryEnvironmentWeb = false;

  /**
   * The source for loading$;
   */
  private loadingSource: BehaviorSubject<boolean>;

  /**
   * Whether the component is loading
   */
  loading$: Observable<boolean>;

  /**
   * Whether the Power BI chart has loaded
   */
  private loaded = false;

  accessLoaded$ = this.authorisationService.accessLoaded$;

  client$ = this.infoService.client$;
  principalName$ = this.infoService.principalName$;

  private powerbi: pbi.service.Service;
  private subsink: SubSink;

  constructor(
    private resvega: ResvegaService,
    private authorisationService: AuthorisationService,
    private infoService: InfoService
  ) {
    this.subsink = new SubSink();
    this.loadingSource = new BehaviorSubject<boolean>(!this.isPrimaryEnvironmentWeb);
    this.loading$ = this.loadingSource.asObservable();
  }

  ngOnInit(): void {
    this.subsink.sink = this.accessLoaded$.subscribe(loaded => {
      if (loaded) {
        this.isPrimaryEnvironmentWeb = this.authorisationService.isPrimaryEnvironmentWeb();

        if (!this.isPrimaryEnvironmentWeb) {
          this.loadingSource.next(false);
        }
      }
    });
  }

  ngAfterViewInit() {
    this.subsink.sink = this.accessLoaded$.subscribe(loaded => {
      if (loaded && this.isPrimaryEnvironmentWeb) {
        this.subsink.sink = combineLatest([
          this.resvega.reports$,
          this.resvega.token$,
        ]).subscribe(([reports, token]) => {
          if (reports && token) {
            if (!this.loaded) {
              const report = reports.find(tmpReport => tmpReport.id === this.id);
              if (report) {
                this.renderReport(token, report.id, report.embedUrl);
              }
            }
          }
          if (this.loaded && token) {
            const report = this.powerbi.get(this.reportContainer.nativeElement);
            report.setAccessToken(token);
          }
        });
      }
    });
  }

  ngOnDestroy() {
    if (this.loaded) {
      this.powerbi.reset(this.reportContainer.nativeElement);
    }

    this.subsink.unsubscribe();
  }

  renderReport(accessToken: string, id: string, embedUrl: string) {
    const config: pbi.IEmbedConfiguration = {
      type: 'report',
      accessToken,
      embedUrl,
      id,
      settings: {
        navContentPaneEnabled: false
      },
      tokenType: 1,
      pageView: 'fitToWidth',
    };

    const reportContainer = this.reportContainer.nativeElement;
    const powerbi = new pbi.service.Service(pbi.factories.hpmFactory, pbi.factories.wpmpFactory, pbi.factories.routerFactory);
    const report = powerbi.embed(reportContainer, config);

    this.powerbi = powerbi;

    report.off('loaded');
    report.on('loaded', () => {
      this.loaded = true;
      this.loadingSource.next(false);
    });
    report.on('error', () => {
    });
  }
}
