import { Injectable, OnDestroy } from '@angular/core';
import { BehaviorSubject, Observable, combineLatest } from 'rxjs';
import { SubSink } from 'subsink';
import { SubOption } from '../core/sub-option.model';
import { PageSection } from '../core/ui-config/page-section.model';
import { first, map, filter } from 'rxjs/operators';
import { LinkedSubOption } from '../core/ui-config/linked-suboption.model';
import { ChartContainerOptionService } from './chart-container-option.service';
import { ChartContainerManager } from '../core/chart-container-manager';
import { ChartManager } from '../core/chart-manager';

@Injectable()
export class ChartContainerOptionsPageService implements OnDestroy {
  private optionValues: { source: string, option: string, value: any }[];

  /**
   * The source for optionsChanged$.
   */
  private optionsChangedSource: BehaviorSubject<boolean>;

  /**
   * Manages subscriptions.
   */
  private subsink: SubSink;

  /**
   * The manager for the chart container that is having its options displayed.
   */
  private manager: ChartContainerManager;

  private managers: { [key: string]: ChartManager; } = {};

  /**
   * The map of inputs where each entry is keyed by the combination
   * of the option id and the sub-option name: optionId_subOptionName
   */
  inputMap: { [key: string]: SubOption };

  /**
   * The current values of the sub options.
   */
  valueMap: { [key: string]: any };

  private chartsOptionValues: {
    [chartId: string]: { source: string, option: string, value: any }[]
  };

  /**
   * The map of sub options and other sub options that they are linked to.
   */
  linkedMap: { [key: string]: LinkedSubOption[] };

  /**
   * Whether the options have changed or not.
   */
  optionsChanged$: Observable<boolean>;

  /**
   * The source for pageToSections$.
   */
  private pageToSectionsSource: BehaviorSubject<{ [key: string]: PageSection[] }>;

  /**
   * A map of the page name to the sections it has.
   */
  pageToSections$: Observable<{ [key: string]: PageSection[] }>;

  /**
   * The source for showLoading$.
   */
  private showLoadingSource: BehaviorSubject<boolean>;

  /**
   * Whether to show the loading indicator or not.
   */
  showLoading$: Observable<boolean>;

  /**
   * The source for refreshing$.
   */
  private refreshingSource: BehaviorSubject<boolean>;

  /**
   * Whether the charts that have had their options applied are refreshing.
   */
  refreshing$: Observable<boolean>;

  constructor(private optionsService: ChartContainerOptionService) {
    this.optionsChangedSource = new BehaviorSubject<boolean>(false);
    this.optionsChanged$ = this.optionsChangedSource.asObservable();
    this.subsink = new SubSink();
    this.inputMap = {};
    this.valueMap = {};
    this.chartsOptionValues = {};
    this.optionValues = [];
    this.pageToSectionsSource = new BehaviorSubject<{ [key: string]: PageSection[] }>({});
    this.pageToSections$ = this.pageToSectionsSource.asObservable();
    this.showLoadingSource = new BehaviorSubject<boolean>(false);
    this.showLoading$ = this.showLoadingSource.asObservable();
    this.refreshingSource = new BehaviorSubject<boolean>(false);
    this.refreshing$ = this.refreshingSource.asObservable();

    this.subsink.sink = optionsService.manager$.subscribe(manager => {
      if (manager !== null) {
        this.manager = manager;
        this.reset();
        this.buildPageMap();
      }

      manager.chartManagers$.subscribe(chartManagers => {
        chartManagers.forEach(chartManager => {
          chartManager.options$.subscribe(options => {
            if (options) {
              const chartId = chartManager.chart.id;
              this.managers[chartId] = chartManager;
              this.chartsOptionValues[chartId] = [];
            }
          });
        });
      });
      this.showLoadingSource.next(false);
    });
  }

  ngOnDestroy() {
    this.subsink.unsubscribe();
  }

  /**
   * Builds the mapping of options and option names to inputs.
   */
  private buildInputMap(): void {
    this.inputMap = {};
    this.valueMap = {};

    this.manager.getOptions().forEach(option => {
      option.options.forEach(subOption => {
        const key = `${option.name}_${subOption.name}`;
        this.inputMap[key] = subOption;
        this.valueMap[key] = subOption.value;
      });
    });
  }

  /**
   * Builds the map of sub-options that are linked to other sub-options.
   */
  private buildLinkMap(): void {
    const map: { [key: string]: LinkedSubOption[] } = {};

    if (!this.manager.chartContainer.config) {
      this.linkedMap = map;
      return;
    }

    this.manager.chartContainer.config.ui.menu.forEach(uiSection => {
      uiSection.pages.forEach(page => {
        page.config.sections.forEach(section => {
          section.options.forEach(option => {
            option.config.options.forEach(subOption => {
              const key = `${option.source}_${subOption.name}`;
              map[key] = [];
              subOption.links.forEach(link => {
                map[key].push(link);
              });
            });
          });
        });
      });
    });

    this.linkedMap = map;
  }

  /**
   * Builds the mapping of pages to sections.
   */
  private buildPageMap(): void {
    const map = {};

    if (!this.manager.chartContainer.config) {
      this.linkedMap = map;
      return;
    }

    this.manager.chartContainer.config.ui.menu.forEach(uiSection => {
      uiSection.pages.forEach(page => {
        map[page.name] = page.config.sections;
      });
    });

    this.buildInputMap();
    this.buildLinkMap();
    this.pageToSectionsSource.next(map);
  }

  /**
   * Called when the value of a sub-option has changed.
   * @param optionId The id of the option.
   * @param subOptionName The name of the sub-option.
   * @param value The value of the sub-option.
   */
  onSubOptionChanged(optionId: string, subOptionName: string, value: any, valueChanged: boolean): void {
    this.updateSubOption(optionId, subOptionName, value, valueChanged);
    const key = `${optionId}_${subOptionName}`;
    // Update linked sub-options.
    this.linkedMap[key].forEach(link => {
      if (link.chartId === '') {
        this.updateSubOption(link.source, link.option, value, valueChanged);
      } else {
        this.updateLinkedChartSubOption(link.chartId, link.source, link.option, value, valueChanged);
      }
    });
    this.optionsChangedSource.next(this.hasChanges());
  }

  /**
   * Update the value of the sub-option.
   * @param optionId The id of the option.
   * @param subOptionName The name of the sub-option.
   * @param value The value of the sub-option.
   */
  private updateSubOption(optionId: string, subOptionName: string, value: any, valueChanged: boolean): void {
    if (valueChanged) {
      const index = this.optionValues.findIndex(option => {
        return option.source === optionId && option.option === subOptionName;
      });

      if (index >= 0) {
        this.optionValues[index].value = value;
      } else {
        this.optionValues.push({ source: optionId, option: subOptionName, value });
      }
    } else {
      const index = this.optionValues.findIndex(option => {
        return option.source === optionId && option.option === subOptionName;
      });

      if (index >= 0) {
        this.optionValues.splice(index, 1);
      }
    }

    this.valueMap[`${optionId}_${subOptionName}`] = value;
  }

  private updateLinkedChartSubOption(chartId: string, optionId: string, subOptionName: string, value: any, valueChanged: boolean): void {
    if (this.chartsOptionValues[chartId]) {
      if (valueChanged) {
        const index = this.chartsOptionValues[chartId].findIndex(option => {
          return option.source === optionId && option.option === subOptionName;
        });

        if (index >= 0) {
          this.chartsOptionValues[chartId][index].value = value;
        } else {
          this.chartsOptionValues[chartId].push({ source: optionId, option: subOptionName, value });
        }
      } else {
        const index = this.chartsOptionValues[chartId].findIndex(option => {
          return option.source === optionId && option.option === subOptionName;
        });

        if (index >= 0) {
          this.chartsOptionValues[chartId].splice(index, 1);
        }
      }
    }
  }

  /**
   * Whether any of the options have changed value.
   */
  private hasChanges(): boolean {
    return this.optionValues.length > 0;
  }

  /**
   * Reset the current value of the options.
   */
  reset(): void {
    this.optionsChangedSource.next(false);
    this.buildPageMap();
  }

  /**
   * Apply the current option values to the chart.
   */
  apply(): void {
    this.applyChanges();
  }

  /**
   * Attempts to fetch the new chart data.
   */
  private applyChanges(): void {
    this.refreshingSource.next(true);

    const managers = [];
    Object.entries(this.managers).forEach(([chartId, manager]) => {
      const options = this.chartsOptionValues[chartId];

      if (options.length > 0) {
        manager.update(options);
        managers.push(manager.refreshing$);
      }
    });

    if (managers.length) {
      combineLatest(managers).pipe(
        map(updates => updates.every(updating => !updating)),
        filter(refreshed => refreshed),
        first()
      ).subscribe(() => {
        this.optionValues = [];
        this.optionsChangedSource.next(this.hasChanges());
        this.refreshingSource.next(false);
      });
    } else {
      this.refreshingSource.next(false);
    }
  }

  /**
   * Apply all settings including unchanged settings.
   */
   applyAll() {
    Object.entries(this.valueMap).forEach(([key, value]) => {
      if (key in this.linkedMap) {
        const [optionId, subOptionName] = key.split('_');
        this.onSubOptionChanged(optionId, subOptionName, value, true);
      }
    });

    this.applyChanges();
  }

  /**
   * Refreshes the chart data.
   */
  refresh(): void {
    this.showLoadingSource.next(true);
    this.manager.updated$.pipe(first()).subscribe(data => {
      this.showLoadingSource.next(false);
      this.optionsService.refresh();
    });
    this.manager.refresh();
  }
}
