import { Component, ElementRef, OnInit, TemplateRef, ViewChild } from '@angular/core';
import { FormBuilder, FormGroup, Validators } from '@angular/forms';
import { MatDialog } from '@angular/material/dialog';
import * as moment from 'moment';
import { first } from 'rxjs/operators';
import { CheckInService } from '../../../services/check-in/checkin.service';
import { GuestService } from '../../../services/guest/guest.service';


@Component({
  selector: 'app-check-in-form',
  templateUrl: './check-in-form.component.html',
  styleUrls: ['./check-in-form.component.scss']
})
export class CheckInFormComponent implements OnInit {
  @ViewChild('indemnityMessageDialog') indemnityMessageDialogRef: TemplateRef<any>;
  @ViewChild('termsAndConditionsDialog') termsAndConditionsDialogRef: TemplateRef<any>;

  indemnityMessage: string;
  propertiesIndemnityMessage: {[propertyId: string]: string;};
  termsAndConditions: string;
  detailsForm: FormGroup;
  public isViewable: boolean;
  datePickerStartYear;
  guest$ = this.checkInService.guest$;
  item$ = this.checkInService.item$;
  guestDetails$ = this.checkInService.guestDetails$;
  guests$ = this.checkInService.guests$;

  constructor(
    private formBuilder: FormBuilder,
    private checkInService: CheckInService,
    private dialog: MatDialog,
    private el: ElementRef
  ) {
    this.checkInService.guest$.subscribe(guest => {
      if (guest) {
        this.checkInService.guestSettings$.subscribe(guestSettings => {
          if (guestSettings) {
            this.indemnityMessage = guestSettings.indemnityMessage;
            this.propertiesIndemnityMessage = guestSettings.propertiesIndemnityNotes;
            this.termsAndConditions = guestSettings.terms;

            const emailPattern = GuestService.emailPattern;

            const firstName: any = [guest.firstName];
            const lastName: any = [guest.lastName];
            const email: any = [guest.email];
            const mobileNumber: any = [''];
            const birthDate: any = [''];
            const passportNumber: any = [''];
            const covidYn: any = [null];
            const termsAndConditions: any = [false, [Validators.requiredTrue]];

            if (guestSettings.fields.find(field => field.name === 'guest_name_first').compulsory) {
              firstName.push([Validators.required]);
            }

            if (guestSettings.fields.find(field => field.name === 'guest_name_last').compulsory) {
              lastName.push([Validators.required]);
            }

            if (guestSettings.fields.find(field => field.name === 'guest_email').compulsory) {
              email.push([Validators.required, Validators.pattern(emailPattern)]);
            } else {
              email.push([Validators.pattern(emailPattern)]);
            }

            if (guestSettings.fields.find(field => field.name === 'guest_phone_number').compulsory) {
              mobileNumber.push([Validators.required]);
            }

            if (guestSettings.fields.find(field => field.name === 'guest_birthdate').compulsory) {
              birthDate.push([Validators.required]);
            }

            if (guestSettings.fields.find(field => field.name === 'guest_passport_number').compulsory) {
              passportNumber.push([Validators.required]);
            }

            if (guestSettings.fields.find(field => field.name === 'covid_exposure').compulsory) {
              covidYn.push([Validators.required]);
            }

            this.detailsForm = this.formBuilder.group({
              firstName,
              lastName,
              email,
              mobileNumber,
              birthDate,
              passportNumber,
              notesAppend: '',
              covidYn,
              termsAndConditions
            });
          }
        });
      } else {
        this.detailsForm.reset();
      }

      this.onChanges();
    });

    this.datePickerStartYear = moment().subtract(30, 'years');
  }

  ngOnInit(): void { }

  onSubmit(guestData) {
    this.detailsForm.markAllAsTouched();

    if (this.detailsForm.valid) {
      this.checkInService.guest$.pipe(first()).subscribe(selectedGuest => {
        this.checkInService.checkInGuests({ ...selectedGuest, ...guestData });
      });
    } else {
      const controls = Object.keys(this.detailsForm.controls).reverse();
      for (const key of controls) {
        if (this.detailsForm.controls[key].invalid) {
          const invalidControl = this.el.nativeElement.querySelector('[formcontrolname="' + key + '"]');
          invalidControl.focus();
        }
      }
    }
  }

  openIndemnityMessageDialog() {
    this.dialog.open(this.indemnityMessageDialogRef, {maxHeight: '70%', height: '50%'});
  }

  openTermsAndConditionsDialog() {
    this.dialog.open(this.termsAndConditionsDialogRef, {maxHeight: '70%', height: '50%'});
  }

  closeAllDialogs() {
    this.dialog.closeAll();
  }
  

  cancel(): void {
    this.checkInService.reset();
  }

  onChanges(): void {
    this.detailsForm.valueChanges.subscribe(value => {
    });
  }

  sameDay(d2) {
    const d1 = new Date();
    return d1.getFullYear() === d2.getFullYear() &&
      d1.getMonth() === d2.getMonth() &&
      d1.getDate() === d2.getDate();
  }

  clearGuest() {
    this.checkInService.clearGuest();
  }
}
