import { ChangeDetectorRef, Component, OnInit, ViewChildren } from '@angular/core';
import { CdkDragDrop, moveItemInArray, transferArrayItem } from '@angular/cdk/drag-drop';
import { GuestListService, SaveGuestsItems } from '../../../services/guest-list/guest-list.service';
import { FormBuilder, FormGroup, Validators } from '@angular/forms';
import { MatDialog } from '@angular/material/dialog';
import { GuestListItemComponent } from '../guest-list-item/guest-list-item.component';
import { BehaviorSubject, Observable } from 'rxjs';
import { take } from 'rxjs/operators';
import { Router } from '@angular/router';

interface Guest {
  id: false|string;
  updated: boolean; // Marks existing contact for update
  locked: boolean;
  firstName: string;
  lastName: string;
  email: string;
  initialData?: {
    firstName: string;
    lastName: string;
    email: string;
  };
}

@Component({
  selector: 'app-guest-list-form',
  templateUrl: './guest-list-form.component.html',
  styleUrls: ['./guest-list-form.component.scss']
})

export class GuestListFormComponent implements OnInit {
  @ViewChildren('guestItem') guestItems: GuestListItemComponent[];

  DATE_FORMAT = 'DD MMMM YYYY';

  guestListForm: FormGroup;
  public isViewable: boolean;

  showValidationError = false;
  reservationDetails$ = this.listService.reservationDetails$;
  item;
  items;
  item$ = this.listService.item$;
  items$ = this.listService.items$;
  showUnallocatedGuestList = false;

  /**
   * The source of itemIndex$
   */
  private itemIndexSource: BehaviorSubject<number>;

  /**
   * The index of the selected reservation item
   */
  itemIndex$: Observable<number>;

  unallocatedGuests = [];
  shownUnallocatedGuests = [];
  groups = []; // Maps guests to groups and groups to items
  guests: Guest[] = [];
 
  constructor(
    private listService: GuestListService,
    private router: Router,
    private cd: ChangeDetectorRef
  ) {
    this.itemIndexSource = new BehaviorSubject<number>(null);
    this.itemIndex$ = this.itemIndexSource.asObservable();

    this.items$.subscribe(items => {
      this.groups = [];

      if (items) {
        let index = 0;
        items.forEach(item => {
          this.groups.push([]);

          for (let i = 0; i < item.accommCount; i++) {
            this.groups[index].push([]);
            const groupIndex = this.groups[index].length -1;

            for (let j = 0; j < item.paxCount; j++) {
              this.guests.push(
                {
                  id: false,
                  updated: false,
                  locked: false,
                  firstName: '',
                  lastName: '',
                  email: ''
                }
              );

              this.groups[index][groupIndex].push(this.guests.length -1);
            }
          }

          item.guests.forEach((group, groupIndex) => {
            group.forEach((tmpGuest, groupSlot) => {
              // Assign allocated guests to their groups
              let guestIndex = this.guests.findIndex(guest => guest.id === tmpGuest.id);

              if (guestIndex < 0) {
                this.guests.push({
                  ...tmpGuest,
                  updated: false,
                  locked: true
                })
                guestIndex = this.guests.findIndex(guest => guest.id === tmpGuest.id);
              }

              this.groups[index][groupIndex][groupSlot] = guestIndex;
            });
          });

          index++;
        });
      }

      this.reservationDetails$.subscribe(details => {
        this.unallocatedGuests = [[]];
        if (details) {
          details.guests.forEach(tmpGuest => {
            let guestIndex = this.guests.findIndex(guest => guest.id === tmpGuest.id);

            if (guestIndex < 0) {
              this.guests.push({
                ...tmpGuest,
                updated: false,
                locked: true
              })
              guestIndex = this.guests.findIndex(guest => guest.id === tmpGuest.id);
            }

            this.unallocatedGuests[0].push(guestIndex);
          });
  
          items.forEach((item, itemIndex) => {
            this.groups[itemIndex].forEach((group, groupIndex) => {
              group.forEach((guestIndex, groupSlot) => {
                const guest = this.guests[guestIndex];

                if (
                  this.unallocatedGuests[0].length > 0
                  && guest.id === false
                ) {
                  // Auto room
                  this.groups[itemIndex][groupIndex][groupSlot] = this.unallocatedGuests[0].shift();
                }
              });
            });
  
            // After auto rooming, if there are no unallocated guests left then hide the list
            this.showUnallocatedGuestList = this.unallocatedGuests[0].length > 0;
          });
          
          this.items = items;
        }
      });

      this.item$.subscribe(item => {
        if (item) {
          this.itemIndexSource.next(items.findIndex(tmpItem => tmpItem.id === item.id));
          this.item = item;

          if (this.guestItems) {
            this.guestItems.forEach(guestItem => {
              guestItem.update();
            });
          }
        }
      });

      this.itemIndex$.subscribe(itemIndex => {
        this.shownUnallocatedGuests[0] = [];
        this.unallocatedGuests[0].forEach((guestIndex, unallocatedItemIndex) => {
          if (this.showUnallocatedItem(guestIndex, itemIndex, unallocatedItemIndex)) {
            this.shownUnallocatedGuests[0].push(guestIndex);
          }
        });
      });
    });
  }

  ngOnInit(): void {
  }

  public drop(event: CdkDragDrop<string[]>) {
    if (event.previousContainer === event.container) {
      moveItemInArray(event.container.data, event.previousIndex, event.currentIndex);
    } else {
      transferArrayItem(
        event.previousContainer.data,
        event.container.data,
        event.previousIndex,
        event.currentIndex
      );
    }
  }

  setItem(id) {
    this.listService.setItem(id);
  }

  save() {
    if (this.validate()) {
      let itemsRequest: SaveGuestsItems[] = [];
      this.items$.pipe(take(1)).subscribe(items => {
        items.forEach((item, itemIndex) => {
          let rooms = [];

          this.groups[itemIndex].forEach((group, groupIndex) => {
            rooms.push([]);

            this.groups[itemIndex][groupIndex].forEach((guestIndex, groupSlot) => {
              const guest = this.guests[guestIndex];
              let link = false;

              const id = itemIndex + '_' + groupIndex  + '_' + groupSlot;
              const item = this.guestItems.find(guestItem => guestItem.id === id);

              if (item.isValid()) {
                if (this.isGuestLinked(guestIndex)) {
                  link = guestIndex;
                }
                rooms[groupIndex].push({
                  id: guest.id,
                  updated: guest.updated,
                  first_name: guest.firstName,
                  last_name: guest.lastName,
                  email: guest.email,
                  link
                });
              }
            });
          });

          itemsRequest.push({
            id: item.id,
            rooms
          });
        });
      });
      this.listService.saveGuests(itemsRequest);
    }
  }

  clear(itemIndex, groupIndex, groupSlot) {
    const id = itemIndex + '_' + groupIndex  + '_' + groupSlot;
    const item = this.guestItems.find(guestItem => guestItem.id === id);

    if (item) {
      item.clear();
    }
  }

  restore(itemIndex, groupIndex, groupSlot) {
    const id = itemIndex + '_' + groupIndex  + '_' + groupSlot;
    const item = this.guestItems.find(guestItem => guestItem.id === id);

    if (item) {
      item.restore();
    }
  }

  cancel() {
    this.listService.reset();
    this.router.navigate(['/guest']);
  }

  validate() {
    let valid = false;
    let validItems = 0;
    let emptyItems = 0;

    if (this.guestItems && this.guestItems.length > 0) {
      this.guestItems.forEach(item => {
        if (item.isValid()) {
          validItems++;
        }

        if (item.isEmpty()) {
          emptyItems++;
        }
      });

      if (validItems > 0) {
        if (this.guestItems.length - emptyItems !== validItems) {
          alert('There are incomplete guest entries. Please fix the entries by clearing or completing all the required fields.');
        } else {
          let fullyAllocated = true;
          this.items.forEach((item, itemIndex) => {
            this.groups[itemIndex].forEach(group => {
              if (group.length !== item.paxCount) {
                fullyAllocated = false;
              }
            });
          });
          if (emptyItems > 0 || !fullyAllocated) {
            valid = confirm('The guest list is incomplete, are you sure you want to save?')
          } else {
            valid = true;
          }
        }
      } else {
        this.showValidationError = true;
      }
    } else {
      valid = false;
    }

    if (valid) {
      this.showValidationError = false;
    }

    return valid;
  }

  /**
   * Guests can only be cloned if
   * - The guest does not belong to the item
   * - The guest is not in unallocated
   *   The item is not not linked
   */
  canCloneGuest(guestIndex, itemIndex) {
    if (
      !this.isGuestUnallocated(guestIndex) &&
      !this.isGuestAllocatedToItem(guestIndex, itemIndex) &&
      !this.isGuestLinked(guestIndex)
    ) {
      return true;
    }

    return false;
  }

  isGuestUnallocated(guestIndex) {
    return this.unallocatedGuests[0].includes(guestIndex)
  }

  isGuestAllocatedToItem(guestIndex, itemIndex) {
    let allocatedToItem = false;
    if (guestIndex >= 0) {
      this.groups[itemIndex].forEach(group => {
          if (!allocatedToItem) {
            allocatedToItem = group.includes(guestIndex)
          }
      });
    }

    return allocatedToItem;
  }

  /**
   * Whether to show or hide the unallocated item in the unallocated list
   * @param guestIndex
   * @param itemIndex
   */
  showUnallocatedItem(guestIndex, itemIndex, unallocatedItemIndex) {
    let showItem = true;

    showItem = !this.isGuestAllocatedToItem(guestIndex, itemIndex);


    // Hide duplicate items in the unallocated list
    if (this.isGuestLinked(guestIndex)) {
      const guestLinksCount = this.unallocatedGuests[0].reduce((a, v) => (v === guestIndex ? a + 1 : a), 0);

      if (guestLinksCount > 1) {
        if (this.unallocatedGuests[0].indexOf(guestIndex) !== unallocatedItemIndex) {
          showItem = false;
        }
      }
    }

    return showItem;
  }

  /**
   * Clone the guest to an itinerary.
   * @param itemIndex 
   * @param groupIndex 
   * @param groupSlot 
   * @param toItem 
   */
  cloneGuest(guestIndex, toItem) {
    if (this.groups[toItem] !== 'undefined' && this.canCloneGuest(guestIndex, toItem)) {
      let groupIndex;
      let emptyGroupSlot = null;
      this.groups[toItem].forEach((group, tmpGroupIndex) => {
          group.forEach((guestIndex, groupSlot)=> {
            if (emptyGroupSlot === null) {
              if (this.isGuestEmpty(guestIndex) && !this.isGuestLinked(guestIndex)) {
                groupIndex = tmpGroupIndex;
                emptyGroupSlot = groupSlot;
              }
            }
          });
      });

      // If there is an empty, non linked guest slot, replace it with the cloned guest.
      if (emptyGroupSlot !== null) {
        this.groups[toItem][groupIndex][emptyGroupSlot] = guestIndex;
      } else {
        this.unallocatedGuests[0].push(guestIndex)
      }

      this.showUnallocatedGuestList = true;
      this.cd.detectChanges();
      return true;
    }

    return false;
  }

  cloneGroup(itemIndex, groupIndex, toItem) {
    this.groups[itemIndex][groupIndex].forEach(guestIndex => {
      this.cloneGuest(guestIndex, toItem)
    });
  }

  cloneItem(toItem) {
    const itemIndex = this.itemIndexSource.value;

    this.groups[itemIndex].forEach(groups => {
      groups.forEach(guestIndex => {
        this.cloneGuest(guestIndex, toItem)
      });
    });
  }

  cloneAllItems() {
    const currentItemIndex = this.itemIndexSource.value;

    this.groups.forEach((item, itemIndex) => {
      if (itemIndex !== currentItemIndex) {
        this.cloneItem(itemIndex);
      }
    });
  }

  /**
   * Whether it's a new and empty guest entry
   * @param guestIndex
   */
  isGuestEmpty(guestIndex) {
    const guest = this.guests[guestIndex];

    if (
      !guest.id &&
      !guest.updated &&
      !guest.locked &&
      guest.firstName === '' &&
      guest.lastName === '' &&
      guest.email === ''
    ) {
      return true;
    }

    return false;
  }

  /**
   * Whether there are linked guest items for the given guest.
   * @param guestIndex 
   */
  isGuestLinked(guestIndex) {
    let itemCount = 0;

    this.groups.forEach(item => {
      item.forEach(group => {
        if (group.includes(guestIndex)) {
          itemCount++;
        }
      });
    });

    const unallocatedCount = this.unallocatedGuests[0].indexOf(guestIndex);

    if (unallocatedCount >= 0) {
      itemCount++;
    }
    
    return itemCount > 1;
  }

  getUnallocatedGuestCount() {
    let count = 0;
    this.unallocatedGuests[0].forEach((guestIndex, unallocatedItemIndex) => {
      if (this.showUnallocatedItem(guestIndex, this.itemIndexSource.value, unallocatedItemIndex)) {
        count++;
      }
    });

    return count;
  }
}
