import { HttpClient } from '@angular/common/http';
import { Injectable } from '@angular/core';
import * as moment from 'moment';
import { BehaviorSubject, Observable, of } from 'rxjs';
import { catchError, first } from 'rxjs/operators';

export interface ReservationRequest {
  reservationId: string;
  email: string;
}

export interface GuestDetailsItemResponse {
  id: string;
  property_id: string;
  property_name: string;
  accomm_id: string;
  accomm_name: string;
  from: string;
  to: string;
  accomm_count: string;
  adult_count: string;
  child_count: string;
  pax_count: string;
}

export interface GuestDetailsGuestsResponse {
  id: string;
  item_id: string;
  first_name: string;
  last_name: string;
  email: string;
}
export interface GuestDetailsResponse {
  reservation_id: string;
  reservation_name: string;
  status_id: string;
  status_name: string;
  status_prov_expiry: string;
  agent_id: string;
  agent_name: string;
  voucher: string;
  items: GuestDetailsItemResponse[];
  guests: GuestDetailsGuestsResponse[];
}

export interface GuestDetailsItem {
  id: string;
  propertyId: string;
  propertyName: string;
  accommId: string;
  accommName: string;
  from: moment.Moment;
  to: moment.Moment;
  accommCount: number;
  adultCount: number;
  childCount: number;
  paxCount: number;
}

export interface GuestDetailsGuests {
  id: string;
  itemId: string;
  firstName: string;
  lastName: string;
  email: string;
}
export interface GuestDetails {
  reservationId: string;
  reservationName: string;
  statusId: string;
  statusName: string;
  statusProvExpiry: string;
  agentId: string;
  agentName: string;
  voucher: string;
  items: GuestDetailsItem[];
  guests: GuestDetailsGuests[];
}

export interface GuestRequest {
  id: string;
  item_id: string;
  first_name: string;
  last_name: string;
  email: string;
  mobile_number: string;
  birth_date: string;
  passport_number: string;
  notes_append: string;
  covid_yn: boolean;
}

export interface Guest {
  id: string;
  itemId: string;
  firstName: string;
  lastName: string;
  email: string;
  mobileNumber: string;
  birthDate: moment.Moment;
  passportNumber: string;
  notesAppend: string;
  covidYn: boolean;
}

export interface GuestField {
  id: string;
  name: string;
  compulsory: boolean;
  code: number;
}
export interface GuestSettings {
  indemnityMessage: string;
  fields: GuestField[];
  reservationOfficeEmail: string;
  principalId: string;
  terms: string;
  propertiesIndemnityNotes: {[propertyId: string]: string;}
}
@Injectable()
export class CheckInService {
  static DATE_FORMAT = 'YYYY/MM/DD';
  guestApiUrl = '/api/v1/public';

  /**
   * The booking email used for authentication
   */
  reservationEmail = '';

  /**
   * The source of guestSettings$.
   */
  private guestSettingsSource: BehaviorSubject<GuestSettings>;

  /**
   * The current guest settings
   */
  guestSettings$: Observable<GuestSettings>;

  /**
   * The source of guestDetails$.
   */
  private guestDetailsSource: BehaviorSubject<GuestDetails>;

  /**
   * The current guest details
   */
  guestDetails$: Observable<GuestDetails>;

  /**
   * The source of guest$.
   */
  private guestSource: BehaviorSubject<Guest>;

  /**
   * The current guest
   */
  guest$: Observable<Guest>;

  /**
   * The source of guests$.
   */
  private guestsSource: BehaviorSubject<Guest[]>;

  /**
   * The current guests
   */
  guests$: Observable<Guest[]>;

  /**
   * The source of item$.
   */
  private itemSource: BehaviorSubject<GuestDetailsItem>;

  /**
   * The current item
   */
  item$: Observable<GuestDetailsItem>;

  /**
   * The source of items$.
   */
  private itemsSource: BehaviorSubject<GuestDetailsItem[]>;

  /**
   * The current items
   */
  items$: Observable<GuestDetailsItem[]>;

  /**
   * The source of loading$.
   */
  private loadingSource: BehaviorSubject<boolean>;

  /**
   * Whether to show the loading indicator or not.
   */
  loading$: Observable<boolean>;

  /**
   * The source of error$.
   */
  private errorSource: BehaviorSubject<any>;

  /**
   * The current API error
   */
  error$: Observable<any>;

  /**
   * The source of step$.
   */
  private stepSource: BehaviorSubject<number>;

  /**
   * The current reservation step
   */
  step$: Observable<number>;

  constructor(
    private api: HttpClient
  ) {
    this.guestSettingsSource = new BehaviorSubject<GuestSettings>(null);
    this.guestSettings$ = this.guestSettingsSource.asObservable();
    this.guestDetailsSource = new BehaviorSubject<GuestDetails>(null);
    this.guestDetails$ = this.guestDetailsSource.asObservable();
    this.guestSource = new BehaviorSubject<Guest>(null);
    this.guest$ = this.guestSource.asObservable();
    this.guestsSource = new BehaviorSubject<Guest[]>(null);
    this.guests$ = this.guestsSource.asObservable();
    this.itemSource = new BehaviorSubject<GuestDetailsItem>(null);
    this.item$ = this.itemSource.asObservable();
    this.itemsSource = new BehaviorSubject<GuestDetailsItem[]>(null);
    this.items$ = this.itemsSource.asObservable();
    this.loadingSource = new BehaviorSubject<boolean>(false);
    this.loading$ = this.loadingSource.asObservable();
    this.errorSource = new BehaviorSubject<number>(null);
    this.error$ = this.errorSource.asObservable();
    this.stepSource = new BehaviorSubject<number>(1);
    this.step$ = this.stepSource.asObservable();

    this.getGuestSettings();

    this.guestDetails$.subscribe(guestDetails => {
      if (guestDetails) {
        const guests = [];
        guestDetails.guests.forEach(guest => {
          guests.push({
            id: guest.id,
            itemId: guest.itemId,
            firstName: guest.firstName,
            lastName: guest.lastName,
            email: guest.email,
            mobile: '',
            birthDate: '',
            passportNumber: '',
            covidYn: '',
          });
        });

        this.guestsSource.next(guests);

        if (guests.length === 1) {
          this.guestSource.next(guests[0]);
        }

        const items = [];
        guestDetails.items.forEach(item => {
          items.push({
            id: item.id,
            propertyId: item.propertyId,
            propertyName: item.propertyName,
            accommId: item.accommId,
            accommName: item.accommName,
            from: item.from,
            to: item.to,
            accommCount: item.accommCount,
            adultCount: item.adultCount,
            childCount: item.childCount,
            paxCount: item.paxCount,
          });
        });

        this.itemsSource.next(items);

        if (items.length === 1) {
          this.itemSource.next(items[0]);
        }
      }
    });

    this.item$.subscribe(item => {
      if (item) {
        const guests = this.getGuestsByItem(item);
        if (guests.length === 1) {
          this.guestSource.next(guests[0]);
        }
      }
    });

    this.guest$.subscribe(guest => {
      if (guest && this.stepSource.value === 1) {
        this.stepSource.next(2);
      }
    });
  }

  getGuestsByItem(item) {
    const guests = this.guestsSource.value.filter(guest => guest.itemId === item.id);

    return guests;
  }

  async getGuestSettings(): Promise<boolean> {
    const url = `${this.guestApiUrl}/get_guest_settings`;

    const guestSettingsResponse: GuestSettings = await this.api.get(url)
      .pipe(
        catchError(error => {
          return of(null);
        }))
      .toPromise();

    if (guestSettingsResponse === null) {
      return null;
    }

    if (guestSettingsResponse) {
      this.guestSettingsSource.next(guestSettingsResponse);
      return true;
    } else {
      return false;
    }
  }

  async setReservation(reservationRequest: ReservationRequest): Promise<boolean> {
    this.errorSource.next(null);
    this.loadingSource.next(true);
    const guestDetails = await this.getGuests(reservationRequest);

    if (guestDetails) {
      this.reservationEmail = reservationRequest.email;
      this.guestDetailsSource.next(guestDetails);
      this.loadingSource.next(false);
      return true;
    } else {
      this.loadingSource.next(false);
      return false;
    }
  }

  async getGuests(reservationRequest: ReservationRequest): Promise<GuestDetails | null> {
    const url = `${this.guestApiUrl}/get_guests`;

    const request = {
      reservation_id: reservationRequest.reservationId,
      email: reservationRequest.email,
    };

    const guestDetailsItems: GuestDetailsItem[] = [];
    const guestDetailsGuests: GuestDetailsGuests[] = [];
    let guestDetails: GuestDetails;

    const guestDetailsResponse: GuestDetailsResponse = await this.api.post(url, request)
      .pipe(
        catchError(error => {
          this.errorSource.next(error);
          return of(null);
        }))
      .toPromise();

    if (guestDetailsResponse === null) {
      return null;
    }

    guestDetailsResponse.items.forEach(guestDetailsItem => {
      guestDetailsItems.push({
        id: guestDetailsItem.id,
        propertyId: guestDetailsItem.property_id,
        propertyName: guestDetailsItem.property_name,
        accommId: guestDetailsItem.accomm_id,
        accommName: guestDetailsItem.accomm_name,
        from: moment(guestDetailsItem.from, CheckInService.DATE_FORMAT),
        to: moment(guestDetailsItem.to, CheckInService.DATE_FORMAT),
        accommCount: parseInt(guestDetailsItem.accomm_count, 10),
        adultCount: parseInt(guestDetailsItem.adult_count, 10),
        childCount: parseInt(guestDetailsItem.child_count, 10),
        paxCount: parseInt(guestDetailsItem.pax_count, 10),
      });
    });

    guestDetailsResponse.guests.forEach(guestDetailsGuest => {
      guestDetailsGuests.push({
        id: guestDetailsGuest.id,
        itemId: guestDetailsGuest.item_id,
        firstName: guestDetailsGuest.first_name,
        lastName: guestDetailsGuest.last_name,
        email: guestDetailsGuest.email,
      });
    });

    guestDetails = {
      reservationId: guestDetailsResponse.reservation_id,
      reservationName: guestDetailsResponse.reservation_name,
      statusId: guestDetailsResponse.status_id,
      statusName: guestDetailsResponse.status_name,
      statusProvExpiry: guestDetailsResponse.status_prov_expiry,
      agentId: guestDetailsResponse.agent_id,
      agentName: guestDetailsResponse.agent_name,
      voucher: guestDetailsResponse.voucher,
      items: guestDetailsItems,
      guests: guestDetailsGuests,
    };

    return guestDetails;
  }

  setGuest(guest) {
    this.guestSource.next(guest);
  }

  setItem(item) {
    this.itemSource.next(item);
  }

  checkInGuests(guest: Guest) {
    const url = `${this.guestApiUrl}/check_in_guests`;

    this.loadingSource.next(true);

    let birthDate = '';
    if (guest.birthDate) {
      birthDate = guest.birthDate.format(CheckInService.DATE_FORMAT);
    }

    const guestRequest: GuestRequest = {
      id: guest.id,
      item_id: guest.itemId,
      first_name: guest.firstName,
      last_name: guest.lastName,
      email: guest.email,
      mobile_number: guest.mobileNumber,
      birth_date: birthDate,
      passport_number: guest.passportNumber,
      notes_append: guest.notesAppend,
      covid_yn: guest.covidYn,
    };

    const request = {
      reservation_id: this.guestDetailsSource.value.reservationId,
      email: this.reservationEmail,
      guests: [guestRequest]
    };


    this.api.post(url, request)
      .pipe(
        first(),
        catchError(error => {
          return of(false);
        }))
      .subscribe(response => {
        this.loadingSource.next(false);
        this.stepSource.next(3);
      });
  }

  reset() {
    this.stepSource.next(1);
    this.guestDetailsSource.next(null);
    this.guestSource.next(null);
    this.guestsSource.next(null);
    this.itemSource.next(null);
    this.itemsSource.next(null);
  }

  /**
   * Clears the selected item
   */
  clearItem() {
    this.itemSource.next(null);
  }

  /**
   * Clears the selected guest
   */
  clearGuest() {
    this.stepSource.next(1);
    this.guestSource.next(null);
  }
}
