import { HttpClient } from '@angular/common/http';
import { Injectable } from '@angular/core';
import * as moment from 'moment';
import { BehaviorSubject, Observable, of } from 'rxjs';
import { catchError } from 'rxjs/operators';

export interface ReservationRequest {
  reservationId: string;
  email: string;
}

export interface ReservationDetailsItemResponse {
  id: string;
  property_id: string;
  property_name: string;
  accomm_id: string;
  accomm_name: string;
  from: string;
  to: string;
  accomm_count: string;
  adult_count: string;
  child_count: string;
  pax_count: string;
  guests: [{
    id: string;
    first_name: string;
    last_name: string;
    email: string;
  }[]];
}

export interface ReservationDetailsResponse {
  reservation_id: string;
  reservation_name: string;
  status_id: string;
  status_name: string;
  status_prov_expiry: string;
  agent_id: string;
  agent_name: string;
  contact_name_first: string;
  contact_name_last: string;
  voucher: string;
  items: ReservationDetailsItemResponse[];
  guests: {
    id: string;
    first_name: string;
    last_name: string;
    email: string;
  }[];
}

export interface ReservationItem {
  id: string;
  propertyId: string;
  propertyName: string;
  accommId: string;
  accommName: string;
  from: moment.Moment;
  to: moment.Moment;
  accommCount: number;
  adultCount: number;
  childCount: number;
  paxCount: number;
  guests: [{
    id: string;
    firstName: string;
    lastName: string;
    email: string;
  }][];
}

export interface ReservationDetails {
  reservationId: string;
  reservationName: string;
  statusId: string;
  statusName: string;
  statusProvExpiry: string;
  agentId: string;
  agentName: string;
  contactNameFirst: string;
  contactNameLast: string;
  voucher: string;
  items: ReservationItem[];
  guests: {
    id: string;
    firstName: string;
    lastName: string;
    email: string;
  }[];
}

export interface GuestRequest {
  id: string;
  first_name: string;
  last_name: string;
  email: string;
  mobile_number: string;
  birth_date: string;
  passport_number: string;
  notes_append: string;
  covid_yn: boolean;
}

export interface Guest {
  id: string;
  firstName: string;
  lastName: string;
  email: string;
  mobileNumber: string;
  birthDate: moment.Moment;
  passportNumber: string;
  notesAppend: string;
  covidYn: boolean;
}

export interface GuestField {
  id: string;
  name: string;
  compulsory: boolean;
  code: number;
}
export interface GuestSettings {
  indemnityMessage: string;
  fields: GuestField[];
  reservationOfficeEmail: string;
  principalId: string;
  terms: string;
  propertiesIndemnityNotes: {[propertyId: string]: string;}
}

export interface SaveGuestsItems {
  id: string,
  rooms: {
    firstName: string,
    lastName: string,
    email: string
  }[]
}
@Injectable()
export class GuestListService {
  static DATE_FORMAT = 'YYYY/MM/DD';
  guestApiUrl = '/api/v1/public';

  /**
   * The source of guestSettings$.
   */
  private guestSettingsSource: BehaviorSubject<GuestSettings>;

  /**
   * The current guest settings
   */
  guestSettings$: Observable<GuestSettings>;

  /**
   * The source of guestDetails$.
   */
  private reservationDetailsSource: BehaviorSubject<ReservationDetails>;

  /**
   * The current guest details
   */
  reservationDetails$: Observable<ReservationDetails>;

  /**
   * The source of guest$.
   */
  private guestSource: BehaviorSubject<Guest>;

  /**
   * The current guest
   */
  guest$: Observable<Guest>;

  /**
   * The source of guests$.
   */
  private guestsSource: BehaviorSubject<Guest[]>;

  /**
   * The current guests
   */
  guests$: Observable<Guest[]>;

  /**
   * The source of item$.
   */
  private itemSource: BehaviorSubject<ReservationItem>;

  /**
   * The current item
   */
  item$: Observable<ReservationItem>;

  /**
   * The source of items$.
   */
  private itemsSource: BehaviorSubject<ReservationItem[]>;

  /**
   * The current items
   */
  items$: Observable<ReservationItem[]>;

  /**
   * The source of loading$.
   */
  private loadingSource: BehaviorSubject<boolean>;

  /**
   * Whether to show the loading indicator or not.
   */
  loading$: Observable<boolean>;

  /**
   * The source of error$.
   */
  private errorSource: BehaviorSubject<any>;

  /**
   * The current API error
   */
  error$: Observable<any>;

  /**
   * The source of step$.
   */
  private stepSource: BehaviorSubject<number>;

  /**
   * The current reservation step
   */
  step$: Observable<number>;

  /**
   * The consultnat email used for authentication
   */
  protected email: string;

  constructor(
    private api: HttpClient
  ) {
    this.guestSettingsSource = new BehaviorSubject<GuestSettings>(null);
    this.guestSettings$ = this.guestSettingsSource.asObservable();
    this.reservationDetailsSource = new BehaviorSubject<ReservationDetails>(null);
    this.reservationDetails$ = this.reservationDetailsSource.asObservable();
    this.guestSource = new BehaviorSubject<Guest>(null);
    this.guest$ = this.guestSource.asObservable();
    this.guestsSource = new BehaviorSubject<Guest[]>(null);
    this.guests$ = this.guestsSource.asObservable();
    this.itemSource = new BehaviorSubject<ReservationItem>(null);
    this.item$ = this.itemSource.asObservable();
    this.itemsSource = new BehaviorSubject<ReservationItem[]>(null);
    this.items$ = this.itemsSource.asObservable();
    this.loadingSource = new BehaviorSubject<boolean>(false);
    this.loading$ = this.loadingSource.asObservable();
    this.errorSource = new BehaviorSubject<number>(null);
    this.error$ = this.errorSource.asObservable();
    this.stepSource = new BehaviorSubject<number>(1);
    this.step$ = this.stepSource.asObservable();

    this.getGuestSettings();
  }

  async getGuestSettings(): Promise<boolean> {
    const url = `${this.guestApiUrl}/get_guest_settings`;

    const guestSettingsResponse: GuestSettings = await this.api.get(url)
      .pipe(
        catchError(error => {
          return of(null);
        }))
      .toPromise();

    if (guestSettingsResponse === null) {
      return null;
    }

    if (guestSettingsResponse) {
      this.guestSettingsSource.next(guestSettingsResponse);
      return true;
    } else {
      return false;
    }
  }

  async setReservation(reservationRequest: ReservationRequest): Promise<boolean> {
    this.loadingSource.next(true);
    this.errorSource.next(null);
    const reservationDetails = await this.getReservationDetails(reservationRequest);
    if (reservationDetails) {
      const items = reservationDetails.items;
      this.itemsSource.next(items);
      this.reservationDetailsSource.next(reservationDetails);
      this.itemSource.next(items[0]);
      this.stepSource.next(2);
      this.loadingSource.next(false);
      return true;
    } else {
      this.loadingSource.next(false);
      return false;
    }
  }

  async getReservationDetails(reservationRequest: ReservationRequest): Promise<ReservationDetails | null> {
    const url = `${this.guestApiUrl}/get_reservation`;

    const request = {
      reservation_id: reservationRequest.reservationId,
      email: reservationRequest.email,
    };

    const reservationItems: ReservationItem[] = [];
    let reservationDetails: ReservationDetails;

    const reservationDetailsResponse: ReservationDetailsResponse = await this.api.post(url, request)
      .pipe(
        catchError(error => {
          this.errorSource.next(error);
          return of(null);
        }))
      .toPromise();

    if (reservationDetailsResponse === null) {
      return null;
    }

    this.email = reservationRequest.email;

    reservationDetailsResponse.items.forEach(guestDetailsItem => {
      let guests = [];

      guestDetailsItem.guests.forEach((group, groupIndex) => {
        guests.push([]);
        group.forEach(guest => {
          guests[groupIndex].push({
            id: guest.id,
            firstName: guest.first_name,
            lastName: guest.last_name,
            email: guest.email,
          });
        });
      });

      reservationItems.push({
        id: guestDetailsItem.id,
        propertyId: guestDetailsItem.property_id,
        propertyName: guestDetailsItem.property_name,
        accommId: guestDetailsItem.accomm_id,
        accommName: guestDetailsItem.accomm_name,
        from: moment(guestDetailsItem.from, GuestListService.DATE_FORMAT),
        to: moment(guestDetailsItem.to, GuestListService.DATE_FORMAT),
        accommCount: parseInt(guestDetailsItem.accomm_count, 10),
        adultCount: parseInt(guestDetailsItem.adult_count, 10),
        childCount: parseInt(guestDetailsItem.child_count, 10),
        paxCount: parseInt(guestDetailsItem.pax_count, 10),
        guests: guests,
      });
    });

    let guests = [];
    reservationDetailsResponse.guests.forEach(guest => {
      guests.push({
        id: guest.id,
        firstName: guest.first_name,
        lastName: guest.last_name,
        email: guest.email,
      });
    });

    reservationDetails = {
      reservationId: reservationDetailsResponse.reservation_id,
      reservationName: reservationDetailsResponse.reservation_name,
      statusId: reservationDetailsResponse.status_id,
      statusName: reservationDetailsResponse.status_name,
      statusProvExpiry: reservationDetailsResponse.status_prov_expiry,
      agentId: reservationDetailsResponse.agent_id,
      agentName: reservationDetailsResponse.agent_name,
      contactNameFirst: reservationDetailsResponse.contact_name_first,
      contactNameLast: reservationDetailsResponse.contact_name_last,
      voucher: reservationDetailsResponse.voucher,
      items: reservationItems,
      guests: guests
    };

    return reservationDetails;
  }

  async saveGuests(items: SaveGuestsItems[]): Promise<boolean> {
    const url = `${this.guestApiUrl}/save_guests`;

    this.loadingSource.next(true);

    const reservationDetails = this.reservationDetailsSource.value;

    const request = {
      reservation_id: reservationDetails.reservationId,
      email: this.email,
      items
    };


    const SaveGuestsResponse = await this.api.post(url, request)
      .pipe(
        catchError(error => {
          console.log(error);
          return of(false);
        }))
      .toPromise();

    if (SaveGuestsResponse) {
      this.stepSource.next(3);
      this.loadingSource.next(false);
      return true;
    } else {
      this.loadingSource.next(false);
      return false;
    }
  }

  setItem(id: string) {
    const item = this.itemsSource.value.find(item => item.id === id);

    if (item) {
      this.itemSource.next(item);
    }
  }

  reset() {
    this.stepSource.next(1);
    this.reservationDetailsSource.next(null);
    this.guestSource.next(null);
    this.guestsSource.next(null);
    this.itemSource.next(null);
    this.itemsSource.next(null);

  }
}
