import { HttpClient } from '@angular/common/http';
import { Injectable } from '@angular/core';
import { config } from 'process';
import { BehaviorSubject, Observable, of } from 'rxjs';
import { catchError } from 'rxjs/operators';

export interface GuestField {
  id: string;
  name: string;
  compulsory: boolean;
  code: number;
}

export interface GuestSettings {
  indemnityMessage: string;
  fields: GuestField[];
  reservationOfficeEmail: string;
  principalId: string;
  terms: string;
  propertiesIndemnityNotes: {[propertyId: string]: string;}
  isV3Imageset: boolean;
}

@Injectable({
  providedIn: 'root'
})
export class GuestService {
  // Email regex is hard: https://fightingforalostcause.net/content/misc/2006/compare-email-regex.php
  // Built in Validators.email does not require a top level domain
  // This regex does not match
  public static emailPattern = /^[-a-z0-9~!$%^&*_=+}{\'?]+(\.[-a-z0-9~!$%^&*_=+}{\'?]+)*@([a-z0-9_][-a-z0-9_]*(\.[-a-z0-9_]+)*\.(aero|arpa|biz|com|coop|edu|gov|info|int|mil|museum|name|net|org|pro|travel|mobi|[a-z][a-z])|([0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}))(:[0-9]{1,5})?$/i;
  
  guestApiUrl = '/api/v1/public';
  
  /**
   * The source of guestSettings$.
   */
  private guestSettingsSource: BehaviorSubject<GuestSettings>;

  /**
   * The current guest settings
   */
  guestSettings$: Observable<GuestSettings>;

  /**
   * The source of loading$.
   */
  private loadingSource: BehaviorSubject<boolean>;

  /**
   * Whether to show the loading indicator or not.
   */
  loading$: Observable<boolean>;

  constructor(
    private api: HttpClient
  ) {
    this.guestSettingsSource = new BehaviorSubject<GuestSettings>(null);
    this.guestSettings$ = this.guestSettingsSource.asObservable();
    this.loadingSource = new BehaviorSubject<boolean>(true);
    this.loading$ = this.loadingSource.asObservable();

    this.getGuestSettings();

    this.guestSettings$.subscribe(guestSettings => {
      if (guestSettings) {
        this.loadingSource.next(false);
      }
    });
  }

  async getGuestSettings(): Promise<boolean> {
    const url = `${this.guestApiUrl}/get_guest_settings`;

    const guestSettingsResponse: GuestSettings = await this.api.get(url)
      .pipe(
        catchError(error => {
          return of(null);
        }))
      .toPromise();

    if (guestSettingsResponse === null) {
      return null;
    }

    if (guestSettingsResponse) {
      this.guestSettingsSource.next(guestSettingsResponse);
      return true;
    } else {
      return false;
    }
  }
}
