import { Observable, BehaviorSubject, combineLatest } from 'rxjs';
import { Component, OnInit, OnDestroy, ViewChild, TemplateRef } from '@angular/core';
import { NgxMaterialTimepickerTheme } from 'ngx-material-timepicker';
import { MinLOSService } from './minlos.service';
import { trigger, transition, style, animate } from '@angular/animations';
import { MatDialog } from '@angular/material/dialog';
import { MatTableDataSource } from '@angular/material/table';
import { FormControl, Validators, FormGroup, FormGroupDirective, ValidatorFn } from '@angular/forms';
import { MatPaginator } from '@angular/material/paginator';
import { MatSnackBar } from '@angular/material/snack-bar';

import * as moment from 'moment-timezone';
import { SidebarService } from 'src/app/shared/services/sidebar/sidebar.service';
import { FilterService } from 'src/app/rooming-calendar/services/filter/filter.service';
import { AccommodationFilterComponent } from 'src/app/shared/components/accommodation-filter/accommodation-filter.component';
import { first } from 'rxjs/operators';
import { AuthorisationService } from 'src/app/shared/services/authorisation/authorisation.service';

import { Title } from '@angular/platform-browser';

export interface Rule {
  id: string;
  description: string;
  ratePeriods: string[];
  rateTypes: string[];
  agents: {id: string; name: string}[];
  accommodationTypes: string[];
  minLos: number;
  lapseLeadNights: number;
  applyToAllocations: string;
  lastDayDeparture: boolean;
  minLosDays: {
    Sunday: boolean,
    Monday: boolean,
    Tuesday: boolean,
    Wednesday: boolean,
    Thursday: boolean,
    Friday: boolean,
    Saturday: boolean
  };
};

export function daysOfWeekValidator(minRequired = 1): ValidatorFn {
  return function validate (formGroup: FormGroup) {
    let checked = 0;

    Object.keys(formGroup.controls).forEach(key => {
      const control = formGroup.controls[key];

      if (control.value === true) {
        checked ++;
      }
    });

    if (checked < minRequired) {
      return {
        requireCheckboxesToBeChecked: true,
      };
    }

    return null;
  };
}

@Component({
  selector: 'app-minlos',
  templateUrl: './minlos.component.html',
  styleUrls: ['./minlos.component.scss'],
  providers: [MinLOSService],
  animations: [
    trigger(
      'fadeOutAnimation',
      [
        transition(
          ':leave',
          [
            style({ opacity: 1 }),
            animate('500ms ease-in',
              style({ opacity: 0 }))
          ]
        )
      ]
    )
  ]
})
export class MinLOSComponent implements OnInit, OnDestroy {
  @ViewChild(FormGroupDirective) formGroupDirective: FormGroupDirective;
  @ViewChild('portal') portal: TemplateRef<any>;
  @ViewChild(AccommodationFilterComponent) propertyFilter: AccommodationFilterComponent;

  properties$ = this.filterService.filterProperties$;
  selectedAccommodations$ = this.filterService.accommodations$;

  paginator: MatPaginator;
  displayedColumns: string[] = ['description','minLos', 'applyToAllocations', 'minLosDays', 'actions'];
  dataSource: MatTableDataSource<Rule> = new MatTableDataSource([]);
  filterWorkflows = '';

  rulesLoaded: Promise<boolean>;

  rule: Rule = null;
  rules = [];
  ruleId: string;

  rateTypes = [];
  ratePeriods = [];
  agents = [];

  selectedAccommodationTypesText: string = 'Select accommodation types';
  selectedRatePeriods: string[] = [];
  selectedAgents: {id: string; name: string}[] = [];
  selectedRateTypes: string[] = [];

  /**
   * Used for the legacy agents select popup
   */
   legacySelectedAgentsIds = '';
  legacySelectedAgentsNames = 'All';


  public frmMinLosRules: FormGroup = new FormGroup({
    frmControlMinLosRuleDescription: new FormControl('', [Validators.required]),
    frmControlMinLosRatePeriodInd: new FormControl([], [Validators.required]),
    frmControlMinLosRateTypeInd: new FormControl([], [Validators.required]),
    frmControlMinLosAgentInd: new FormControl([]),
    frmControlMinLosAccomodationType: new FormControl([], [Validators.required]),
    frmControlMinLosMinLos: new FormControl(1, [Validators.required, Validators.min(1), Validators.max(99), Validators.pattern(/^[0-9]\d*$/)]),
    frmControlMinLosLapseLeadTime: new FormControl(0, [Validators.required, Validators.min(0), , Validators.pattern(/^[0-9]\d*$/)]),
    frmControlMinLosApplyToAllocationsYn: new FormControl('0', []),
    frmControlMinLosDayDepartureYn: new FormControl('0', []),

    // Days of the week
    checkboxGroup: new FormGroup({
      frmControlMinLosDaySunday: new FormControl(false, []),
      frmControlMinLosDayMonday: new FormControl(false, []),
      frmControlMinLosDayTuesday: new FormControl(false, []),
      frmControlMinLosDayWednesday: new FormControl(false, []),
      frmControlMinLosDayThursday: new FormControl(false, []),
      frmControlMinLosDayFriday: new FormControl(false, []),
      frmControlMinLosDaySaturday: new FormControl(false, []),
    }, daysOfWeekValidator(1))
  });

  @ViewChild(MatPaginator)
  set appBacon(paginator: MatPaginator) {
    this.paginator = paginator;
  }

  /**
   * Whether the workflows is/are loading
   */
  loading$: Observable<boolean>;

  /**
   * The source for loading$.
   */
  private loadingSource: BehaviorSubject<boolean>;

  public canEdit: boolean = true;

  constructor(
    private minLosService: MinLOSService,
    public dialog: MatDialog,
    private _snackBar: MatSnackBar,
    private filterService: FilterService,
    private sidebarService: SidebarService,
    public authorisation: AuthorisationService,
    private titleService: Title
  ) {
    (window as any)['ng_legacy_set_agents'] = (agentIds, agentNames) => {
      this.onAgentsSelect(agentIds, agentNames);
    };

    this.loadingSource = new BehaviorSubject<boolean>(false);
    this.loading$ = this.loadingSource.asObservable();
  }

  ngOnInit(): void {
    this.refresh();

    this.titleService.setTitle('MinLOS Rules Code Table');
    this.selectedAccommodations$.subscribe((selectedAccommodationTypes) => {
      this.frmMinLosRules.get('frmControlMinLosAccomodationType').setValue(selectedAccommodationTypes);

      if (selectedAccommodationTypes.length === 0) {
        this.selectedAccommodationTypesText = 'Select accommodation types';
      } else {
        this.properties$.pipe(first()).subscribe(properties => {
          let accommodations = [];
          properties.forEach(property => {
            const names = property.accommodations.filter(accomm => selectedAccommodationTypes.includes(accomm.id)).map(accomm => accomm.name + ' - ' + property.shortName);
            accommodations = [...accommodations, ...names];
          });

          this.selectedAccommodationTypesText = accommodations.join(', ');
        })
      }
    });

    this.authorisation.accessLoaded$.subscribe(accessLoaded => {
      if (accessLoaded) {
        if (this.authorisation.functionAccessLevel(68) < 10) {
          this.frmMinLosRules.disable();
        }
      }
    });
  }

  refresh() {
    this.rulesLoaded = Promise.resolve(false);

    this.resetForm();

    this.getRules().subscribe(
      (data: any) => {

        this.rules = [];
        data.rules.forEach((rule: any) => {
          this.rules.push(rule)
        });


        this.rateTypes = data.rateTypes;
        this.ratePeriods = data.ratePeriods;
        this.agents = data.agents;

        this.dataSource = new MatTableDataSource(this.rules);
        this.dataSource.paginator = this.paginator;
        this.rulesLoaded = Promise.resolve(true);
      },
      (err: any) => console.error(err),
      () => {
        this.loadingSource.next(false);
      }
    );
  }

  ngOnDestroy() { }

  resetForm() {
    this.frmMinLosRules.reset();
    this.ruleId = null;
    this.rule = null;
    this.filterService.onAccommodationsChanged([], false);
    this.selectedAgents = [];
    this.selectedRatePeriods = [];
    this.selectedRateTypes = [];

    this.legacySelectedAgentsNames = 'All';
    this.legacySelectedAgentsIds = '';
  }

  save() {
    if (this.frmMinLosRules.valid) {
      this.loadingSource.next(true);

      if (this.ruleId) {
        const minLosDays = {
          Sunday: this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDaySunday').value,
          Monday: this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDayMonday').value,
          Tuesday: this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDayTuesday').value,
          Wednesday: this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDayWednesday').value,
          Thursday: this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDayThursday').value,
          Friday: this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDayFriday').value,
          Saturday: this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDaySaturday').value
        };

        this.minLosService.updateRule(
          this.ruleId,
          this.frmMinLosRules.get('frmControlMinLosRuleDescription').value,
          this.frmMinLosRules.get('frmControlMinLosRatePeriodInd').value,
          this.frmMinLosRules.get('frmControlMinLosRateTypeInd').value,
          this.frmMinLosRules.get('frmControlMinLosAgentInd').value,
          this.frmMinLosRules.get('frmControlMinLosAccomodationType').value,
          this.frmMinLosRules.get('frmControlMinLosMinLos').value,
          this.frmMinLosRules.get('frmControlMinLosLapseLeadTime').value,
          this.frmMinLosRules.get('frmControlMinLosApplyToAllocationsYn').value,
          this.frmMinLosRules.get('frmControlMinLosDayDepartureYn').value,
          minLosDays
        ).subscribe(
          (data: any) => { },
          (err: any) => console.error(err),
          () => {
            this.refresh();
            this.loadingSource.next(false);
          }
        );
      } else {
        const minLosDays = {
          Sunday: this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDaySunday').value,
          Monday: this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDayMonday').value,
          Tuesday: this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDayTuesday').value,
          Wednesday: this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDayWednesday').value,
          Thursday: this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDayThursday').value,
          Friday: this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDayFriday').value,
          Saturday: this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDaySaturday').value
        };

        this.minLosService.addRule(
          this.frmMinLosRules.get('frmControlMinLosRuleDescription').value,
          this.frmMinLosRules.get('frmControlMinLosRatePeriodInd').value,
          this.frmMinLosRules.get('frmControlMinLosRateTypeInd').value,
          this.frmMinLosRules.get('frmControlMinLosAgentInd').value,
          this.frmMinLosRules.get('frmControlMinLosAccomodationType').value,
          this.frmMinLosRules.get('frmControlMinLosMinLos').value,
          this.frmMinLosRules.get('frmControlMinLosLapseLeadTime').value,
          this.frmMinLosRules.get('frmControlMinLosApplyToAllocationsYn').value,
          this.frmMinLosRules.get('frmControlMinLosDayDepartureYn').value,
          minLosDays
        ).subscribe(
          (data: any) => { },
          (err: any) => console.error(err),
          () => {
            this.refresh();
            this.loadingSource.next(false);
          }
        );
      }
    } else {
      this.frmMinLosRules.get('frmControlMinLosAccomodationType').markAsTouched();
      this.frmMinLosRules.get('checkboxGroup').markAsTouched();
    }
  }

  delete(id = null) {
    if (!id) {
      id = this.ruleId;
    }
    if (id) {
      this.loadingSource.next(true);
      this.minLosService.deleteRule(id).subscribe(
        (data: any) => { },
        (err: any) => console.error(err),
        () => {
          this.refresh();
          this.resetForm();
          this.loadingSource.next(false);
        }
      );
    }
  }

  numberOnly(event): boolean {
    const charCode = (event.which) ? event.which : event.keyCode;
    if (charCode > 31 && (charCode < 48 || charCode > 57)) {
      return false;
    }
    return true;
  }

  getRules() {
    return this.minLosService.getRules();
  }

  hasError (controlName: string, errorName: string) {
    return this.frmMinLosRules.controls[controlName].hasError(errorName);
  }

  edit(ruleId: string): void {
    this.loadingSource.next(true);
    this.sidebarService.dispose();
    this.minLosService.getRules([ruleId])
      .subscribe(
        (data: any) => {
          const rule = data.rules[0] as Rule ?? null;
          if (rule) {
            this.resetForm();

            this.rule = rule;

            this.selectedRateTypes = rule.rateTypes;
            this.selectedRatePeriods = rule.ratePeriods;
            this.selectedAgents = rule.agents;

            this.legacySelectedAgentsNames = 'All';
            this.legacySelectedAgentsIds = '';

            let agentNames = [];
            let agentIds = [];
            rule.agents.forEach(agentId => {
              agentNames.push(data.agents.find(agent => agent.id === agentId).name);
              agentIds.push(data.agents.find(agent => agent.id === agentId).id);

            });

            if (agentIds.length > 0) {
              this.legacySelectedAgentsNames = agentNames.join(', ');
              this.legacySelectedAgentsIds = agentIds.join(':');
            }

            this.ruleId = rule.id;
            this.frmMinLosRules.get('frmControlMinLosRuleDescription').setValue(rule.description);
            this.frmMinLosRules.get('frmControlMinLosRatePeriodInd').setValue(this.selectedRatePeriods);
            this.frmMinLosRules.get('frmControlMinLosRateTypeInd').setValue(this.selectedRateTypes);
            this.frmMinLosRules.get('frmControlMinLosAgentInd').setValue(this.selectedAgents);

            this.filterService.onAccommodationsChanged(rule.accommodationTypes, false);
            this.frmMinLosRules.get('frmControlMinLosMinLos').setValue(rule.minLos);
            this.frmMinLosRules.get('frmControlMinLosLapseLeadTime').setValue(rule.lapseLeadNights);
            this.frmMinLosRules.get('frmControlMinLosApplyToAllocationsYn').setValue(rule.applyToAllocations);
            this.frmMinLosRules.get('frmControlMinLosDayDepartureYn').setValue(rule.lastDayDeparture);

            this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDaySunday').setValue(rule.minLosDays.Sunday);
            this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDayMonday').setValue(rule.minLosDays.Monday);
            this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDayTuesday').setValue(rule.minLosDays.Tuesday);
            this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDayWednesday').setValue(rule.minLosDays.Wednesday);
            this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDayThursday').setValue(rule.minLosDays.Thursday);
            this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDayFriday').setValue(rule.minLosDays.Friday);
            this.frmMinLosRules.get('checkboxGroup.frmControlMinLosDaySaturday').setValue(rule.minLosDays.Saturday);

            this.frmMinLosRules.markAsDirty();

            this.rule = rule;
            this.ruleId = rule.id;
          }
        },
        (err: any) => console.error(err),
        () => {
          this.loadingSource.next(false);
        }
      );
  }

  openPopup(url: string, name?: string, w?: number, h?: number): void {
    if (!name) { name = ''; }
    if (!w) { w = 400; }
    if (!h) { h = 300; }

    const dualScreenLeft = window.screenLeft != undefined ? window.screenLeft : screenLeft;

    const windowWidth = window.innerWidth ? window.innerWidth : document.documentElement.clientWidth ? document.documentElement.clientWidth : screen.width;
    const windowHeight = window.innerHeight ? window.innerHeight : document.documentElement.clientHeight ? document.documentElement.clientHeight : screen.height;

    var systemZoom = windowWidth / window.screen.availWidth;

    const x = ((windowWidth / 2) - (w / 2))/ systemZoom + dualScreenLeft; // X co-ord to center screen horizontally
    const y = ((windowHeight - h)/2); // X co-ord to center screen virtically

    const size = 'fullscreen=0,width=' + w + ',height=' + h + ',left=' + x + ',top=' + y;

    const windowRef = window.open(url, name, size);
  }

  selectAgents() {
    const url = [
      "reservation.php?661",
      'agent',
      this.legacySelectedAgentsIds
    ];

    this.openPopup(url.join("+"), '', 280,360);
  }

  onAgentsSelect(agentIds: string, agentNames: string) {

    if (agentIds) {
      this.frmMinLosRules.get('frmControlMinLosAgentInd').setValue(agentIds.split(':'));
    } else {
      this.frmMinLosRules.get('frmControlMinLosAgentInd').setValue([]);
    }

    this.legacySelectedAgentsIds = agentIds;
    this.legacySelectedAgentsNames = agentNames;
  }

  openSidebar() {
    const sidebar = {
      id: 'filters',
      template: this.portal
    };
    this.sidebarService.showSidebar(sidebar);
  }

  close(): void {
    this.sidebarService.close();
  }

  apply(): void {
    combineLatest([this.filterService.accommodations$]).pipe(
      first()
    ).subscribe(([accommodations]) => {
      const currentAccommodations = this.propertyFilter.selectedAccommodations;

      if (this.accommodationsChanged(accommodations, currentAccommodations)) {
        this.filterService.onAccommodationsChanged([...currentAccommodations], false);
      }
    });

    this.close();
  }

  private accommodationsChanged(original: string[], current: string[]): boolean {
    if (original.length !== current.length) {
      return true;
    }

    const originalClone = [...original];
    originalClone.sort();
    const currentClone = [...current];
    currentClone.sort();

    for (let i = 0; i < originalClone.length; i++) {
      if (originalClone[i] !== currentClone[i]) {
        return true;
      }
    }

    return false;
  }
}
