import { Observable, BehaviorSubject } from 'rxjs';
import { Component, OnInit, OnDestroy, ViewChild, ElementRef, TemplateRef } from '@angular/core';
import { NgxMaterialTimepickerTheme } from 'ngx-material-timepicker';
import { WorkflowService } from './workflow.service';
import { trigger, transition, style, animate } from '@angular/animations';
import { ConfirmationDialogComponent } from '../../../shared/components/confirmation-dialog/confirmation-dialog.component';
import { MatDialog } from '@angular/material/dialog';
import { MatTableDataSource } from '@angular/material/table';
import { FormControl, Validators, FormGroup, FormGroupDirective, ValidatorFn } from '@angular/forms';
import { MatPaginator } from '@angular/material/paginator';
import { MatSnackBar } from '@angular/material/snack-bar';
import { Title } from '@angular/platform-browser';
import * as moment from 'moment-timezone';



export interface Workflow {
  id: string;
  name: string;
  workflowType: string;
  description: string;
  inactiveYn: string;
  lastRunText: string;
  lastRunHours: number;
}

export interface WorkflowType {
  id: string;
  name: string;
}

export interface WorkflowTrigger {
  id: string;
  name: string;
}

export function minRequirement(minRequired = 1): ValidatorFn {
  return function validate (formGroup: FormGroup) {
    let checked = 0;

    Object.keys(formGroup.controls).forEach(key => {
      const control = formGroup.controls[key];

      if (control.value === true) {
        checked ++;
      }
    });

    if (checked < minRequired) {
      return {
        requireCheckboxesToBeChecked: true,
      };
    }

    return null;
  };
}

@Component({
  selector: 'app-workflows',
  templateUrl: './workflows.component.html',
  styleUrls: ['./workflows.component.scss'],
  providers: [WorkflowService],
  animations: [
    trigger(
      'fadeOutAnimation',
      [
        transition(
          ':leave',
          [
            style({ opacity: 1 }),
            animate('500ms ease-in',
              style({ opacity: 0 }))
          ]
        )
      ]
    )
  ]
})
export class WorkflowsComponent implements OnInit, OnDestroy {
  @ViewChild(FormGroupDirective) formGroupDirective: FormGroupDirective;
  @ViewChild('emailBodyTemplate', { read: ElementRef }) emailBodyTemplate: ElementRef<any>;
  @ViewChild('emailSubjectTemplate', { read: ElementRef }) emailSubjectTemplate: ElementRef<any>;

  private provExpiryEmailSubject = 'Confirm today to reserve your space - {{ResOffice|Organisation Name}} - {{ResDetails|NumberAndName}}';
  private provExpiryEmailMessage = `Dear {{ResDetails|Contact Firstname}},

Our availability is filling up, please reply to us to confirm how we should manage this provisional reservation:

{{ResDetails|Details}}

We look forward to receiving a reply from you today to confirm your space!

Regards
{{Consultant|Firstname}} {{Consultant|Lastname}}
{{Consultant|Email}}`;

  private paymentsDueEmailSubject = 'Payment due {{ResDetails|NumberAndName}} - {{Invoice|Number}}';
  private paymentsDueEmailMessage = `This is a reminder for payment due on {{ResDetails|Number}} - {{ResDetails|Name}}. Payment details are on the attached documentation. Please arrange payment and notify us when payment is made. Payment can be made via bank details or online on our website: {{Principal|Payment URL}}.

If you have any queries regarding your payment please reach me at {{Consultant|Email}}.

Regards,
{{Consultant|Firstname}} {{Consultant|Lastname}}`;

  paginator: MatPaginator;
  displayedColumns: string[] = ['name', 'workflowType', 'description', 'inactiveYn', 'lastRun', 'actions'];
  dataSource: MatTableDataSource<Workflow> = new MatTableDataSource([]);
  filterWorkflows = '';

  /**
   * Used for the legacy source select popup
   */
  legacySelectedSourceIds = '';
  legacySelectedSourceNames = 'All';

  @ViewChild(MatPaginator)
  set appBacon(paginator: MatPaginator) {
    this.paginator = paginator;
    this.dataSource.paginator = this.paginator;
  }

  dialogTitle = 'confirmation-dialog';
  public frmWorkflows: FormGroup = this.getForm();

  rrTimePickerTheme: NgxMaterialTimepickerTheme = {
    container: {
      bodyBackgroundColor: '#fff',
      buttonColor: '#c9252f',
    },
    dial: {
      dialBackgroundColor: '#c9252f',
    },
    clockFace: {
      clockFaceBackgroundColor: '#f0f0f0',
      clockHandColor: '#c9252f',
      clockFaceTimeInactiveColor: '#6c6c6c'
    }
  };

  workflowInactiveText = 'Off';
  workflowsLoaded: Promise<boolean>;
  workflows: Workflow[] = [];
  workflowTypes: WorkflowType[] = [
    { id: '0', name: 'Guest check-in reminders'},
    { id: '1', name: 'Provisional expiry reminders'},
    { id: '2', name: 'Payments due reminders'},
  ];
  workflowTriggers: WorkflowTrigger[] = [];
  workflowId: string = null;

  /**
   * Whether the workflows is/are loading
   */
  loading$: Observable<boolean>;

  /**
   * The source for loading$.
   */
  private loadingSource: BehaviorSubject<boolean>;

  public canEdit: boolean = true;

  constructor(
    private workflowService: WorkflowService,
    public dialog: MatDialog,
    private _snackBar: MatSnackBar,
    private titleService: Title,
  ) {
    this.loadingSource = new BehaviorSubject<boolean>(false);
    this.loading$ = this.loadingSource.asObservable();

    this.frmWorkflows.valueChanges.subscribe(value => {
      this.onChangeInactive(value.frmControlawWorkflowInactiveYn);
    });

    (window as any)['ng_legacy_set_agents'] = (sourceIds, sourceNames) => {
      this.onSourceSelect(sourceIds, sourceNames);
    };
  }

  ngOnInit(): void {
    this.titleService.setTitle('Automated Workflows');

    this.frmWorkflows.get('frmControlawWorkflowTypeInd').valueChanges.subscribe(value => {
      this.frmWorkflows.get('provisionalExpiry').disable();
      this.frmWorkflows.get('paymentsDue').disable();
      if (value == 0) {
        this.frmWorkflows.get('frmControlawWorkflowScheduleDays').enable();
        this.frmWorkflows.get('frmControlawWorkflowBeforeAfterInd').enable();
        this.frmWorkflows.get('frmControlawWorkflowTriggerId').enable();
        this.frmWorkflows.get('frmControlawWorkflowTime').enable();
      } else {
        this.frmWorkflows.get('frmControlawWorkflowBeforeAfterInd').disable();
        this.frmWorkflows.get('frmControlawWorkflowTriggerId').disable();
        this.frmWorkflows.get('frmControlawWorkflowTime').disable();
        this.frmWorkflows.get('frmControlawWorkflowScheduleDays').disable();

        if (value == 1) {
          this.frmWorkflows.get('provisionalExpiry').enable();
        } else if (value == 2) {
          this.frmWorkflows.get('paymentsDue').enable();
        }

      }
    });
    this.frmWorkflows.get('frmControlawWorkflowTypeInd').updateValueAndValidity();

    this.refresh();
  }

  getWorkflows(_awWorkflowId: string = null): Observable<Workflow[]> {
    return this.workflowService.getWorkflows(_awWorkflowId);
  }

  getWorkflowTriggers(_awWorkflowTriggerId: string = null): Observable<WorkflowTrigger[]> {
    return this.workflowService.getWorkflowTriggers(_awWorkflowTriggerId);
  }

  openSnackBar(message: string, action: string) {
    this._snackBar.open(message, action, {
      duration: 2000,
    });
  }

  refresh(): void {
    this.loadingSource.next(true);
    this.workflowsLoaded = Promise.resolve(false);
    this.getWorkflowTriggers().subscribe(
      (data: any) => {
        this.workflowTriggers = [];
        data.data.forEach((item: any) => {
          const workflowTrigger: WorkflowTrigger = {
            id: item.awWorkflowTriggerId,
            name: item.awWorkflowTriggerName,
          };
          this.workflowTriggers.push(workflowTrigger)
        });
      }
    );
    this.getWorkflows().subscribe(
      (data: any) => {
        this.workflows = [];
        data.data.forEach((item: any) => {
          if (!item.canEdit) {
            this.canEdit = false;
          }
          const lastRunTimeStamp = moment.tz(item.awWorkflowLastRun,item.awWorkflowLastRunTz).format();
          let text = 'Not run yet';
          let hours = -1;
          if (moment(lastRunTimeStamp).isValid()) {
            text = moment(lastRunTimeStamp).fromNow();
            hours = moment().diff(lastRunTimeStamp, 'hours');
          }

          if (this.workflowTypes[item.awWorkflowTypeInd]) {
            const workflow: Workflow = {
              id: item.awWorkflowId,
              name: item.awWorkflowName,
              workflowType: this.workflowTypes[item.awWorkflowTypeInd].name,
              description: item.awWorkflowDescription,
              inactiveYn: item.awWorkflowInactiveYn,
              lastRunText: text,
              lastRunHours: hours
            };
            this.workflows.push(workflow)
          }
        });
        this.dataSource = new MatTableDataSource(this.workflows);
        this.dataSource.paginator = this.paginator;
        this.workflowsLoaded = Promise.resolve(true);
      },
      (err: any) => console.error(err),
      () => {
        this.loadingSource.next(false);
      }
    );

  }

  ngOnDestroy() { }

  onChangeInactive(value: boolean): void {
    this.workflowInactiveText = value ? 'On' : 'Off';
  }

  generateConfig() {

    if (this.frmWorkflows.get('frmControlawWorkflowTypeInd').value == 1) {
      const config = {
        email: {
          subject: this.frmWorkflows.get('provisionalExpiry').get('emailSubject').value,
          message: this.frmWorkflows.get('provisionalExpiry').get('emailMessage').value
        },
        consolidateEmailYn: this.frmWorkflows.get('provisionalExpiry').get('consolidateEmailYn').value,
        schedule: {
          dayOfExpiry: this.frmWorkflows.get('provisionalExpiry').get('periodGroup').get('dayOfExpiry').value,
          dayBeforeExpiry: this.frmWorkflows.get('provisionalExpiry').get('periodGroup').get('dayBeforeExpiry').value,
          periods: [
            {
              enabled: this.frmWorkflows.get('provisionalExpiry').get('periodGroup').get('period1Yn').value,
              interval: this.frmWorkflows.get('provisionalExpiry').get('period1Interval').value,
            },
            {
              enabled: this.frmWorkflows.get('provisionalExpiry').get('periodGroup').get('period2Yn').value,
              interval: this.frmWorkflows.get('provisionalExpiry').get('period2Interval').value,
            },
            {
              enabled: this.frmWorkflows.get('provisionalExpiry').get('periodGroup').get('period3Yn').value,
              interval: this.frmWorkflows.get('provisionalExpiry').get('period3Interval').value,
            }
          ]
        }
      };

      return config;
    } else if (this.frmWorkflows.get('frmControlawWorkflowTypeInd').value == 2) {
      let source = [];
      if (this.frmWorkflows.get('paymentsDue').get('sourceIds').value != null) {
        source = this.frmWorkflows.get('paymentsDue').get('sourceIds').value;
      }
      const config = {
        email: {
          subject: this.frmWorkflows.get('paymentsDue').get('emailSubject').value,
          message: this.frmWorkflows.get('paymentsDue').get('emailMessage').value
        },
        filters: {
          source: source,
          status: {
            quotation: this.frmWorkflows.get('paymentsDue').get('reservationStatusGroup').get('quotation').value,
            waitlisted: this.frmWorkflows.get('paymentsDue').get('reservationStatusGroup').get('waitlisted').value,
            validProvisional: this.frmWorkflows.get('paymentsDue').get('reservationStatusGroup').get('validProvisional').value,
            expiredProvisional: this.frmWorkflows.get('paymentsDue').get('reservationStatusGroup').get('expiredProvisional').value,
            confirmed: this.frmWorkflows.get('paymentsDue').get('reservationStatusGroup').get('confirmed').value,
            deleted: this.frmWorkflows.get('paymentsDue').get('reservationStatusGroup').get('deleted').value
          },
        },
        attachFoliosYn: this.frmWorkflows.get('paymentsDue').get('attachFoliosYn').value,
        schedule: {
          dayOfPayment: this.frmWorkflows.get('paymentsDue').get('periodGroup').get('dayOfPayment').value,
          periods: [
            {
              enabled: this.frmWorkflows.get('paymentsDue').get('periodGroup').get('period1Yn').value,
              interval: this.frmWorkflows.get('paymentsDue').get('period1Interval').value,
            },
            {
              enabled: this.frmWorkflows.get('paymentsDue').get('periodGroup').get('period2Yn').value,
              interval: this.frmWorkflows.get('paymentsDue').get('period2Interval').value,
            },
            {
              enabled: this.frmWorkflows.get('paymentsDue').get('periodGroup').get('period3Yn').value,
              interval: this.frmWorkflows.get('paymentsDue').get('period3Interval').value,
            }
          ]
        }
      };

      return config;
    } else {
      return {};
    }
  }

  save(): void {
    if (this.frmWorkflows.valid) {
      this.loadingSource.next(true);
      this.workflowService.addWorkflow(
        this.frmWorkflows.get('frmControlawWorkflowName').value,
        this.frmWorkflows.get('frmControlawWorkflowDescription').value,
        this.frmWorkflows.get('frmControlawWorkflowTypeInd').value,
        this.frmWorkflows.get('frmControlawWorkflowInactiveYn').value,
        this.frmWorkflows.get('frmControlawWorkflowScheduleDays').value,
        this.frmWorkflows.get('frmControlawWorkflowBeforeAfterInd').value,
        this.frmWorkflows.get('frmControlawWorkflowTriggerId').value,
        this.frmWorkflows.get('frmControlawWorkflowTime').value,
        this.generateConfig()
      ).subscribe(
        (data: any) => { },
        (err: any) => console.error(err),
        () => {
          this.refresh();
          this.resetForm();
          this.loadingSource.next(false);
          this.openSnackBar('Workflow created', 'Ok')
        }
      );
    }
  }

  update(): void {
    if (this.frmWorkflows.valid) {
      this.loadingSource.next(true);
      this.workflowService.updateWorkflow(
        this.workflowId,
        this.frmWorkflows.get('frmControlawWorkflowName').value,
        this.frmWorkflows.get('frmControlawWorkflowDescription').value,
        this.frmWorkflows.get('frmControlawWorkflowTypeInd').value,
        this.frmWorkflows.get('frmControlawWorkflowInactiveYn').value,
        this.frmWorkflows.get('frmControlawWorkflowScheduleDays').value,
        this.frmWorkflows.get('frmControlawWorkflowBeforeAfterInd').value,
        this.frmWorkflows.get('frmControlawWorkflowTriggerId').value,
        this.frmWorkflows.get('frmControlawWorkflowTime').value,
        this.generateConfig()
      ).subscribe(
        (data: any) => { },
        (err: any) => console.error(err),
        () => {
          this.refresh();
          this.resetForm();
          this.loadingSource.next(false);
          this.openSnackBar('Workflow updated', 'Ok')
        }
      );
    }
  }

  delete(_awWorkflowId: string): void {
    this.workflowService.deleteWorkflow(
      _awWorkflowId
    ).subscribe(
      (data: any) => { },
      (err: any) => console.error(err),
      () => {
        this.refresh();
        this.openSnackBar('Workflow deleted', 'Ok')
      }
    );
  }

  edit(_awWorkflowId: string): void {
    this.loadingSource.next(true);
    this.workflowService.getWorkflows(_awWorkflowId)
    .subscribe(
      (data: any) => {
        const workflow = data.data[0] ?? null;
        if (workflow) {
          this.resetForm();
          this.workflowId = workflow.awWorkflowId;
          this.frmWorkflows.get('frmControlawWorkflowName').setValue(workflow.awWorkflowName);
          this.frmWorkflows.get('frmControlawWorkflowDescription').setValue(workflow.awWorkflowDescription);
          this.frmWorkflows.get('frmControlawWorkflowTypeInd').setValue(workflow.awWorkflowTypeInd);
          const inactiveYn: boolean = workflow.awWorkflowInactiveYn === '0' ? true : false;
          this.frmWorkflows.get('frmControlawWorkflowInactiveYn').setValue(inactiveYn);
          this.frmWorkflows.get('frmControlawWorkflowScheduleDays').setValue(workflow.awWorkflowScheduleDays);
          this.frmWorkflows.get('frmControlawWorkflowBeforeAfterInd').setValue(workflow.awWorkflowBeforeAfterInd);
          this.frmWorkflows.get('frmControlawWorkflowTriggerId').setValue(workflow.awWorkflowTriggerId);
          // account for expected HH:MM time format in ngxTimepicker, but HH:MM:SS format from db/api
          this.frmWorkflows.get('frmControlawWorkflowTime').setValue(workflow.awWorkflowTime.slice(0, 5));

          this.frmWorkflows.get('provisionalExpiry').disable();
          this.frmWorkflows.get('paymentsDue').disable();

          this.legacySelectedSourceNames = 'All';
          this.legacySelectedSourceIds = '';

          if (workflow.awWorkflowTypeInd == '1') {
            this.frmWorkflows.get('provisionalExpiry').enable();
            if (workflow.awWorkflowConfig.email) {
              this.frmWorkflows.get('provisionalExpiry').get('emailSubject').setValue(workflow.awWorkflowConfig.email.subject);
              this.frmWorkflows.get('provisionalExpiry').get('emailMessage').setValue(workflow.awWorkflowConfig.email.message);
            } else {
              this.frmWorkflows.get('provisionalExpiry').get('emailSubject').setValue(workflow.awWorkflowConfig.email.subject);
              this.frmWorkflows.get('provisionalExpiry').get('emailMessage').setValue(workflow.awWorkflowConfig.email.message);
            }

            if (workflow.awWorkflowConfig.consolidateEmailYn) {
              this.frmWorkflows.get('provisionalExpiry').get('consolidateEmailYn').setValue(workflow.awWorkflowConfig.consolidateEmailYn);
            } else {
              this.frmWorkflows.get('provisionalExpiry').get('consolidateEmailYn').setValue('');
            }

            if (workflow.awWorkflowConfig.schedule) {
              this.frmWorkflows.get('provisionalExpiry').get('periodGroup').get('dayOfExpiry').setValue(workflow.awWorkflowConfig.schedule.dayOfExpiry);
              this.frmWorkflows.get('provisionalExpiry').get('periodGroup').get('dayBeforeExpiry').setValue(workflow.awWorkflowConfig.schedule.dayBeforeExpiry);

              this.frmWorkflows.get('provisionalExpiry').get('periodGroup').get('period1Yn').setValue(workflow.awWorkflowConfig.schedule.periods[0].enabled);
              this.frmWorkflows.get('provisionalExpiry').get('periodGroup').get('period2Yn').setValue(workflow.awWorkflowConfig.schedule.periods[1].enabled);
              this.frmWorkflows.get('provisionalExpiry').get('periodGroup').get('period3Yn').setValue(workflow.awWorkflowConfig.schedule.periods[2].enabled);

              this.frmWorkflows.get('provisionalExpiry').get('period1Interval').setValue(workflow.awWorkflowConfig.schedule.periods[0].interval);
              this.frmWorkflows.get('provisionalExpiry').get('period2Interval').setValue(workflow.awWorkflowConfig.schedule.periods[1].interval);
              this.frmWorkflows.get('provisionalExpiry').get('period3Interval').setValue(workflow.awWorkflowConfig.schedule.periods[2].interval);
            } else {
              this.frmWorkflows.get('provisionalExpiry').get('periodGroup').get('periodGroup').get('dayOfExpiry').setValue('');
              this.frmWorkflows.get('provisionalExpiry').get('periodGroup').get('dayBeforeExpiry').setValue('');

              this.frmWorkflows.get('provisionalExpiry').get('periodGroup').get('period1Yn').setValue('');
              this.frmWorkflows.get('provisionalExpiry').get('periodGroup').get('period2Yn').setValue('');
              this.frmWorkflows.get('provisionalExpiry').get('periodGroup').get('period3Yn').setValue('');

              this.frmWorkflows.get('provisionalExpiry').get('period1Interval').setValue(1);
              this.frmWorkflows.get('provisionalExpiry').get('period2Interval').setValue(1);
              this.frmWorkflows.get('provisionalExpiry').get('period3Interval').setValue(1);
            }
          } else if (workflow.awWorkflowTypeInd == '2') {
            this.frmWorkflows.get('paymentsDue').enable();
            if (workflow.awWorkflowConfig.email) {
              this.frmWorkflows.get('paymentsDue').get('emailSubject').setValue(workflow.awWorkflowConfig.email.subject);
              this.frmWorkflows.get('paymentsDue').get('emailMessage').setValue(workflow.awWorkflowConfig.email.message);
            } else {
              this.frmWorkflows.get('paymentsDue').get('emailSubject').setValue(workflow.awWorkflowConfig.email.subject);
              this.frmWorkflows.get('paymentsDue').get('emailMessage').setValue(workflow.awWorkflowConfig.email.message);
            }

            if (workflow.awWorkflowConfig.attachFoliosYn) {
              this.frmWorkflows.get('paymentsDue').get('attachFoliosYn').setValue(workflow.awWorkflowConfig.attachFoliosYn);
            } else {
              this.frmWorkflows.get('paymentsDue').get('attachFoliosYn').setValue('');
            }

            if (workflow.awWorkflowConfig.schedule) {
              this.frmWorkflows.get('paymentsDue').get('periodGroup').get('dayOfPayment').setValue(workflow.awWorkflowConfig.schedule.dayOfPayment);

              this.frmWorkflows.get('paymentsDue').get('periodGroup').get('period1Yn').setValue(workflow.awWorkflowConfig.schedule.periods[0].enabled);
              this.frmWorkflows.get('paymentsDue').get('periodGroup').get('period2Yn').setValue(workflow.awWorkflowConfig.schedule.periods[1].enabled);
              this.frmWorkflows.get('paymentsDue').get('periodGroup').get('period3Yn').setValue(workflow.awWorkflowConfig.schedule.periods[2].enabled);

              this.frmWorkflows.get('paymentsDue').get('period1Interval').setValue(workflow.awWorkflowConfig.schedule.periods[0].interval);
              this.frmWorkflows.get('paymentsDue').get('period2Interval').setValue(workflow.awWorkflowConfig.schedule.periods[1].interval);
              this.frmWorkflows.get('paymentsDue').get('period3Interval').setValue(workflow.awWorkflowConfig.schedule.periods[2].interval);
            } else {
              this.frmWorkflows.get('paymentsDue').get('periodGroup').get('dayOfPayment').setValue('');

              this.frmWorkflows.get('paymentsDue').get('periodGroup').get('period1Yn').setValue('');
              this.frmWorkflows.get('paymentsDue').get('periodGroup').get('period2Yn').setValue('');
              this.frmWorkflows.get('paymentsDue').get('periodGroup').get('period3Yn').setValue('');

              this.frmWorkflows.get('paymentsDue').get('period1Interval').setValue(1);
              this.frmWorkflows.get('paymentsDue').get('period2Interval').setValue(1);
              this.frmWorkflows.get('paymentsDue').get('period3Interval').setValue(1);
            }

            let sourceNames = [];
            let sourceIds = [];
            workflow.awWorkflowConfig.filters.source.forEach(sourceId => {
              let source = workflow.sources.find(source => source.id === sourceId)
              if (source) {
                sourceNames.push(source.name);
                sourceIds.push(source.id);
              }
            });

            if (sourceIds.length > 0) {
              this.legacySelectedSourceNames = sourceNames.join(', ');
              this.legacySelectedSourceIds = sourceIds.join(':');
            }

            this.frmWorkflows.get('paymentsDue').get('sourceIds').setValue(sourceIds);

            this.frmWorkflows.get('paymentsDue').get('reservationStatusGroup').get('quotation').setValue(workflow.awWorkflowConfig.filters.status.quotation);
            this.frmWorkflows.get('paymentsDue').get('reservationStatusGroup').get('waitlisted').setValue(workflow.awWorkflowConfig.filters.status.waitlisted);
            this.frmWorkflows.get('paymentsDue').get('reservationStatusGroup').get('validProvisional').setValue(workflow.awWorkflowConfig.filters.status.validProvisional);
            this.frmWorkflows.get('paymentsDue').get('reservationStatusGroup').get('expiredProvisional').setValue(workflow.awWorkflowConfig.filters.status.expiredProvisional);
            this.frmWorkflows.get('paymentsDue').get('reservationStatusGroup').get('confirmed').setValue(workflow.awWorkflowConfig.filters.status.confirmed);
            this.frmWorkflows.get('paymentsDue').get('reservationStatusGroup').get('deleted').setValue(workflow.awWorkflowConfig.filters.status.deleted);
          }

          this.frmWorkflows.markAsDirty();
        } else {
          const dialogRef = this.dialog.open(ConfirmationDialogComponent, {
            width: '350px',
            data: 'Workflow not found. It may have been deleted. Refresh results?'
          });

          dialogRef.afterClosed().subscribe((result: any) => {
            if(result) {
              this.refresh();
            }
          });
        }
      },
      (err: any) => console.error(err),
      () => {
        this.loadingSource.next(false);
      }
    );
  }

  run(_awWorkflowId: string = null): void {
    if (_awWorkflowId === null) {
      _awWorkflowId = this.workflowId;
    }
    if (_awWorkflowId != null) {
      this.loadingSource.next(true);
      this.workflowService.runWorkflow(
        _awWorkflowId
      ).subscribe(
        (data: any) => {
          this.loadingSource.next(false);
          this.refresh();
          this.openSnackBar('Workflow has been run', 'Ok')
        },
        (err: any) => console.error(err),
        () => { }
      );
    }
  }

  resetForm(): void {
    this.legacySelectedSourceNames = 'All';
    this.legacySelectedSourceIds = '';

    this.formGroupDirective.resetForm(this.getForm().value);
    this.workflowId = null;
  }

  openDialog(awWorkflowId: string): void {
    const dialogRef = this.dialog.open(ConfirmationDialogComponent, {
      width: '350px',
      data: 'Delete this workflow?'
    });

    dialogRef.afterClosed().subscribe((result: any) => {
      if(result) {
        this.delete(awWorkflowId);
        this.resetForm();
      }
    });
  }

  applyFilter(event: Event) {
    const filterValue = event ? (event.target as HTMLInputElement).value : '';
    this.dataSource.filter = filterValue.trim().toLowerCase();
  }

  public hasError (controlName: string, errorName: string) {
    return this.frmWorkflows.controls[controlName].hasError(errorName);
  }

  insertAtCursor(myValue, field) {
    let myField;
    let formField;
    let value;

    let workflowType = this.frmWorkflows.get('frmControlawWorkflowTypeInd').value;

    let workflowFrm;

    if (workflowType == 1) {
      workflowFrm = 'provisionalExpiry';
    } else if (workflowType == 2) {
      workflowFrm = 'paymentsDue';
    }

    if (!workflowFrm) {
      return;
    }


    if (field == 0) {
      myField = this.emailSubjectTemplate.nativeElement as any;
      formField = this.frmWorkflows.get(workflowFrm).get('emailSubject');
    } else {
      myField = this.emailBodyTemplate.nativeElement as any;
      formField = this.frmWorkflows.get(workflowFrm).get('emailMessage');
    }

    myValue = '{{' + myValue + '}}';
    if (myField.selectionStart || myField.selectionStart == '0') {
        var startPos = myField.selectionStart;
        var endPos = myField.selectionEnd;
        value = myField.value.substring(0, startPos)
            + myValue
            + myField.value.substring(endPos, myField.value.length);
    } else {
        value = formField.value += myValue;
    }


    if (formField) {
      formField.setValue(value);
    }
  }

  openPopup(url: string, name?: string, w?: number, h?: number): void {
    if (!name) { name = ''; }
    if (!w) { w = 400; }
    if (!h) { h = 300; }

    const dualScreenLeft = window.screenLeft != undefined ? window.screenLeft : screenLeft;

    const windowWidth = window.innerWidth ? window.innerWidth : document.documentElement.clientWidth ? document.documentElement.clientWidth : screen.width;
    const windowHeight = window.innerHeight ? window.innerHeight : document.documentElement.clientHeight ? document.documentElement.clientHeight : screen.height;

    var systemZoom = windowWidth / window.screen.availWidth;

    const x = ((windowWidth / 2) - (w / 2))/ systemZoom + dualScreenLeft; // X co-ord to center screen horizontally
    const y = ((windowHeight - h)/2); // X co-ord to center screen virtically

    const size = 'fullscreen=0,width=' + w + ',height=' + h + ',left=' + x + ',top=' + y;

    const windowRef = window.open(url, name, size);
  }

  selectSource() {
    const url = [
      "reservation.php?661",
      'source',
      this.legacySelectedSourceIds
    ];

    this.openPopup(url.join("+"), '', 280,360);
  }

  onSourceSelect(sourceIds: string, sourceNames: string) {

    if (sourceIds) {
      this.frmWorkflows.get('paymentsDue').get('sourceIds').setValue(sourceIds.split(':'));
    } else {
      this.frmWorkflows.get('paymentsDue').get('sourceIds').setValue([]);
    }

    this.legacySelectedSourceIds = sourceIds;
    this.legacySelectedSourceNames = sourceNames;
  }

  getForm() {
    return new FormGroup({
      frmControlawWorkflowName: new FormControl('', [Validators.required]),
      frmControlawWorkflowDescription: new FormControl('', []),
      frmControlawWorkflowTypeInd: new FormControl('0', [Validators.required]),
      frmControlawWorkflowInactiveYn: new FormControl('', []),
      frmControlawWorkflowScheduleDays: new FormControl('0', [Validators.required, Validators.min(1), , Validators.pattern(/^[0-9]\d*$/)]),
      frmControlawWorkflowBeforeAfterInd: new FormControl('', [Validators.required, Validators.min(0), Validators.max(1)]),
      frmControlawWorkflowTriggerId: new FormControl('', [Validators.required, Validators.min(1), Validators.max(4)]),
      frmControlawWorkflowTime: new FormControl('', [Validators.required]),

      provisionalExpiry: new FormGroup({
        emailSubject: new FormControl(this.provExpiryEmailSubject, [Validators.required]),
        emailMessage: new FormControl(this.provExpiryEmailMessage, [Validators.required]),
        consolidateEmailYn: new FormControl('', []),

        periodGroup: new FormGroup({
          dayOfExpiry: new FormControl('', []),
          dayBeforeExpiry: new FormControl('', []),
          period1Yn: new FormControl('', []),
          period2Yn: new FormControl('', []),
          period3Yn: new FormControl('', []),
        }, minRequirement(1)),

        period1Interval: new FormControl(1, [Validators.min(1), Validators.required]),
        period2Interval: new FormControl(1, [Validators.min(1), Validators.required]),
        period3Interval: new FormControl(1, [Validators.min(1), Validators.required]),
      }),

      paymentsDue: new FormGroup({
        sourceIds: new FormControl([]),
        statusIds: new FormControl([]),

        emailSubject: new FormControl(this.paymentsDueEmailSubject, [Validators.required]),
        emailMessage: new FormControl(this.paymentsDueEmailMessage, [Validators.required]),
        attachFoliosYn: new FormControl('', []),

        periodGroup: new FormGroup({
          dayOfPayment: new FormControl('', []),
          period1Yn: new FormControl('', []),
          period2Yn: new FormControl('', []),
          period3Yn: new FormControl('', []),
        }, minRequirement(1)),

        period1Interval: new FormControl(1, [Validators.min(1), Validators.required]),
        period2Interval: new FormControl(1, [Validators.min(1), Validators.required]),
        period3Interval: new FormControl(1, [Validators.min(1), Validators.required]),

        reservationStatusGroup: new FormGroup({
          quotation: new FormControl(false, []),
          waitlisted: new FormControl(false, []),
          validProvisional: new FormControl(false, []),
          expiredProvisional: new FormControl(false, []),
          confirmed: new FormControl(false, []),
          deleted: new FormControl(false, []),
        }, minRequirement(1))
      })
    });
  }
}
