import { Component, OnInit, QueryList, ViewChildren, OnDestroy } from '@angular/core';
import { ActivatedRoute, Router } from '@angular/router';
import { Title } from '@angular/platform-browser';
import { ChartContainerComponent } from 'projects/charts/src/app/chart-container/chart-container.component';
import { SubSink } from 'subsink';
import { ChartContainerService } from 'projects/charts/src/app/services/chart-container.service';
import { DashboardService } from './dashboard.service';
import { BehaviorSubject, Observable } from 'rxjs';
import { distinctUntilChanged } from 'rxjs/operators';

@Component({
  selector: 'app-dashboard',
  templateUrl: './dashboard.component.html',
  styleUrls: ['./dashboard.component.scss']
})
export class DashboardComponent implements OnInit, OnDestroy {
  @ViewChildren(ChartContainerComponent) chartContainerComponents: QueryList<ChartContainerComponent>;
  private subsink: SubSink;

  tabs: {id: string; label: string}[] = [];

  activeContainerSource: BehaviorSubject<string>;
  activeContainer$: Observable<string>;
  activeTabSource: BehaviorSubject<number>;
  activeTab$: Observable<number>;

  engagementLabel: string = "Product usage";
  productUsageAllowed: boolean = false;

  chartContainers = [
    {
      label: 'Reservation management',
      id: '11e9d3a1cbb8051b8f6902399910d15e',
      access: true,
    },
    {
      label: 'Sales overview',
      id: '11ea1c3fae309e1a91f502399910d15e',
      access: true,
    }
  ];

  loadedContainer: {[key: number]: boolean};

  constructor(
      private route: ActivatedRoute,
      private router: Router,
      private titleService: Title,
      private dashboardService: DashboardService,
      private chartContainerService: ChartContainerService
    ) {
    this.subsink = new SubSink();
    this.loadedContainer = {};

    this.chartContainers.forEach(container => {
      this.tabs.push({id: container.id, label: container.label});
    });

    this.tabs.push({
      id: 'product_usage',
      label: 'Product Usage'
    });

    this.tabs.push({
      id: 'tourism.today',
      label: 'Tourism.Today'
    });

    this.activeTabSource = new BehaviorSubject<number>(null);
    this.activeTab$ = this.activeTabSource.asObservable();

    this.activeContainerSource = new BehaviorSubject<string>(null);
    this.activeContainer$ = this.activeContainerSource.asObservable();
  }

  ngOnInit() {
    this.subsink.sink = this.route.queryParamMap.subscribe(paramMap => {
      let tab: string;
      if (paramMap.has('tab')) {
        tab = paramMap.get('tab');
      }

      this.changeTab(tab);
    });

    this.subsink.sink = this.activeTabSource.pipe(distinctUntilChanged()).subscribe(tab => {
      if (tab >= 0 && this.tabs[tab]) {
        this.updateTab(this.tabs[tab].id);
      } else {
        this.changeTab();
      }
    });
  }

  ngOnDestroy() {
    this.subsink.unsubscribe();
  }

  onTabChange(index: number) {
    this.activeTabSource.next(index);
  }

  changeTab(id: string = null) {
    let index = this.tabs.findIndex(tab => tab.id === id);

    if (index < 0) {
      const tab = localStorage.getItem('dashboards_last_tab');
      if (tab) {
        this.changeTab(tab);
        return;
      } else {
        index = 0;
      }
    }

    this.activeTabSource.next(index);
  }

  updateTab(id: string) {
    const containerIndex = this.chartContainers.findIndex(container => container.id === id);

    if (containerIndex >= 0) {
      this.loadedContainer[containerIndex] = true;
      this.activeContainerSource.next(id);
    } else {
      this.activeContainerSource.next(null);
    }

    const tab = this.tabs.find(tmpTab => tmpTab.id === id);
    this.setDocumentTitle(tab.label);

    if (containerIndex >= 0 && this.chartContainerComponents) {
      // Tab is chart container
      const container = this.chartContainerComponents.toArray()[containerIndex];
      if (container) {
        container.resize();
      }
    }

    localStorage.setItem('dashboards_last_tab', id);

    this.router.navigate(
      [],
      {
        queryParamsHandling: 'merge',
        relativeTo: this.route,
        queryParams: {
          tab: id
        },
      }
    );
  }

  setDocumentTitle(title = null) {
    if (title) {
      this.titleService.setTitle('Dashboards - ' + title);
    } else {
      this.titleService.setTitle('Dashboards');
    }
  }

}
