import { Injectable } from '@angular/core';
import { RcmApiService } from 'resrequest-angular-common';
import { Observable, BehaviorSubject, of } from 'rxjs';
import { catchError } from 'rxjs/operators';

interface ChartAccess {
  id: string;
  allowed: boolean;
}

interface ContainerAccess {
  id: string;
  hidden: boolean;
  disabled: boolean;
  allowed: boolean;
}

interface Access {
  charts: ChartAccess[];
  containers: ContainerAccess[];
}

@Injectable({
  providedIn: 'root'
})
export class DashboardService {
  chartUrl = '/api/v1/chart';

  /**
   * The source for access$;
   */
  private accessSource: BehaviorSubject<Access>;

  /**
   * The current access level for charts and containers
   */
  access$: Observable<Access>;

  constructor(private api: RcmApiService) {
    this.accessSource = new BehaviorSubject<Access>(null);
    this.access$ = this.accessSource.asObservable();

    this.refreshAccess();
  }

  public async refreshAccess() {
    const access = await this.getAccess().toPromise();
    this.accessSource.next(access);
  }

  public getAccess(): Observable<Access> {
    const url = `${this.chartUrl}/get_access`;

    return this.api.get(url).pipe(
      catchError(this.handleError<Access>('Failed to get dashboard access'))
    );
  }

  isChartContainerAllowed(id: string) {
    const container = this.accessSource.value.containers.find(containerAccess => containerAccess.id === id);

    if (container) {
      return container.allowed;
    } else {
      return false;
    }
  }

  isChartContainerDisabled(id: string) {
    const container = this.accessSource.value.containers.find(containerAccess => containerAccess.id === id);

    if (container) {
      return container.disabled;
    } else {
      return false;
    }
  }

  /**
   * Handle Http operation that failed.
   * Let the app continue.
   * @param operation - name of the operation that failed
   * @param result - optional value to return as the observable result
   */
  private handleError<T>(operation = 'operation', result?: T) {
    return (error: any): Observable<T> => {
      console.error(error); // log to console instead
      return of(result as T);
    };
  }
}
