import { Component, OnInit, ViewChild, TemplateRef, ChangeDetectorRef, ElementRef } from '@angular/core';
import { MatDialog, MatDialogRef } from '@angular/material/dialog';
import { ResvegaService } from '../resvega.service';
import { Observable, BehaviorSubject } from 'rxjs';
import ConfettiGenerator from 'confetti-js';

@Component({
  selector: 'app-about-tourism-today',
  templateUrl: './about-tourism-today.component.html',
  styleUrls: ['./about-tourism-today.component.scss']
})
export class AboutTourismTodayComponent implements OnInit {
  @ViewChild('signUpDialog') private signUpDialog: TemplateRef<any>;
  @ViewChild('signUpCompleteDialog') private signUpCompleteDialog: TemplateRef<any>;
  @ViewChild('signUpFailedDialog') private signUpFailedDialog: TemplateRef<any>;
  @ViewChild('confettiCanvas') private confettiCanvas: ElementRef<any>;
  dialogRef: MatDialogRef<any>;

  includeDataChecked = false;

  /**
   * The source for signUpEnabled$;
   */
  private signUpEnabledSource: BehaviorSubject<boolean>;

  /**
   * Whether the sign up button is enabled
   */
  signUpEnabled$: Observable<boolean>;

  access$ = this.resvega.access$;
  allowed$ = this.resvega.allowed$;

  confetti: ConfettiGenerator;

  constructor(
    private dialog: MatDialog,
    private cd: ChangeDetectorRef,
    private resvega: ResvegaService,
  ) {
    this.signUpEnabledSource = new BehaviorSubject<boolean>(false);
    this.signUpEnabled$ = this.signUpEnabledSource.asObservable();

    this.access$.subscribe(access => {
      if (access === 'free') {
        this.includeDataChecked = true;
      }
    });
  }

  ngOnInit(): void {
  }

  openSignUpModal(version: 'free' | 'pro') {
    this.dialogRef = this.dialog.open(this.signUpDialog, {
      width: '380px',
      data: { version }
    });
  }

  toggleChange() {
    this.cd.detectChanges();
  }

  sendSignUpEmail(version: 'free' | 'pro') {
    this.resvega.sendSignUpEmail(version).subscribe(response => {
      this.dialog.closeAll();

      if (response.success) {
        this.showConfetti();

        this.dialogRef = this.dialog.open(this.signUpCompleteDialog, {
          width: '550px',
          disableClose: true,
          hasBackdrop: false
        });

        this.dialogRef.beforeClosed().subscribe(() => {
          this.hideConfetti();
        });
      } else {
        this.dialogRef = this.dialog.open(this.signUpFailedDialog, {
          width: '380px',
          disableClose: true
        });
      }
    });
  }

  closeDialog() {
    if (this.dialogRef) {
      this.dialogRef.close();
    }
  }

  showConfetti() {
    const confettiSettings = {
      target: this.confettiCanvas.nativeElement,
      max: 100,
      size: 1.5,
      animate: true,
      props:  ['circle', 'square', 'triangle', 'line'],
      colors: [[165, 104, 246], [230, 61, 135], [0, 199, 228], [253, 214, 126]],
      clock: 40,
      rotate: true,
      start_from_edge: true,
      respawn: true
    };

    this.confetti = new ConfettiGenerator(confettiSettings);
    this.confettiCanvas.nativeElement.style.display = 'unset';
    this.confetti.render();
  }

  hideConfetti() {
    if (this.confetti) {
      this.confetti.clear();
      this.confettiCanvas.nativeElement.style.display = 'none';
    }
  }
}
