import { Injectable, OnDestroy } from '@angular/core';
import { RcmApiService } from 'resrequest-angular-common';
import { map, catchError } from 'rxjs/operators';
import { Observable, BehaviorSubject, of } from 'rxjs';
import { ChartSummary } from 'projects/charts/src/app/core/chart-summary';
import { AuthorisationService } from 'src/app/shared/services/authorisation/authorisation.service';
import { SubSink } from 'subsink';

interface Container {
  id: string;
  version: string;
  reportId: string;
  summary: ChartSummary;
}

interface AccessResponse {
  allowed: boolean;
  access: 0 | 1 | 2;
  containers: Container[];
}

interface Report {
  id: string;
  reportType: string;
  name: string;
  webUrl: string;
  embedUrl: string;
  isFromPbix: boolean;
  isOwnedByMe: boolean;
  datasetId: string;
}

interface ReportsResponse {
  embedToken: string;
  reports: Report[];
}

@Injectable({
  providedIn: 'root'
})
export class ResvegaService implements OnDestroy {
  private subsink: SubSink;

  private resvegaUrl = '/api/v1/resvega';
  /**
   * The source for containers$;
   */
  private containersSource: BehaviorSubject<Container[]>;

  /**
   * All ResVega containers for each version
   */
  containers$: Observable<Container[]>;

  /**
   * The source for reports$;
   */
  private reportsSource: BehaviorSubject<Report[]>;

  /**
   * Power BI reports
   */
  reports$: Observable<Report[]>;

  /**
   * The source for token$;
   */
  private tokenSource: BehaviorSubject<string>;

  /**
   * Power BI reports embed token
   */
  token$: Observable<string>;


  /**
   * The source for allowed$;
   */
  private allowedSource: BehaviorSubject<boolean>;

  /**
   * Whether ResVega is allowed.
   */
  allowed$: Observable<boolean>;

  /**
   * The source for access$;
   */
  private accessSource: BehaviorSubject<'none' | 'free' | 'pro'>;

  /**
   * The current ResVega access level
   */
  access$: Observable<'none' | 'free' | 'pro'>;

  constructor(
    private api: RcmApiService,
    private authorisationService: AuthorisationService,
  ) {
    this.subsink = new SubSink();

    this.containersSource = new BehaviorSubject<Container[]>(null);
    this.containers$ = this.containersSource.asObservable();
    this.reportsSource = new BehaviorSubject<any[]>(null);
    this.reports$ = this.reportsSource.asObservable();
    this.tokenSource = new BehaviorSubject<string>(null);
    this.token$ = this.tokenSource.asObservable();
    this.allowedSource = new BehaviorSubject<boolean>(null);
    this.allowed$ = this.allowedSource.asObservable();
    this.accessSource = new BehaviorSubject<'none' | 'free' | 'pro'>(null);
    this.access$ = this.accessSource.asObservable();

    this.subsink.sink = this.authorisationService.accessLoaded$.subscribe(loaded => {
      if (loaded) {
        this.updateAccess();
      }
    });
  }

  ngOnDestroy() {
    this.subsink.unsubscribe();
  }

  updateAccess() {
    this.getAccess().subscribe(access => {
      this.accessSource.next(access.access);
      this.containersSource.next(access.containers);
      this.allowedSource.next(access.allowed);
    });

    if (this.authorisationService.isPrimaryEnvironmentWeb()) {
      let refreshTokenTimer = null;
      this.getReports().subscribe(reportsResponse => {
        this.reportsSource.next(reportsResponse.reports);
        this.tokenSource.next(reportsResponse.embedToken);
      });

      this.token$.subscribe((token) => {
        clearTimeout(refreshTokenTimer);

        if (token) {
          // Refresh token every 55 minutes
          refreshTokenTimer = setTimeout(() => this.refreshToken(), 3300000);
        }
      });
    }
  }

  getAccess(): Observable<{ allowed: boolean, access: 'none' | 'free' | 'pro', containers: Container[] }> {
    const url = `${this.resvegaUrl}/get_access`;

    return this.api.get(url).pipe(map(
      (response: AccessResponse) => {
        let access: 'none' | 'free' | 'pro' = 'none';

        if (response.access === 2) {
          access = 'pro';
        } else if (response.access === 1) {
          access = 'free';
        }

        return { allowed: response.allowed, access, containers: response.containers };
      }));
  }

  getReports(): Observable<ReportsResponse> {
    const url = `${this.resvegaUrl}/get_reports`;
    return this.api.get(url).pipe(
      catchError(error => {
        console.log(error);
        return of(null);
      })
    );
  }

  getContainerIds(): string[] {
    const containerAccess = this.containersSource.value;
    const containers = [];
    containerAccess.forEach(container => {
      containers.push(container.id);
    });

    return containers;
  }

  public sendSignUpEmail(version: 'free' | 'pro'): Observable<{ success: boolean }> {
    const url = `${this.resvegaUrl}/sign_up`;

    return this.api.post(url, { version }).pipe(
      catchError(error => {
        console.log(error);
        return of(null);
      })
    );
  }

  /**
   * Update the Power BI embed token
   */
  refreshToken() {
    if (this.authorisationService.isPrimaryEnvironmentWeb()) {
      this.getReports().subscribe(reportsResponse => {
        this.tokenSource.next(reportsResponse.embedToken);
      });
    }
  }
}
