import { Component, OnInit, OnDestroy } from '@angular/core';
import { ResvegaService } from './resvega.service';
import { DashboardService } from '../dashboard.service';
import { combineLatest, BehaviorSubject, Observable } from 'rxjs';
import { SubSink } from 'subsink';
import { ActivatedRoute, Router } from '@angular/router';
import { distinctUntilChanged, take } from 'rxjs/operators';

@Component({
  selector: 'app-tourism-today',
  templateUrl: './tourism-today.component.html',
  styleUrls: ['./tourism-today.component.scss']
})
export class TourismTodayComponent implements OnInit, OnDestroy {
  subsink: SubSink;

  staticTabs = [
    {
      id: 'about'
    }
  ];

  tabs = [];

  activeTabSource: BehaviorSubject<number>;
  activeTab$: Observable<number>;

  access$ = this.resvega.access$;

  containersSource: BehaviorSubject<any>;
  containers$: Observable<any>;

  constructor(
    private route: ActivatedRoute,
    private router: Router,
    private resvega: ResvegaService,
    private dashboardService: DashboardService
  ) {
    this.subsink = new SubSink();

    this.activeTabSource = new BehaviorSubject<number>(null);
    this.activeTab$ = this.activeTabSource.asObservable();
    this.containersSource = new BehaviorSubject<any>([]);
    this.containers$ = this.containersSource.asObservable();

    combineLatest([
      this.resvega.containers$,
      this.dashboardService.access$
    ]).subscribe(([resvegaContainers, access]) => {
      if (resvegaContainers && access) {
        this.tabs = this.staticTabs;
        const containers = [];
        resvegaContainers.forEach(container => {
          const id = container.id;
          const allowed = this.dashboardService.isChartContainerAllowed(id);
          const disabled = this.dashboardService.isChartContainerDisabled(id);
          const display = allowed;

          containers.push({
            id,
            reportId: container.reportId,
            allowed,
            disabled,
            display,
            summary: container.summary
          });

          this.tabs.push({id});
        });

        this.containersSource.next(containers);

        this.route.queryParamMap.pipe(take(1)).subscribe(paramMap => {
          let tab: string;
          if (paramMap.has('tt_tab')) {
            tab = paramMap.get('tt_tab');
          }

          this.changeTab(tab);
        });
      }
    });
  }

  ngOnInit(): void {
    this.subsink.sink = this.route.queryParamMap.subscribe(paramMap => {
      let tab: string;
      if (paramMap.has('tt_tab')) {
        tab = paramMap.get('tt_tab');
      }

      this.changeTab(tab);
    });

    this.subsink.sink = this.activeTabSource.pipe(distinctUntilChanged()).subscribe(tab => {
      if (this.tabs[tab]) {
        this.updateTab(this.tabs[tab].id);
      } else {
        this.changeTab(null);
      }
    });
  }

  ngOnDestroy() {
    this.subsink.unsubscribe();
  }

  onTabChange(index: number) {
    this.activeTabSource.next(index);
  }

  changeTab(id: string = null) {
    if (id === null) {
      const tab = localStorage.getItem('resvega_last_tab');
      if (tab) {
        this.changeTab(tab);
        return;
      }
    }

    let index = this.tabs.findIndex(tab => tab.id === id);

    if (index < 0) {
      index = 0;
    }

    this.activeTabSource.next(index);
  }

  updateTab(id: string) {
    const tab = this.tabs.find(tmpTab => tmpTab.id === id);

    localStorage.setItem('resvega_last_tab', id);

    if (tab) {
      this.router.navigate(
        [],
        {
          queryParamsHandling: 'merge',
          relativeTo: this.route,
          queryParams: {
            tt_tab: id
          },
        }
      );
    }
  }

}
