import { Component, OnInit, Input } from '@angular/core';
import { RoomService } from '../../services/room/room.service';
import { BehaviorSubject, Observable, combineLatest } from 'rxjs';
import { SubSink } from 'subsink';
import { map, first } from 'rxjs/operators';
import { CalendarItem } from '../../core/calendar/calendar-item';
import * as moment from 'moment';
import { FrontOfHouseService } from '../../services/front-of-house/front-of-house.service';
import { PortalService } from 'src/app/shared/components/portal';
import { CalendarAccessService } from '../../services/access/calendar-access.service';
import { LegacyService } from '../../services/legacy/legacy.service';

@Component({
  selector: 'app-bulk-check-in-out',
  templateUrl: './bulk-check-in-out.component.html',
  styleUrls: ['./bulk-check-in-out.component.scss'],
  providers: [
    LegacyService,
    FrontOfHouseService
  ]
})
export class BulkCheckInOutComponent implements OnInit {
  @Input() mode: 'in'|'out';
  total$: Observable<number>;
  items$: Observable<CalendarItem[]>;
  showSearchBorder: boolean;
  searchTerm$: BehaviorSubject<string>;
  private subsink: SubSink;
  private selectedItemsSource: BehaviorSubject<number[]>;
  selectedItems$: Observable<number[]>;
  totalSelected$: Observable<number>;
  checkoutDate: moment.Moment;
  canDateCheckouts$ = this.accessService.canDateCheckouts$;

  constructor(
    private roomService: RoomService,
    private portalService: PortalService,
    private frontOfHouseService: FrontOfHouseService,
    private accessService: CalendarAccessService
  ) {
    this.mode = 'in';
    this.showSearchBorder = false;
    this.subsink = new SubSink();
    this.searchTerm$ = new BehaviorSubject<string>('');
    this.selectedItemsSource = new BehaviorSubject<number[]>([]);
    this.selectedItems$ = this.selectedItemsSource.asObservable();
    this.totalSelected$ = this.selectedItems$.pipe(map(items => items.length));
    this.checkoutDate = moment();
  }

  ngOnInit() {
    if (this.mode === 'in') {
      this.total$ = this.roomService.checkInTotal$;
      this.items$ = this.roomService.checkInItems$;
    } else {
      this.total$ = this.roomService.checkOutTotal$;
      this.items$ = this.roomService.checkOutItems$;
    }

    this.subsink.sink = this.items$.subscribe(() => {
      this.selectedItemsSource.next([]);
    });
  }

  searchFocused(focused: boolean): void {
    this.showSearchBorder = focused;
  }

  onSearchTermChanged(term: string): void {
    this.searchTerm$.next(term);
  }

  async onAllChanged(selected: boolean): Promise<void> {
    let items = [...(await this.selectedItems$.pipe(first()).toPromise())];
    const visibleIndexes = await this.visibleIndexes().pipe(first()).toPromise();

    if (selected) {
      visibleIndexes.forEach(index => {
        if (!items.includes(index)) {
          items.push(index);
        }
      });
    } else {
      visibleIndexes.forEach(index => {
        if (items.includes(index)) {
          items.splice(items.indexOf(index), 1);
        }
      });
    }

    this.selectedItemsSource.next(items);
  }

  isSelected(index: number): Observable<boolean> {
    return this.selectedItems$.pipe(
      map(items => {
        return items.includes(index);
      })
    );
  }

  async toggleSelection(index: number): Promise<void> {
    let items = [...(await this.selectedItems$.pipe(first()).toPromise())];

    if (items.includes(index)) {
      items.splice(items.indexOf(index), 1);
    } else {
      items.push(index);
    }

    this.selectedItemsSource.next(items);
  }

  private visibleIndexes(): Observable<number[]> {
    return combineLatest([this.items$, this.searchTerm$]).pipe(
      map(([items, term]) => {
        const termLower = term.trim().toLocaleLowerCase();

        const matches = items.map((item, index) => {

          const id = item.data.reservationId.toLocaleLowerCase();

          if (id.includes(termLower)) {
            return {matches: true, index};
          }

          const name = item.data.name.toLocaleLowerCase();

          if (name.includes(termLower)) {
            return {matches: true, index};
          }

          const propertyName = item.data.propertyName.toLocaleLowerCase();

          if (propertyName.includes(termLower)) {
            return {matches: true, index};
          }

          const accommodationName = item.data.realAccommName.toLocaleLowerCase();

          if (accommodationName.includes(termLower)) {
            return {matches: true, index};
          }

          const roomName = item.data.roomName.toLocaleLowerCase();

          if (roomName.includes(termLower)) {
            return {matches: true, index};
          }

          const display = `${item.data.propertyName} - ${item.data.realAccommName} - ${item.data.roomName}`.toLocaleLowerCase();

          if (display.includes(termLower)) {
            return {matches: true, index};
          }

          return {matches: false, index};
        });

        return matches.filter(match => match.matches).map(match => match.index);
      })
    );
  }

  isVisible(index: number): Observable<boolean> {
    return this.visibleIndexes().pipe(
      map(indexes => indexes.includes(index))
    );
  }

  async confirmAction(): Promise<void> {
    const items = await this.items$.pipe(first()).toPromise();
    const selectedItems = await this.selectedItems$.pipe(first()).toPromise();
    const actionItems = selectedItems.map(item => items[item]);

    if (this.mode === 'in') {
      this.checkIn(actionItems);
    } else {
      this.checkOut(actionItems);
    }
  }

  async checkOut(items: CalendarItem[]): Promise<void> {
    const canDateCheckout = await this.canDateCheckouts$.pipe(first()).toPromise();
    let checkOutDate = null;

    if (canDateCheckout) {
      checkOutDate = this.checkoutDate.format("YYYY-MM-DD HH:mm:ss");
    }

    this.portalService.onShowLoading(true);
    this.frontOfHouseService.checkOut(items, checkOutDate).then(() => {
      this.portalService.onShowLoading(false);
      this.selectedItemsSource.next([]);
    });
  }

  checkIn(items: CalendarItem[]): void {
    this.portalService.onShowLoading(true);
    this.frontOfHouseService.checkIn(items).then(() => {
      this.portalService.onShowLoading(false);
      this.selectedItemsSource.next([]);
    });
  }
}
