import { Component, OnInit, EventEmitter } from '@angular/core';
import { CalendarService } from '../../services/calendar/calendar.service';
import { first, map } from 'rxjs/operators';
import { Observable, combineLatest } from 'rxjs';

@Component({
  selector: 'app-conflict-info',
  templateUrl: './conflict-info.component.html',
  styleUrls: ['./conflict-info.component.scss']
})
export class ConflictInfoComponent implements OnInit {
  conflicts$ = this.calendarService.conflicts$;
  conflictIndex$ = this.calendarService.conflictIndex$;
  conflictItemIndex$ = this.calendarService.conflictItemIndex$;
  totalConflicts$ = this.calendarService.totalConflicts$;
  totalConflictItems$ = this.calendarService.totalConflictItems$;
  conflictItem$ = this.calendarService.selectedConflictItem$;
  isConflictItemActive$: Observable<boolean>;

  constructor(private calendarService: CalendarService) {
  }

  ngOnInit(): void {
    const activeItem$ = this.calendarService.activeItem$;
    this.isConflictItemActive$ = combineLatest([activeItem$, this.conflictItem$]).pipe(
      map(([activeItem, conflictItem]) => {
        return activeItem && conflictItem && activeItem.id === conflictItem.id;
      })
    )
  }

  async dismiss(): Promise<void> {
    const isConflictItemActive = await this.isConflictItemActive$.pipe(first()).toPromise();

    if (isConflictItemActive) {
      this.calendarService.cancelItemInteraction();
    }

    this.calendarService.onConflictsDismissedChange(true);
  }

  nextConflict(): void {
    this.calendarService.onNextConflict();
  }

  previousConflict(): void {
    this.calendarService.onPreviousConflict();
  }

  nextConflictItem(): void {
    this.calendarService.onNextConflictItem();
  }

  previousConflictItem(): void {
    this.calendarService.onPreviousConflictItem();
  }

  async unallocateItem(): Promise<void> {
    const item = await this.conflictItem$.pipe(first()).toPromise();
    this.calendarService.unallocateItem(item);
  }
}
