import { Component, OnInit, Input, ChangeDetectionStrategy, ChangeDetectorRef, OnDestroy } from '@angular/core';
import { CalendarDate } from '../../core/calendar/calendar-date';
import { ReportService } from '../../services/report/report.service';
import { FilterService, AccommodationMap } from '../../services/filter/filter.service';
import * as moment from 'moment';
import { CalendarDateBuilder } from '../../core/calendar/calendar-date-builder';
import { CalendarService } from '../../services/calendar/calendar.service';
import { SubSink } from 'subsink';
import { Observable, combineLatest } from 'rxjs';
import { tap, map } from 'rxjs/operators';

@Component({
  selector: 'app-date-header',
  templateUrl: './date-header.component.html',
  styleUrls: ['./date-header.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush
})
export class DateHeaderComponent implements OnInit, OnDestroy {
  @Input() date: CalendarDate;

  menuOpen = false;
  /**
   * Whether multiple properties have been selected
   */
  multipleProperties = false;

  /**
   * Selected accommodations
   */
  private accommodations: string[];
  /**
   * Selected properties
   */
  private properties: string[];

  /**
   * Whether this date is the active conflict date.
   */
  isActiveConflictDate$: Observable<boolean>;

  private subsink: SubSink;

  constructor(
    private report: ReportService,
    private filterService: FilterService,
    private calendarService: CalendarService,
    private cd: ChangeDetectorRef
  ) {
    this.subsink = new SubSink();
    this.subsink.sink = this.filterService.accommodations$.subscribe(accommodations => this.accommodations = accommodations);
    this.subsink.sink = this.filterService.accommodationMap$.subscribe((accommMap: AccommodationMap) => {
      const properties = [];
      for (const value in accommMap) {
        if (this.accommodations.indexOf(accommMap[value].accommodation.id) >= 0) {
          if (properties.indexOf(accommMap[value].property.id) < 0) {
            properties.push(accommMap[value].property.id);
          }
        }
      }

      this.properties = properties;
      if (properties.length > 1) {
        this.multipleProperties = true;
      } else {
        this.multipleProperties = false;
      }
    });
  }

  ngOnDestroy(): void {
    this.subsink.unsubscribe();
  }

  ngOnInit() {
    const selectedConflict$ = this.calendarService.selectedConflict$;
    const conflictsDismissed$ = this.calendarService.conflictsDismissed$;

    this.isActiveConflictDate$ = combineLatest([selectedConflict$, conflictsDismissed$]).pipe(
      map(([conflict, dismissed]) => {
        if (conflict && !dismissed) {
          return conflict.date === this.date.date;
        } else {
          return false;
        }
      }),
      tap(() => {
        this.cd.markForCheck();
      })
    );
  }

  onMenuOpen() {
    this.menuOpen = true;
  }

  onMenuClose() {
    this.menuOpen = false;
  }

  daySheet() {
    this.report.daySheet(this.properties[0], moment(this.date.date, CalendarDateBuilder.DATE_FORMAT));
  }

  paxReport() {
    this.report.paxReport('day', this.accommodations, moment(this.date.date, CalendarDateBuilder.DATE_FORMAT));
  }

  extrasReport() {
    this.report.extrasReport(this.properties[0], moment(this.date.date, CalendarDateBuilder.DATE_FORMAT));
  }

  foliosReport(type: string) {
    this.report.foliosReport(type, this.properties[0], moment(this.date.date, CalendarDateBuilder.DATE_FORMAT));
  }
}
