import { Component, OnInit, Input, ChangeDetectionStrategy, ChangeDetectorRef, OnDestroy, ViewChild, TemplateRef } from '@angular/core';
import { Property } from '../../core/room/property';
import { CalendarService } from '../../services/calendar/calendar.service';
import { Observable, of, BehaviorSubject, combineLatest } from 'rxjs';
import { CalendarGrid } from '../../core/calendar/calendar-grid';
import { CalendarItem } from '../../core/calendar/calendar-item';
import { map, first, switchMap } from 'rxjs/operators';
import { RoomDropZone } from '../../core/calendar/room-drop-zone';
import { CdkDragDrop } from '@angular/cdk/drag-drop';
import { CalendarSortOrder } from '../../core/filters/calendar-sort-order';
import { SubSink } from 'subsink';
import { RoomService } from '../../services/room/room.service';
import { CalendarBlockItem } from '../../core/calendar/calendar-block-item';
import { SidebarService } from 'src/app/shared/services/sidebar/sidebar.service';
import { Sidebar } from 'src/app/shared/services/sidebar/sidebar';
import { CalendarAccessService } from '../../services/access/calendar-access.service';

@Component({
  selector: 'app-property',
  templateUrl: './property.component.html',
  styleUrls: ['./property.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush
})
export class PropertyComponent implements OnInit, OnDestroy {
  private subs: SubSink;
  calendarGrid: CalendarGrid;
  @Input() property: Property;
  items$: Observable<CalendarItem[]>;
  blockItems$: Observable<CalendarBlockItem[]>;
  roomDropZones$: Observable<RoomDropZone[]>;
  roomFirstColumnDisplay: 'accommodation' | 'room';
  roomSecondColumnDisplay: 'accommodation' | 'room';
  roomFirstColumnGridColumn: string;
  roomSecondColumnGridColumn: string;
  roomGridRows: {[key: string]: string};
  startColumnByDate: {[key: string]: number};
  midColumnByDate: {[key: string]: number};
  itemGridColumnById: {[key: string]: string};
  blockItemGridColumnById: {[key: string]: string};
  dropZonesGridColumnById: {[key: string]: string};
  columnBlockGridRow: string;
  @ViewChild('propInfoSidenav') propInfoSidenav: TemplateRef<any>;
  propId:string;
  propSideNavtitle: string;
  canUpdate$ = this.accessService.canUpdate$;
  activeItem$ = this.calendar.activeItem$;
  private dragging$: BehaviorSubject<boolean>;
  showDropZones$: Observable<boolean>;

  constructor(
    public calendar: CalendarService,
    private roomService: RoomService,
    private sidebarService: SidebarService,
    private accessService: CalendarAccessService,
    private cd: ChangeDetectorRef) {
    this.calendarGrid = new CalendarGrid(calendar);
    this.subs = new SubSink();
    this.dragging$ = new BehaviorSubject<boolean>(false);
  }

  ngOnInit() {
    this.updateRoomColumnInfo();
    this.updateRoomGridRows();
    this.updateDateInfo();
    this.subs.sink = this.calendar.properties$.subscribe(() => {
      this.updateRoomGridRows();
      this.columnBlockGridRow = this.gridRowForColumnBlock();
    });
    this.items$ = this.items();
    this.subs.sink = this.items$.subscribe(items => {
      this.updateItemsGridColumn(items);
    });
    this.blockItems$ = this.blockItems();
    this.subs.sink = this.blockItems$.subscribe(items => {
      this.updateBlockItemsGridColumn(items);
    });
    this.roomDropZones$ = this.roomDropZones();
    this.subs.sink = this.roomDropZones$.subscribe(zones => {
      this.updateDropZonesGridColumn(zones);
    });
    this.columnBlockGridRow = this.gridRowForColumnBlock();
    this.listenForRoomingChanges();
    this.showDropZones$ = combineLatest([this.canUpdate$, this.activeItem$, this.dragging$]).pipe(
      map(([canUpdate, activeItem, dragging]) => {
        return canUpdate && (dragging || activeItem !== null);
      })
    );
  }

  ngOnDestroy() {
    this.subs.unsubscribe();
  }

  /**
   * Updates the properties required for the room headers to render
   * properly.
   */
  private updateRoomColumnInfo(): void {
    this.subs.sink = this.calendar.sortBy$.subscribe(sortBy => {
      if (sortBy === CalendarSortOrder.Accommmodation) {
        this.roomFirstColumnDisplay = 'accommodation';
        this.roomSecondColumnDisplay = 'room';
        this.roomFirstColumnGridColumn = '1 / span 4';
        this.roomSecondColumnGridColumn = '5 / span 2';
      } else {
        this.roomFirstColumnDisplay = 'room';
        this.roomSecondColumnDisplay = 'accommodation';
        this.roomFirstColumnGridColumn = '1 / span 2';
        this.roomSecondColumnGridColumn = '3 / span 4';
      }
    });
  }

  /**
   * Update the grid row properties for all the rooms.
   */
  private updateRoomGridRows(): void {
    this.roomGridRows = {};
    this.property.rooms.forEach(room => {
      this.subs.sink = this.calendarGrid.gridRowByRoom(room).subscribe(gridRow => {
        this.roomGridRows[room.id] = gridRow;
      });
    });
  }

  /**
   * Update the date column caches for the current period.
   */
  private updateDateInfo(): void {
    this.startColumnByDate = {};
    this.midColumnByDate = {};
    this.subs.sink = this.calendar.dates$.pipe(
      switchMap(dates => of(dates))
    ).subscribe(dates => {
      dates.forEach(date => {
        this.calendar.startColumnForDate(date.date).pipe(first()).subscribe(column => {
          this.startColumnByDate[date.date] = column;
        });

        this.calendar.midColumnForDate(date.date).pipe(first()).subscribe(column => {
          this.midColumnByDate[date.date] = column;
        });
      });
    });
  }

  /**
   * Retrieves the items for this property.
   */
  private items(): Observable<CalendarItem[]> {
    return this.calendar.items$.pipe(
      map(items => {
        return items.filter(item => item.propertyId === this.property.id);
      })
    );
  }

  /**
   * Retrieves the block items for this property.
   */
  private blockItems(): Observable<CalendarBlockItem[]> {
    return this.calendar.blockItems$.pipe(
      map(items => {
        return items.filter(item => item.propertyId === this.property.id);
      })
    );
  }

  /**
   * Update the grid column placements for all the items.
   * @param items The items for this property.
   */
  private updateItemsGridColumn(items: CalendarItem[]): void {
    this.itemGridColumnById = {};

    items.forEach(item => {
      let span = item.intersection.days * 2;

      if (item.intersection.overflowStart) {
        // Add half a day because it fills the start portion of the day too.
        span += 1;
      }

      if (item.intersection.overflowEnd) {
        // Add half a day because it fills the end portion of the day too.
        span += 1;
      }

      if (
        item.intersection.singleDay
        && !item.intersection.overflowEnd
      ) {
        // Add half a day because it is only a single day and still needs
        // to be visible.
        span += 1;
      }

      if (item.intersection.overflowStart) {
        this.calendarGrid.gridStartColumnByRawDate(item.intersection.from, span).pipe(
          first()
        ).subscribe(gridColumn => {
          this.itemGridColumnById[item.id] = gridColumn;
        });
      } else {
        this.calendarGrid.gridMidColumnByRawDate(item.intersection.from, span).pipe(
          first()
        ).subscribe(gridColumn => {
          this.itemGridColumnById[item.id] = gridColumn;
        });
      }
    });
  }

  /**
   * Update the grid column placements for all the block items.
   * @param items The block items for this property.
   */
  private updateBlockItemsGridColumn(items: CalendarBlockItem[]): void {
    this.blockItemGridColumnById = {};

    items.forEach(item => {
      let span = item.intersection.days * 2;

      if (item.intersection.overflowStart) {
        // Add half a day because it fills the start portion of the day too.
        span += 1;
      }

      if (item.intersection.overflowEnd) {
        // Add half a day because it fills the end portion of the day too.
        span += 1;
      }

      if (item.intersection.overflowStart) {
        this.calendarGrid.gridStartColumnByRawDate(item.intersection.from, span).pipe(
          first()
        ).subscribe(gridColumn => {
          this.blockItemGridColumnById[item.id] = gridColumn;
        });
      } else {
        this.calendarGrid.gridMidColumnByRawDate(item.intersection.from, span).pipe(
          first()
        ).subscribe(gridColumn => {
          this.blockItemGridColumnById[item.id] = gridColumn;
        });
      }
    });
  }

  /**
   * Get's all the possible drop zones for this property.
   */
  private roomDropZones(): Observable<RoomDropZone[]> {
    return this.calendar.roomDropZones$.pipe(
      map(zones => {
        return zones.filter(zone => zone.propertyId === this.property.id);
      })
    );
  }

  /**
   * Update the grid column placements for all the drop zones.
   * @param zones The drop zones for this property.
   */
  private updateDropZonesGridColumn(zones: RoomDropZone[]): void {
    this.dropZonesGridColumnById = {};

    zones.forEach(zone => {
      let span = zone.intersection.days * 2;

      if (zone.intersection.overflowStart) {
        // Add half a day because it fills the start portion of the day too.
        span += 1;
      }

      if (zone.intersection.overflowEnd) {
        // Add half a day because it fills the end portion of the day too.
        span += 1;
      }

      if (
        zone.intersection.singleDay
        && !zone.intersection.overflowEnd
      ) {
        // Add half a day because it is only a single day and still needs
        // to be visible.
        span += 1;
      }

      if (zone.intersection.overflowStart) {
        this.calendarGrid.gridStartColumnByRawDate(zone.intersection.from, span).pipe(
          first()
        ).subscribe(gridColumn => {
          this.dropZonesGridColumnById[zone.id] = gridColumn;
        });
      } else {
        this.calendarGrid.gridMidColumnByRawDate(zone.intersection.from, span).pipe(
          first()
        ).subscribe(gridColumn => {
          this.dropZonesGridColumnById[zone.id] = gridColumn;
        });
      }
    });
  }

  /**
   * Determines the grid row property for the block in the navigate
   * next column.
   */
  private gridRowForColumnBlock(): string {
    return '1 / span ' + this.property.rooms.length;
  }

  /**
   * Predicate function to prevent calendar items from being able to
   * be dropped inside each other.
   */
  noEnter(): boolean {
    return false;
  }

  /**
   * Called when an item is being dragged.
   */
  dragStarted(): void {
    this.dragging$.next(true);
  }

  /**
   * Called when an item has finished being dragged.
   */
  dragEnded(): void {
    this.dragging$.next(false);
  }

  /**
   * Called when an item was dropped on a drop zone.
   * @param zone The zone where the item was dropped.
   * @param event The drage and drop event.
   */
  drop(zone: RoomDropZone, event: CdkDragDrop<CalendarItem[]>): void {
    this.calendar.itemDropped(zone, event.item.data);
  }

  /**
   * Listens for rooming changes to trigger change detection.
   */
  async listenForRoomingChanges() {
    const items = await this.items$.pipe(first()).toPromise();
    this.subs.sink = this.roomService.roomingChanges$.subscribe(() => {
      this.updateItemsGridColumn(items);
      this.cd.markForCheck();
    });
  }

  public togglePropInfoSidenav(id: string, title: string): void {
    this.propId = id;
    this.propSideNavtitle = title;
    const sidebar: Sidebar = {
      id: 'prop-info-sidenav',
      template: this.propInfoSidenav
    };
    this.sidebarService.showSidebar(sidebar);
  }

  async dropZoneClicked(zone: RoomDropZone): Promise<void> {
    const canUpdate = await this.canUpdate$.pipe(first()).toPromise();
    const item = await this.activeItem$.pipe(first()).toPromise();

    if (canUpdate && item && item.canDrag) {
      this.calendar.itemDropped(zone, item);
      this.calendar.findDropZones(item);
    }
  }

  findDropZones(item: CalendarItem): void {
    this.calendar.findDropZones(item);
  }
}
