import {
  Component,
  OnInit,
  Input,
  ChangeDetectionStrategy,
  HostListener,
  ElementRef,
  HostBinding,
  ViewChild,
  TemplateRef,
  ChangeDetectorRef,
  OnDestroy
} from '@angular/core';
import { CalendarItem } from '../../core/calendar/calendar-item';
import { CalendarService } from '../../services/calendar/calendar.service';
import { Observable, combineLatest } from 'rxjs';
import { SidebarService } from 'src/app/shared/services/sidebar/sidebar.service';
import { Sidebar } from 'src/app/shared/services/sidebar/sidebar';
import { map, distinctUntilChanged, first } from 'rxjs/operators';
import { FilterService } from '../../services/filter/filter.service';
import { RoomService } from '../../services/room/room.service';
import { SubSink } from 'subsink';
import { environment } from 'src/environments/environment';

@Component({
  selector: 'app-reservation-item',
  templateUrl: './reservation-item.component.html',
  styleUrls: ['./reservation-item.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush
})
export class ReservationItemComponent implements OnInit, OnDestroy {
  @Input() item: CalendarItem;
  @ViewChild('reservationPortal') reservationPortal: TemplateRef<any>;
  zIndexInternal: string;

  isFocused$: Observable<boolean>;
  isFound$: Observable<boolean>;
  isActive$: Observable<boolean>;
  borderColour$: Observable<string>;
  reservationColour$: Observable<string>;
  showResName$: Observable<boolean>;
  showResNumber$: Observable<boolean>;
  showGuestNames$: Observable<boolean>;

  private subsink: SubSink;
  private wasInside = false;

  /**
   * This following variable is a nightmare to think about. It is used to help
   * us know when to cancel the interaction with the item. The logic is as follows:
   * 1. If the user touches or mouses down on the item then we want to make it the
   * active item so that the drop zones are computed in the calendar service. (Allows
   * us to see where we can drop while draggin an item).
   * 2. If the item is active and we click on it then we want to cancel it's interaction.
   * 3. If we however double click on it then we want to open the sidenav and maintain
   * the interaction. The caveat with this is that the click event still fires.
   *
   * So essentially it boils down to these values:
   * 0: Not interacted
   * 1: Some interaction (could be the initial click, or drag and drop, or double click)
   * 2: The final click after the item is still active (Note that we use setTimeout to
   * allow the double click to reset the interaction state back to 1).
   *
   * Don't mess with this unless you fully grasp the logic... It took me one hour too many.
   */
  private interactionState: number;

  @HostBinding('attr.id') get id() {
    return this.item.id;
  }

  @HostBinding('style.zIndex') get zIndex() {
    return this.zIndexInternal;
  }

  constructor(
    public calendar: CalendarService,
    private sidebarService: SidebarService,
    private filterService: FilterService,
    private roomService: RoomService,
    private cd: ChangeDetectorRef
  ) {
    this.showResName$ = this.filterService.showResName$;
    this.showResNumber$ = this.filterService.showResNumber$;
    this.showGuestNames$ = this.filterService.showGuestNames$;
    this.subsink = new SubSink();
    this.interactionState = 0;
  }

  ngOnInit() {
    this.isFocused$ = this.calendar.focusedItem$.pipe(
      map(id => {
        if (id) {
          return id === this.id;
        } else {
          return false;
        }
      }),
      distinctUntilChanged(),
    );

    this.isActive$ = this.calendar.activeItem$.pipe(
      map(item => {
        if (item) {
          return item.id === this.id;
        } else {
          return false;
        }
      }),
      distinctUntilChanged(),
    );

    this.isFound$ = this.calendar.sortedResults$.pipe(
      map(results => {
        if (results.indexOf(this.id) >= 0) {
          return true;
        } else {
          return false;
        }
      }),
      distinctUntilChanged()
    );

    this.subsink.sink = this.calendar.viewItem$.subscribe((item) => {
      if (item && item.id === this.id) {
        this.viewReservation();
      }
    });

    this.subsink.sink = this.roomService.roomingChanges$.subscribe(() => this.cd.markForCheck());
    this.subsink.sink = combineLatest([this.isActive$, this.isFocused$]).subscribe(([isActive, isFocused]) => {
      if (isActive || isFocused) {
        this.zIndexInternal = '1';
      } else {
        this.zIndexInternal = 'unset';
      }
    });

    this.borderColour$ = this.borderColour();
    this.reservationColour$ = this.reservationColour();
  }

  ngOnDestroy() {
    this.subsink.unsubscribe();
  }

  reservationColour(): Observable<string> {
    return this.item.data.statusId != '0' ? this.calendar.reservationColour(this.item.data.reservationId) : this.calendar.reservationColour('#a0a0a0');
  }

  borderColour() {
    return combineLatest([this.isFocused$, this.isFound$, this.isActive$]).pipe(
      map(([isFocused, isFound, isActive]) => {
        if (isFocused && isFound && isActive) {
          return 'inset 0px 0px 0px 2px red, inset 0px 0px 0px 4px blue, inset 0px 0px 0px 5px rgb(112, 252, 20)';
        } else if (isFound && !isFocused && isActive) {
          return 'inset 0px 0px 0px 3px red, inset 0px 0px 0px 5px rgb(112, 252, 20)';
        } else if (isFocused && isFound && !isActive) {
          return 'inset 0px 0px 0px 3px blue, inset 0px 0px 0px 5px rgb(112, 252, 20)';
        } else if (isFound) {
          return 'inset 0px 0px 0px 2px rgb(112, 252, 20)';
        } else if (isActive) {
          return 'inset 0px 0px 0px 5px red';
        } else {
          return 'unset';
        }
      })
    );
  }

  @HostListener('dblclick')
  viewReservation(): void {
    this.interactionState = 1;
    this.calendar.onItemInteraction(this.item);
    this.calendar.onViewItem(this.item);

    const sidebar: Sidebar = {
      id: 'rooming-calendar-reservation',
      template: this.reservationPortal
    };
    this.sidebarService.showSidebar(sidebar, () => this.calendar.onReservationPortalClosed());
  }

  @HostListener('mousedown')
  onMouseDown(): void {
    this.interact();
  }

  @HostListener('touchstart')
  onTouchStart(): void {
    this.interact();
  }

  @HostListener('dragend')
  onDragEnd(): void {
    this.interactionState = 1;
  }

  private interact(): void {
    if (this.interactionState === 0) {
      this.calendar.onItemInteraction(this.item);
    }

    this.interactionState++;
  }

  @HostListener('click')
  async setActive(): Promise<void> {
    if (!environment.production) {
      console.log(this.item);
    }
    this.wasInside = true;

    this.calendar.onItemInteraction(this.item);
  }

  @HostListener('document:click')
  setInactive() {
    this.wasInside = false;
  }
}
