import { Component, OnInit, ChangeDetectionStrategy, OnDestroy } from '@angular/core';
import { FilterService } from '../../services/filter/filter.service';
import { Observable } from 'rxjs';
import { RoomService } from '../../services/room/room.service';
import { SearchCriteria } from '../../core/search/search-criteria';
import { SubSink } from 'subsink';

@Component({
  selector: 'app-search-header',
  templateUrl: './search-header.component.html',
  styleUrls: ['./search-header.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush
})
export class SearchHeaderComponent implements OnInit, OnDestroy {
  searchTerm$: Observable<string>;
  searchCriteria$: Observable<SearchCriteria>;
  totalResults$: Observable<number>;
  resultIndex$: Observable<number>;

  searchCriteria: SearchCriteria;
  showSearchOptions = false;
  private subsink: SubSink;

  constructor(
    private filterService: FilterService,
    private roomService: RoomService,
  ) {
    this.totalResults$ = roomService.totalResults$;
    this.resultIndex$ = roomService.resultIndex$;
    this.searchTerm$ = filterService.searchTerm$;
    this.searchCriteria$ = filterService.searchCriteria$;
    this.subsink = new SubSink();
  }

  ngOnInit() {
    this.subsink.sink = this.searchCriteria$.subscribe(searchCriteria => this.searchCriteria = searchCriteria);
  }

  ngOnDestroy() {
    this.subsink.unsubscribe();
  }

  onSearchTermChanged(term: string): void {
    this.filterService.onSearchTermChanged(term);
  }

  previousResult(): void {
    this.roomService.previousResult();
  }

  nextResult(): void {
    this.roomService.nextResult();
  }

  onKeyPress(key: number): void {
    if (key === 13) {
      this.nextResult();
    }
  }

  toggleShowSearchOptions(): void {
    this.showSearchOptions = !this.showSearchOptions;
  }

  openSearchOptions(): void {
    this.showSearchOptions = true;
  }

  closeSearchOptions(): void {
    this.showSearchOptions = false;
  }

  toggleSearchCriteria(option: string): void {
    if (option === 'guestName') {
      this.searchCriteria.guestName = !this.searchCriteria.guestName;
    } else if (option === 'reservationId') {
      this.searchCriteria.reservationId = !this.searchCriteria.reservationId;
    } else if (option === 'reservationName') {
      this.searchCriteria.reservationName = !this.searchCriteria.reservationName;
    } else if (option === 'voucher') {
      this.searchCriteria.voucher = !this.searchCriteria.voucher;
    }

    this.filterService.onSearchCriteriaChanged(this.searchCriteria);
  }

  closeSearch(): void {
    this.onSearchTermChanged('');
    this.closeSearchOptions();
  }
}
