import * as moment from 'moment';
import { CalendarDate } from './calendar-date';
import { Period } from '../filters/period';
import { CalendarIntersection } from './calendar-intersection';

export class CalendarDateBuilder {
  /**
   * The date format that the CalendarDateBuilder usese for the
   * main date.
   */
  static DATE_FORMAT = 'YYYY/MM/DD';

  /**
   * Creates calendar dates given the start date and the
   * number of days to show.
   * @param start The start date for the calendar.
   * @param days The number of days to render.
   */
  static calendarDates(start: moment.Moment, days: number): CalendarDate[] {
    const dates: CalendarDate[] = [];
    const current = start.clone();
    const end = start.clone().add(days, 'days');

    do {
      dates.push(CalendarDateBuilder.calendarDate(current));
      current.add(1, 'days');
    } while (current.isBefore(end));

    return dates;
  }

  /**
   * Create a calendar date from the given date.
   * @param date The date to transform.
   */
  static calendarDate(date: moment.Moment): CalendarDate {
    return {
        date: date.format(CalendarDateBuilder.DATE_FORMAT),
        shortDate: date.format('D MMM'),
        day: date.format('dddd'),
        shortDay: date.format('ddd'),
        isWeekend: CalendarDateBuilder.isWeekend(date),
        isToday: CalendarDateBuilder.isToday(date),
    };
  }

  /**
   * Whether the given date falls on a weekend or not (i.e. The day is
   * Saturday or Sunday).
   * @param date The date to check.
   */
  static isWeekend(date: moment.Moment): boolean {
    const day = date.day();
    return day === 0 || day === 6;
  }

  /**
   * Whether the given date is today or not.
   * @param date The date to check.
   */
  static isToday(date: moment.Moment): boolean {
    return moment().isSame(date, 'day');
  }

  /**
   * Determines the intersection of a period and a date range.
   * @param period The period to check for intersection.
   * @param from The start date of the range to check.
   * @param to The end date of the range to check.
   */
  static intersection(period: Period, from: string, to: string): CalendarIntersection {
    const periodEnd = period.start.clone().add(period.days - 1, 'days');
    let intersects = true;
    let overflowStart = false;
    let overflowEnd = false;
    const realStart = moment(from, CalendarDateBuilder.DATE_FORMAT);
    let start = realStart.clone();
    const realEnd = moment(to, CalendarDateBuilder.DATE_FORMAT);
    let end = realEnd.clone();

    if (
      (start.isBefore(period.start, 'day') && end.isBefore(period.start, 'day'))
      || (start.isAfter(periodEnd, 'day') && end.isAfter(periodEnd, 'day'))
    ) {
      intersects = false;
    } else {
      if (start.isBefore(period.start, 'day')) {
        overflowStart = true;
        start = period.start.clone();
      }

      if (end.isAfter(periodEnd, 'day')) {
        overflowEnd = true;
        end = periodEnd;
      }
    }

    const days = end.diff(start, 'days');
    const singleDay = realStart.isSame(realEnd, 'day');

    return {
      intersects,
      overflowStart,
      overflowEnd,
      from: start.format(CalendarDateBuilder.DATE_FORMAT),
      to: end.format(CalendarDateBuilder.DATE_FORMAT),
      days,
      singleDay
    };
  }

  /**
   * Returns the later of the two given dates.
   * @param date1 The first date.
   * @param date2 The second date.
   */
  static latestDate(date1: string, date2: string): string {
    const firstDate = moment(date1, CalendarDateBuilder.DATE_FORMAT);
    const secondDate = moment(date2, CalendarDateBuilder.DATE_FORMAT);
    return firstDate.isAfter(secondDate, 'day') ? date1 : date2;
  }

  /**
   * Converts the moment date into a string date.
   * @param date The date to convert.
   */
  static stringDate(date: moment.Moment): string {
    return date.format(this.DATE_FORMAT);
  }

  /**
   * Converts a string date into a moment date.
   * @param date The date to convert.
   */
  static momentDate(date: string): moment.Moment {
    return moment(date, this.DATE_FORMAT);
  }
}
