import { CalendarService } from '../../services/calendar/calendar.service';
import { CalendarDate } from './calendar-date';
import { Observable } from 'rxjs';
import { map } from 'rxjs/operators';
import { Room } from '../room/room';

export class CalendarGrid {

  constructor(private calendarService: CalendarService) {}

  /**
   * Determine the grid column property for the start of a calendar date.
   * @param date The date to get the column for.
   * @param span The number of columns to take up.
   */
  gridStartColumnByDate(date: CalendarDate, span: number = 1): Observable<string> {
    return this.gridStartColumnByRawDate(date.date, span);
  }

  /**
   * Determine the grid column property for the middle of a date.
   * @param date The date to get the column for.
   * @param span The number of columns to take up.
   */
  gridStartColumnByRawDate(date: string, span: number = 1): Observable<string> {
    return this.calendarService.startColumnForDate(date).pipe(
      map(column => column + ' / span ' + span)
    );
  }

  /**
   * Determine the grid column property for the middle of a calendar date.
   * @param date The date to get the column for.
   * @param span The number of columns to take up.
   */
  gridMidColumnByDate(date: CalendarDate, span: number = 1): Observable<string> {
    return this.gridMidColumnByRawDate(date.date, span);
  }

  /**
   * Determine the grid column property for the middle of a date.
   * @param date The date to get the column for.
   * @param span The number of columns to take up.
   */
  gridMidColumnByRawDate(date: string, span: number = 1): Observable<string> {
    return this.calendarService.midColumnForDate(date).pipe(
      map(column => column + ' / span ' + span)
    );
  }

  /**
   * Determine the grid column property for the nav next column.
   * @param span The number of columns to take up.
   */
  gridColumnForNavNext(span: number): Observable<string> {
    return this.calendarService.navNextGridColumn$.pipe(
      map(column => column + ' / span ' + span)
    );
  }

  /**
   * Determine the grid row property by the room.
   * @param room The room.
   * @param span The number of rows to span.
   */
  gridRowByRoom(room: Room, span: number = 1): Observable<string> {
    return this.gridRowByRoomId(room.id, span);
  }

  /**
   * Determine the grid row property by the property and room id.
   * @param propertyId The property id.
   * @param roomId The room id.
   * @param span The number of rows to span.
   */
  gridRowByRoomId(roomId: string, span: number = 1): Observable<string> {
    return this.calendarService.roomRow(roomId).pipe(
      map(row => row + ' / span ' + span)
    );
  }
}
