import { Property } from './property';
import { ItemSplit } from './item-split';
import { CalendarItem } from '../calendar/calendar-item';
import { Period } from '../filters/period';
import { GridRoomStatus } from './grid-room-status';
import { CalendarDate } from '../calendar/calendar-date';
import { CalendarDateBuilder } from '../calendar/calendar-date-builder';
import { CalendarIntersection } from '../calendar/calendar-intersection';
import { Guid } from 'guid-typescript';
import { RoomingChangeList } from './rooming-change-list';
import { RoomingChange } from './rooming-change';
import { Room } from './room';
import { ReservationItem } from './reservation-item';
import { CalendarBlockItem } from '../calendar/calendar-block-item';
import { Conflict, ConflictsByDate } from './conflict';
import { EXTENSION_DAYS } from './config';

interface ConstrainedItem {
  item: CalendarItem;
  availableRows: number;
  targetRow: number;
  days: number;
}

export class PropertyManager {
  /**
   * The statuses of all the rooms for each day in the period.
   */
  private grid: GridRoomStatus[][];

  /**
   * The dates that belong to the current period.
   */
  private dates: CalendarDate[];

  /**
   * A map of dates to columns;
   */
  private columnByDate: { [key: string]: number } = {};

  /**
   * A map of rooms to rows.
   */
  private rowByRoomId: { [key: string]: number } = {};

  /**
   * The row where the unallocated items start being placed.
   */
  private unallocatedStartRow = 0;

  /**
   * The list of calendar items to be rendered.
   */
  private calendarItems: CalendarItem[] = [];

  /**
   * The list of calendar block items to be rendered.
   */
  private calendarBlockItems: CalendarBlockItem[] = [];

  /**
   * The changes that have been made to the rooming arrangement.
   */
  private changes: RoomingChangeList[];

  /**
   * Map of group items to calendar items.
   */
  groupIdToCalendarItemMap: { [key: string]: CalendarItem } = {};

  /**
   * Map of item ids to calendar items.
   */
  calendarItemById: { [key: string]: CalendarItem } = {};

  /**
   * Creates the property manager.
   * @param properties The property to be managed.
   * @param itemSplit The split of the items for this property.
   * @param period The period of the calendar.
   */
  constructor(
    private property: Property,
    private itemSplit: ItemSplit,
    private period: Period,
    private environment: string
  ) {
    this.init();
  }

  /**
   * Initialise the property manager.
   */
  init(): void {
    this.createDates();
    this.createDateToColumnMap();
    this.createRoomToRowMap();
    this.createGrid();
    this.populateGrid();
    this.changes = [];
  }

  /**
   * Create the dates based on the period.
   */
  private createDates(): void {
    const start = this.period.start.clone().subtract(EXTENSION_DAYS, 'days');

    this.dates = CalendarDateBuilder.calendarDates(
      start,
      this.period.days + EXTENSION_DAYS * 2
    );
  }

  /**
   * Create the map of dates to columns.
   */
  private createDateToColumnMap(): void {
    this.dates.forEach((date, index) => {
      this.columnByDate[date.date] = index;
    });
  }

  /**
   * Create the map of room ids to rows.
   */
  private createRoomToRowMap(): void {
    this.property.rooms.forEach((room, index) => {
      this.rowByRoomId[room.id] = index;
    });
  }

  /**
   * Creates the grid for the property.
   */
  private createGrid(): void {
    this.grid = [];

    // Create the grid rows.
    this.property.rooms.forEach(() => {
      this.addGridRow();
    });

    this.unallocatedStartRow = this.grid.length;
  }

  /**
   * Populates the grid with the statuses from the items.
   */
  private populateGrid(): void {
    this.processAllocatedItems();
    this.processUnallocatedItems();
    this.processBlockItems();
  }

  /**
   * Populates the grid with statuses of the allocated items and adds
   * the item to the list of calendar items.
   */
  private processAllocatedItems(): void {
    this.itemSplit.allocated.forEach(item => {
      const latestDate = CalendarDateBuilder.latestDate(item.to, item.toBooked);
      const intersection = this.intersection(item.from, latestDate);
      const extendedIntersection = this.extendedIntersection(item.from, latestDate);
      const isLocked = this.isItemLocked(item);
      const canDrag = !extendedIntersection.overflowStart
        && !extendedIntersection.overflowEnd
        && !isLocked;
      const id = Guid.create().toString();
      const added = this.addIntersectionToGrid(item.roomId, extendedIntersection, id);

      if (added) {
        // Note: If this branch isn't reached then the room for the item does not exist in
        // the currently listed rooms for the property. So we ignore that item and move on.
        const row = this.rowForRoom(item.roomId);
        const room = this.property.rooms[row];
        const isUpgraded = item.accommId !== room.accommodationId;
        const isProvisional = item.statusId === '20';
        const canCheckInOut = (item.statusId === '30' || item.roomStatusInd === 5)      // Confirmed or already checked in
          && item.legend.legend !== 'Future reservation'
          && item.legend.legend !== 'Past reservation'
          && this.environment === item.accessEnvironment;   // On the correct environment
        const canCheckIn = item.currentRoomStatusInd != 5;
        
        const tmpItem = {
          id,
          data: item,
          intersection,
          extendedIntersection,
          propertyId: this.property.id,
          roomId: item.roomId,
          savedRoomId: item.roomId,
          savedStatusInd: item.roomStatusInd,
          savedStatusTime: item.roomStatusTime,
          savedCurrentRoomStatusInd: item.currentRoomStatusInd,
          isRoomed: true,
          isLocked,
          canDrag,
          isUpgraded,
          isProvisional,
          canCheckInOut,
          canCheckIn
        };
        this.calendarItems.push(tmpItem);
        this.groupIdToCalendarItemMap[tmpItem.data.id] = tmpItem;
        this.calendarItemById[id] = tmpItem;
      }
    });
  }

  /**
   * Returns the intersection of the given date range with the current
   * period.
   * @param from The start date.
   * @param to The end date.
   */
  private intersection(from: string, to: string): CalendarIntersection {
    return CalendarDateBuilder.intersection(this.period, from, to);
  }

  /**
   * Returns the intersection of the given date range with the extended period.
   * @param from The start date.
   * @param to The end date.
   */
  private extendedIntersection(from: string, to: string): CalendarIntersection {
    const start = this.period.start.clone().subtract(EXTENSION_DAYS, 'days');
    const days = this.period.days + EXTENSION_DAYS * 2;
    const period: Period = {
      start,
      days
    };

    return CalendarDateBuilder.intersection(period, from, to);
  }

  /**
   * Adds an intersection to the grid.
   * @param roomId The room to place it in.
   * @param intersection The intersection to add.
   */
  private addIntersectionToGrid(
    roomId: string,
    intersection: CalendarIntersection,
    itemId: string
  ): boolean {
    const row = this.rowForRoom(roomId);

    if (row === undefined) {
      return false;
    }

    const startColumn = this.columnForDate(intersection.from);
    const endColumn = startColumn + intersection.days;

    if (startColumn === 0 && intersection.overflowStart) {
      this.grid[row][0].overflowsStart = true;
    }

    // Update all the days that this item ranges in this row.
    for (let column = startColumn; column < endColumn; column++) {
      const status = this.grid[row][column];
      status.isAllocated = true;
      status.items.push(itemId);
    }

    return true;
  }

  /**
   * Adds a block intersection to the grid.
   * @param roomId The room to place it in.
   * @param intersection The intersection to add.
   * @returns Whether the block was able to be added to the grid or not.
   */
  private addBlockIntersectionToGrid(
    roomId: string,
    intersection: CalendarIntersection
  ): boolean {
    const row = this.rowForRoom(roomId);
    if (row === undefined) {
      return false;
    }

    const startColumn = this.columnForDate(intersection.from);
    const endColumn = startColumn + intersection.days;

    if (startColumn === 0 && intersection.overflowStart) {
      this.grid[row][0].overflowsStart = true;
    }

    // Update all the days that this item ranges in this row.
    for (let column = startColumn; column < endColumn; column++) {
      const status = this.grid[row][column];
      status.isBlocked = true;
    }

    return true;
  }

  /**
   * Removes an intersection from the grid.
   * @param roomId The room to remove it from.
   * @param intersection The intersection to remove.
   */
  private removeIntersectionFromGrid(
    roomId: string,
    intersection: CalendarIntersection,
    itemId: string
  ): void {
    const row = this.rowForRoom(roomId);
    const startColumn = this.columnForDate(intersection.from);
    const endColumn = startColumn + intersection.days;

    if (startColumn === 0 && intersection.overflowStart) {
      this.grid[row][0].overflowsStart = false;
    }

    // Update all the days that this item ranges in this row.
    for (let column = startColumn; column < endColumn; column++) {
      const status = this.grid[row][column];
      status.items = status.items.filter(i => i !== itemId);

      if (status.items.length === 0) {
        status.isAllocated = false;
      }
    }
  }

  /**
   * Populates the grid with unallocated items.
   */
  private processUnallocatedItems(): void {
    this.itemSplit.unallocated.forEach(item => {
      const latestDate = CalendarDateBuilder.latestDate(item.to, item.toBooked);
      const intersection = this.intersection(item.from, latestDate);

      // Only add unallocated items that are part of the current period.
      if (intersection.intersects) {
        const extendedIntersection = this.extendedIntersection(item.from, latestDate);
        const isLocked = this.isItemLocked(item);
        const canDrag = !extendedIntersection.overflowStart
          && !extendedIntersection.overflowEnd
          && !isLocked;
        const id = Guid.create().toString();
        const roomId = this.addUnallocatedIntersectionToGrid(extendedIntersection, id);
        const isProvisional = item.statusId === '20';

        const tmpItem = {
          id,
          data: item,
          intersection,
          extendedIntersection,
          propertyId: this.property.id,
          roomId,
          savedRoomId: '',
          savedStatusInd: item.roomStatusInd,
          savedStatusTime: item.roomStatusTime,
          savedCurrentRoomStatusInd: item.currentRoomStatusInd,
          isRoomed: false,
          isLocked,
          canDrag,
          isUpgraded: false,
          isProvisional,
          canCheckInOut: false,
          canCheckIn: false
        };
        this.calendarItems.push(tmpItem);
        this.calendarItemById[id] = tmpItem;
      }
    });

    if (this.itemSplit.unallocated.length === 0) {
      // Have at least one row to move things around.
      this.addRowForUnallocatedItems();
      this.addGridRow();
    }
  }

  /**
   * Adds the intersection to the grid and returns the room id that it
   * was addded to.
   * @param intersection The intersection to be added to the grid.
   */
  private addUnallocatedIntersectionToGrid(intersection: CalendarIntersection, itemId: string): string {
    let availableRow = this.availableRow(intersection, true);

    if (availableRow === false) {
      // There were no spaces available in the unallocated rows so add a new
      // row for unallocated items.
      availableRow = this.grid.length;
      this.addRowForUnallocatedItems();
      this.addGridRow();
    }

    const roomId = this.property.rooms[availableRow].id;
    this.addIntersectionToGrid(roomId, intersection, itemId);

    return roomId;
  }

  /**
   * Find an available row where the given intersection can be placed.
   * @param intersection The intersection of an item.
   */
  private availableRow(
    intersection: CalendarIntersection,
    excludeAllocated: boolean = false,
    excludeUnallocated: boolean = false
  ): number | false {
    const rows = this.availableRows(intersection, excludeAllocated, excludeUnallocated);

    if (rows.length === 0) {
      return false;
    } else {
      return rows[0];
    }
  }

  /**
   * Find available rows where the given intersection can be placed.
   * @param intersection The intersection of an item.
   */
  private availableRows(
    intersection: CalendarIntersection,
    excludeAllocated: boolean = false,
    excludeUnallocated: boolean = false
  ): number[] {
    const results = [];
    let currentRow = 0;

    if (excludeAllocated) {
      currentRow = this.unallocatedStartRow;
    }

    let endRow = this.grid.length;

    if (excludeUnallocated) {
      endRow = this.unallocatedStartRow;
    }

    while (currentRow < endRow) {
      if (
        this.hasAvailableSpace(currentRow, intersection)
      ) {
        results.push(currentRow);
      }
      currentRow++;
    }

    return results;
  }

  /**
   * Checks whether there is space for the given days in the given row (room).
   * @param row The row to check.
   * @param intersection The intersection to check.
   */
  private hasAvailableSpace(row: number, intersection: CalendarIntersection): boolean {
    let column = this.columnForDate(intersection.from);
    let endColumn = column + intersection.days;

    if (intersection.singleDay) {
      // Need to accommodate single day
      endColumn++;
    }

    if (column === 0 && intersection.overflowStart) {
      if (this.grid[row][0].overflowsStart) {
        // The overflow position has already been taken.
        return false;
      }
    }

    for (column; column < endColumn; column++) {
      const roomStatus = this.grid[row][column];
      if (roomStatus.isBlocked || roomStatus.isAllocated) {
        return false;
      }
    }

    return true;
  }

  /**
   * Checks whether there is space for the item in the given row.
   * @param row The row to check for space.
   * @param item The item to be placed.
   */
  private hasSpaceForItem(row: number, item: CalendarItem): boolean {
    return this.hasAvailableSpace(row, item.extendedIntersection);
  }

  /**
   * Adds a new room for unallocated items.
   */
  private addRowForUnallocatedItems(): void {
    const room = {
      id: Guid.create().toString(),
      name: 'Unallocated',
      forUnallocated: true,
      accommodationId: '',
      accommodationName: ''
    };
    const index = this.property.rooms.length;

    this.property.rooms.push(room);
    this.rowByRoomId[room.id] = index;
  }

  /**
   * Adds a new row to the grid.
   */
  private addGridRow(): void {
    this.grid.push(this.dates.map(() => {
      return {
        isBlocked: false,
        isAllocated: false,
        overflowsStart: false,
        items: []
      };
    }));
  }

  /**
   * Retrieve the column for the given date.
   * @param date The date.
   */
  private columnForDate(date: string): number {
    return this.columnByDate[date];
  }

  /**
   * Retrieve the row for the given room.
   * @param roomId The room id.
   */
  private rowForRoom(roomId: string): number {
    return this.rowByRoomId[roomId];
  }

  /**
   * Retrieves the room for the given row.
   * @param row The row to get the room for.
   */
  private roomByRow(row: number): Room {
    return this.property.rooms[row];
  }

  /**
   * Retrieves the calendar items.
   */
  getCalendarItems(): CalendarItem[] {
    return this.calendarItems;
  }

  /**
   * Returns the property id for the property being managed.
   */
  propertyId(): string {
    return this.property.id;
  }

  /**
   * Find a list of available rooms for the calendar item.
   * @param item The item to search for space.
   */
  availableRooms(item: CalendarItem): string[] {
    const rows = this.availableRows(item.extendedIntersection);
    const rooms = rows.map(row => this.property.rooms[row].id);
    return rooms;
  }

  /**
   * Called when an item has been moved.
   * @param item The item that was moved.
   * @param toRoomId The room to which the item was moved.
   */
  onItemMoved(item: CalendarItem, toRoomId: string): void {
    const change = this.changeRoom(item, toRoomId);

    const roomingChange = {
      changes: [change]
    };

    this.changes.push(roomingChange);
    item.roomId = toRoomId;
    item.isRoomed = this.isRoomed(toRoomId);
  }

  /**
   * Wether the room is an actual room and not a virtual one
   * created for unallocated items.
   * @param roomId The room id.
   */
  private isRoomed(roomId: string): boolean {
    const room = this.property.rooms.find(r => r.id === roomId);
    return !room.forUnallocated;
  }

  /**
   * Reverts the previous rooming change made.
   */
  undoLastChange(): void {
    if (this.changes.length === 0) {
      return;
    }

    const index = this.changes.length - 1;
    const roomingChange = this.changes[index];
    this.revertRoomingChange(roomingChange);
    this.changes.splice(index, 1);
    this.pruneGrid();
  }

  /**
   * Reverts the changes made to the rooming arrangement.
   * @param roomingChange The changes made to the rooming arrangement.
   */
  private revertRoomingChange(roomingChange: RoomingChangeList): void {
    const changes = roomingChange.changes.reverse();
    changes.forEach(change => {
      const item = change.item;
      this.changeRoom(item, change.oldRoomId);
    });
  }

  /**
   * Whether there are rooming changes or not.
   */
  hasRoomingChanges(): boolean {
    return this.changes.length > 0;
  }

  /**
   * Retrieves all the items that have had their rooming information changed.
   */
  changedItems(): CalendarItem[] {
    const itemMap: { [key: string]: CalendarItem } = {};

    this.changes.forEach(roomingChangeList => {
      roomingChangeList.changes.forEach(change => {
        const id = change.item.id;

        if (!(id in itemMap)) {
          itemMap[id] = change.item;
        }
      });
    });

    const changes = [];

    Object.keys(itemMap).forEach(key => {
      const item = itemMap[key];

      if (
        (item.isRoomed && !item.savedRoomId)
        || (item.isRoomed && item.roomId !== item.savedRoomId)
        || (!item.isRoomed && item.savedRoomId)
      ) {
        changes.push(item);
      }
    });

    return changes;
  }

  /**
   * Tries to room unallocated items.
   * @returns true if any changes were made otherwise false.
   */
  autoRoom(): boolean {
    const allocatedItems = this.unlockedAllocatedItems();
    let changes = this.unallocateItems(allocatedItems);
    const items = this.unallocatedItems();

    if (items.length === 0) {
      // No items to room
      return false;
    }

    const itemsByAccommType = this.itemsByAccommType(items);

    Object.keys(itemsByAccommType).forEach(accommodationId => {
      const accommItems = itemsByAccommType[accommodationId];
      const roomChanges = this.autoRoomItems(accommodationId, accommItems);

      if (roomChanges.length > 0) {
        changes = [...changes, ...roomChanges];
      }
    });

    const hasChanges = this.anyItemsChanged();

    if (changes.length > 0) {
      this.changes.push({
        changes
      });
    }

    if (!hasChanges) {
      this.changes = [];
    }

    this.pruneGrid();

    return hasChanges;
  }

  /**
   * Whether any of the items have changes.
   */
  private anyItemsChanged(): boolean {
    for (const item of this.calendarItems) {
      if (
        (item.isRoomed && !item.savedRoomId)
        || (item.isRoomed && item.roomId !== item.savedRoomId)
        || (!item.isRoomed && item.savedRoomId)
      ) {
        return true;
      }
    }

    return false;
  }

  /**
   * Retrieve all the unallocated items from the current period.
   */
  private unallocatedItems(includeOverflow: boolean = false): CalendarItem[] {
    if (includeOverflow) {
      return this.calendarItems.filter(item => {
        return !item.isRoomed;
      });
    } else {
      return this.calendarItems.filter(item => {
        return !item.isRoomed
          && item.intersection.intersects
          && !item.extendedIntersection.overflowStart
          && !item.extendedIntersection.overflowEnd;
      });
    }
  }

  /**
   * Creates a dictionary with the accommodation id as the key
   * and an array of calendar items as the value.
   * @param items The items to group.
   */
  private itemsByAccommType(items: CalendarItem[]): { [key: string]: CalendarItem[] } {
    const map: { [key: string]: CalendarItem[] } = {};

    items.forEach(item => {
      const accommodationId = item.data.accommId;
      if (!(accommodationId in map)) {
        map[accommodationId] = [];
      }

      map[accommodationId].push(item);
    });

    return map;
  }

  /**
   * Attempts to autoroom the items for the accommodation.
   * @param accommodationId The id of the accommodation.
   * @param items The items to be roomed.
   */
  private autoRoomItems(
    accommodationId: string,
    items: CalendarItem[]
  ): RoomingChange[] {
    const changes: RoomingChange[] = [];
    const rows = this.rowsForAccommodationType(accommodationId);

    let remainingItems = this.itemsWithAvailablePlacements(rows, items);
    let constrainedItem: ConstrainedItem;
    let change: RoomingChange;
    let room: Room;
    let index: number;

    while (remainingItems.length > 0) {
      constrainedItem = this.mostConstrainedItem(rows, remainingItems);
      // Apply rooming change
      room = this.roomByRow(constrainedItem.targetRow);
      change = this.changeRoom(constrainedItem.item, room.id);
      changes.push(change);
      // Update remaining items
      index = remainingItems.findIndex(item => item.id === constrainedItem.item.id);
      remainingItems.splice(index, 1);
      remainingItems = this.itemsWithAvailablePlacements(rows, remainingItems);
    }

    return changes;
  }

  /**
   * Determines all the rows with rooms that belong to the accommodation type.
   * @param accommodationId The accommodation id.
   */
  private rowsForAccommodationType(accommodationId: string): number[] {
    const rows = [];
    this.property.rooms.forEach((room, index) => {
      if (room.accommodationId === accommodationId) {
        rows.push(index);
      }
    });
    return rows;
  }

  /**
   * Returns items that can be placed in the available rows.
   * @param rows The rows where the items should be placed in.
   * @param items The items to be placed.
   */
  private itemsWithAvailablePlacements(
    rows: number[],
    items: CalendarItem[]
  ): CalendarItem[] {
    return items.filter(item => {
      const validRows = this.availablePlacementsForItem(rows, item);
      return validRows.length > 0;
    });
  }

  /**
   * The rows in which the item can be placed.
   * @param item The item to check for available placements.
   */
  private availablePlacementsForItem(
    rows: number[],
    item: CalendarItem
  ): number[] {
    const validRows = [];

    rows.forEach(row => {
      if (this.hasSpaceForItem(row, item)) {
        validRows.push(row);
      }
    });

    return validRows;
  }

  /**
   * Determines which item is the most constrained item.
   * @param rows The possible rows that the items could be placed in.
   * @param items The items to be evaluated.
   */
  private mostConstrainedItem(
    rows: number[],
    items: CalendarItem[]
  ): ConstrainedItem {
    const constrainedItems = items.map(item => this.constrainItem(rows, item));
    const sortedItems = constrainedItems.sort((a, b) => {
      if (a.availableRows === b.availableRows) {
        return b.days - a.days;
      } else {
        return a.availableRows - b.availableRows;   // Ascending order of available rows.
      }
    });

    return sortedItems[0];
  }

  /**
   * Gets the contraints for an item.
   */
  private constrainItem(rows: number[], item: CalendarItem): ConstrainedItem {
    const startColumn = this.columnForDate(item.extendedIntersection.from);
    const endColumn = startColumn + item.extendedIntersection.days;
    const validRows = rows.filter(row => this.hasSpaceForItem(row, item));
    const rowsAndSpacing = validRows.map(row => {
      const spacesBefore = this.emptySpacesBefore(row, startColumn);
      const spacesAfter = this.emptySpacesAfter(row, endColumn);
      const minSpaces = Math.min(spacesBefore, spacesAfter);

      return {
        row,
        minSpaces,
      };
    });
    const sortedRows = rowsAndSpacing.sort((a, b) => {
      return a.minSpaces - b.minSpaces;   // Ascending order of minimum spaces
    });

    return {
      item,
      availableRows: sortedRows.length,
      targetRow: sortedRows[0].row,
      days: item.extendedIntersection.days
    };
  }

  /**
   * Determines the number of empty spaces before a column.
   * @param row The row to check.
   * @param column The column to start from (exclusive).
   */
  private emptySpacesBefore(row: number, column: number): number {
    let count = 0;

    for (let i = column - 1; i >= 0; i--) {
      const roomStatus = this.grid[row][i];

      if (!roomStatus.isAllocated && !roomStatus.isBlocked) {
        count++;
      } else {
        break;
      }
    }

    return count;
  }

  /**
   * Determines the number of empty spaces after a column.
   * @param row The row to check.
   * @param column The column to start from (inclusive).
   */
  private emptySpacesAfter(row: number, column: number): number {
    let count = 0;

    for (let i = column; i < this.dates.length; i++) {
      const roomStatus = this.grid[row][i];

      if (!roomStatus.isAllocated && !roomStatus.isBlocked) {
        count++;
      } else {
        break;
      }
    }

    return count;
  }

  /**
   * Changes the room of an item.
   * @param item The item to change rooms.
   * @param roomId The room id to change to.
   */
  private changeRoom(item: CalendarItem, roomId: string): RoomingChange {
    const oldRoomId = item.roomId;
    const wasRoomed = this.isRoomed(item.roomId);

    if (item.roomId in this.rowByRoomId) {
      this.removeIntersectionFromGrid(item.roomId, item.extendedIntersection, item.id);
    }

    if (!(roomId in this.rowByRoomId)) {
      const newRoomId = this.addUnallocatedIntersectionToGrid(item.extendedIntersection, item.id);
      item.roomId = newRoomId;
      item.isRoomed = false;
    } else {
      item.roomId = roomId;
      item.isRoomed = this.isRoomed(roomId);
      this.addIntersectionToGrid(roomId, item.extendedIntersection, item.id);
    }

    const newRoomId = item.roomId;
    const change: RoomingChange = {
      item,
      newRoomId,
      oldRoomId,
      wasRoomed
    };

    const row = this.rowForRoom(newRoomId);
    const room = this.property.rooms[row];

    item.isUpgraded = item.isRoomed && item.data.accommId !== room.accommodationId;

    return change;
  }

  /**
   * Whether or not the item is locked.
   * @param item The item to check.
   */
  private isItemLocked(item: ReservationItem): boolean {
    const isCheckedIn = item.roomStatusInd === 5;
    const isRoomRequested = item.requestedYn === 1 ? true : false;
    return isCheckedIn || isRoomRequested;
  }

  /**
   * Retrieve all the unallocated items.
   */
  private unlockedAllocatedItems(): CalendarItem[] {
    return this.calendarItems.filter(item => {
      return item.isRoomed
        && item.canDrag
        && item.intersection.intersects
        && !item.extendedIntersection.overflowStart
        && !item.extendedIntersection.overflowEnd;
    });
  }

  /**
   * Unallocates allocated items.
   * @param items The items to unallocate.
   */
  private unallocateItems(items: CalendarItem[]): RoomingChange[] {
    const changes: RoomingChange[] = [];

    items.forEach(item => {
      const change = this.changeRoom(item, '');
      changes.push(change);
    });

    return changes;
  }

  /**
   * Reorganise the unallocated items and remove any unnecessary rows.
   */
  private pruneGrid(): void {
    const unallocatedItems = this.unallocatedItems(true);
    unallocatedItems.forEach(item => {
      this.removeIntersectionFromGrid(item.roomId, item.extendedIntersection, item.id);
    });
    unallocatedItems.forEach(item => {
      const roomId = this.addUnallocatedIntersectionToGrid(item.extendedIntersection, item.id);
      item.roomId = roomId;
    });
    this.removeUnusedUnallocatedRows();
  }

  /**
   * Remove unused unallocated rows except for the first one which should
   * always be present.
   */
  private removeUnusedUnallocatedRows(): void {
    let removeFrom = this.unallocatedStartRow + 1;
    let hasRowsToRemove = false;

    for (let i = removeFrom; i < this.grid.length; i++) {
      if (!this.isRowUsed(i)) {
        removeFrom = i;
        hasRowsToRemove = true;
        break;
      }
    }

    if (!hasRowsToRemove) {
      return;
    }

    this.grid.splice(removeFrom);

    for (let i = removeFrom; i < this.property.rooms.length; i++) {
      const roomId = this.property.rooms[i].id;
      delete this.rowByRoomId[roomId];
    }

    this.property.rooms.splice(removeFrom);
  }

  /**
   * Whether or not a row in the grid is being used.
   * @param row The row to check.
   */
  private isRowUsed(row: number): boolean {
    for (let i = 0; i < this.dates.length; i++) {
      const roomStatus = this.grid[row][i];

      if (
        roomStatus.overflowsStart
        || roomStatus.isAllocated
        || roomStatus.isBlocked
      ) {
        return true;
      }
    }

    return false;
  }

  /**
   * Adds the block items to the grid.
   */
  private processBlockItems(): void {
    this.itemSplit.blocks.forEach(item => {
      const intersection = this.intersection(item.from, item.to);
      const extendedIntersection = this.extendedIntersection(item.from, item.to);
      const added = this.addBlockIntersectionToGrid(item.roomId, extendedIntersection);

      if (added) {
        // If the room exists and the block was added then also render the block.
        const calendarBlockItem = {
          id: Guid.create().toString(),
          data: item,
          intersection,
          extendedIntersection,
          propertyId: item.propertyId,
          roomId: item.roomId
        };
        this.calendarBlockItems.push(calendarBlockItem);
      }
    });
  }

  /**
   * Retrieves the calendar block items.
   */
  getCalendarBlockItems(): CalendarBlockItem[] {
    return this.calendarBlockItems;
  }

  refreshItems(items: ReservationItem[]): void {
    items.forEach(item => {
      if (item.id in this.groupIdToCalendarItemMap) {
        const calendarItem = this.groupIdToCalendarItemMap[item.id];
        calendarItem.data = item;
        calendarItem.savedStatusInd = item.roomStatusInd;
        calendarItem.savedStatusTime = item.roomStatusTime;
        calendarItem.savedCurrentRoomStatusInd = item.currentRoomStatusInd;
        const canCheckInOut = item.roomId   // It is roomed
          && (item.statusId === '30' || item.roomStatusInd === 5)      // Confirmed or already checked in provisional.
          && item.legend.legend !== 'Future reservation'
          && item.legend.legend !== 'Past reservation'
          && this.environment === item.accessEnvironment;   // On the correct environment
        const canCheckIn = item.currentRoomStatusInd != 5;
        calendarItem.canCheckInOut = canCheckInOut;
        calendarItem.canCheckIn = canCheckIn;
        const latestDate = CalendarDateBuilder.latestDate(calendarItem.data.to, calendarItem.data.toBooked);
        this.removeIntersectionFromGrid(calendarItem.roomId, calendarItem.extendedIntersection, calendarItem.id);
        calendarItem.intersection = this.intersection(calendarItem.data.from, latestDate);
        this.addIntersectionToGrid(calendarItem.roomId, calendarItem.extendedIntersection, calendarItem.id);
      }
    });
  }

  /**
   * Retrieves the rooming conflicts grouped by date.
   */
  getConflicts(): ConflictsByDate {
    const conflicts: ConflictsByDate = {};
    const startColumn = EXTENSION_DAYS;    // First 5 days are not part of current period.
    const endColumn = this.dates.length - EXTENSION_DAYS;    // Last 5 days are not part of the current period.

    for (let j = startColumn; j < endColumn; j++) {
      for (let i = 0; i < this.grid.length; i++) {
        const roomStatus = this.grid[i][j];

        if (roomStatus.items.length > 1) {
          // Room has a conflict
          const conflictDate = this.dates[j].date;

          if (!(conflictDate in conflicts)) {
            conflicts[conflictDate] = [];
          }

          const conflict: Conflict = {
            date: conflictDate,
            room: this.roomByRow(i),
            items: roomStatus.items.map(id => this.calendarItemById[id])
          };

          conflicts[conflictDate].push(conflict);
        }
      }
    }

    return conflicts;
  }

  /**
   * Moves an item to the unallocated zone.
   * @param item The item to move.
   *
   * @returns Whether or not there were any changes.
   */
  moveToUnallocated(item: CalendarItem): boolean {
    const change = this.changeRoom(item, '');
    this.changes.push({changes: [change]});
    const hasChanges = this.anyItemsChanged()

    if (!hasChanges) {
      this.changes = [];
    }

    this.pruneGrid();

    return hasChanges;
  }
}
