import { Component, OnInit, ChangeDetectionStrategy } from '@angular/core';
import { FilterService } from '../../services/filter/filter.service';
import { first } from 'rxjs/operators';
import { combineLatest } from 'rxjs';
import { CalendarSortOrder } from '../../core/filters/calendar-sort-order';
import { SidebarService } from 'src/app/shared/services/sidebar/sidebar.service';

@Component({
  selector: 'app-config-options',
  templateUrl: './config-options.component.html',
  styleUrls: ['./config-options.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush
})
export class ConfigOptionsComponent implements OnInit {
  sortCalendarByValue: CalendarSortOrder;
  sortCalendarBy$ = this.filterService.sortCalendarBy$;
  showResNumberValue: boolean;
  showResNumber$ = this.filterService.showResNumber$;
  showResNameValue: boolean;
  showResName$ = this.filterService.showResName$;
  showGuestNamesValue: boolean;
  showGuestNames$ = this.filterService.showGuestNames$;
  showConflictsOnLoadValue: boolean;
  showConflictsOnLoad$ = this.filterService.showConflictsOnLoad$;

  constructor(
    private filterService: FilterService,
    private sidebarService: SidebarService
  ) {
    this.loadConfig();
  }

  private async loadConfig(): Promise<void> {
    this.sortCalendarByValue = await this.sortCalendarBy$.pipe(first()).toPromise();
    this.showResNumberValue = await this.showResNumber$.pipe(first()).toPromise();
    this.showResNameValue = await this.showResName$.pipe(first()).toPromise();
    this.showGuestNamesValue = await this.showGuestNames$.pipe(first()).toPromise();
    this.showConflictsOnLoadValue = await this.showConflictsOnLoad$.pipe(first()).toPromise();
  }

  ngOnInit(): void {
  }

  close(): void {
    this.sidebarService.close();
  }

  apply(): void {
    combineLatest([this.sortCalendarBy$, this.showResNumber$, this.showResName$, this.showGuestNames$, this.showConflictsOnLoad$ ]).pipe(
      first()
    ).subscribe(([sortCalendarBy, showResNumber, showResName, showGuestNames, showConflictsOnLoad]) => {

      if (sortCalendarBy !== this.sortCalendarByValue) {
        this.filterService.onSortCalendarByChanged(this.sortCalendarByValue);
      }

      if (showResNumber !== this.showResNumberValue) {
        this.filterService.onShowResNumberChanged(this.showResNumberValue);
      }

      if (showResName !== this.showResNameValue) {
        this.filterService.onShowResNameChanged(this.showResNameValue);
      }

      if (showGuestNames !== this.showGuestNamesValue) {
        this.filterService.onShowGuestNamesChanged(this.showGuestNamesValue);
      }

      if (showConflictsOnLoad !== this.showConflictsOnLoadValue) {
        this.filterService.onShowConflictsOnLoadChanged(this.showConflictsOnLoadValue);
      }
    });

    this.close();
  }

  sortByValue(sortBy: CalendarSortOrder): string {
    if (sortBy === CalendarSortOrder.Accommmodation) {
      return 'accommodation';
    } else {
      return 'room';
    }
  }

  onSortCalendarByChange(value: string): void {
    if (value === 'accommodation') {
      this.sortCalendarByValue = CalendarSortOrder.Accommmodation;
    } else {
      this.sortCalendarByValue = CalendarSortOrder.Room;
    }
  }
}
