import { Component, ChangeDetectionStrategy, ViewChild, TemplateRef } from '@angular/core';
import * as moment from 'moment';
import { FilterService } from '../../services/filter/filter.service';
import { Period } from '../../core/filters/period';
import { Observable } from 'rxjs';
import { map } from 'rxjs/operators';
import { RoomService } from '../../services/room/room.service';
import { SidebarService } from 'src/app/shared/services/sidebar/sidebar.service';
import { Sidebar } from 'src/app/shared/services/sidebar/sidebar';
import { CalendarService } from '../../services/calendar/calendar.service';
import { CalendarItem } from '../../core/calendar/calendar-item';
import { CalendarAccessService } from '../../services/access/calendar-access.service';

@Component({
  selector: 'app-filter-header',
  templateUrl: './filter-header.component.html',
  styleUrls: ['./filter-header.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush
})
export class FilterHeaderComponent {
  /**
   * The period to display for the calendar.
   */
  period$: Observable<Period> = this.filterService.period$;

  /**
   * The date string to display.
   */
  display$: Observable<string>;

  /**
   * Whether there are rooming changes or not.
   */
  hasRoomingChanges$: Observable<boolean>;

  /**
   * Hides the colour grouping option in the filters. This property
   * should be removed once the grouping functionality has been
   * imlemented.
   */
  showColourGrouping = false;

  activeItem$: Observable<CalendarItem>;

  @ViewChild('filterPortal') filterPortal: TemplateRef<any>;

  /**
   * Whether the user has calendar update access.
   */
  canUpdate$: Observable<boolean>;

  /**
   * Whether all props/accomms are included in filter or not
   */
  showAllAccommInd$: Observable<boolean> = this.filterService.showAllAccommInd$;

  /**
   * Create the filter header component.
   */
  constructor(
    private filterService: FilterService,
    private sidebarService: SidebarService,
    private roomService: RoomService,
    private calendar: CalendarService,
    private accessService: CalendarAccessService
  ) {
    this.period$ = filterService.period$;
    this.display$ = this.display();
    this.hasRoomingChanges$ = roomService.hasRoomingChanges$;
    this.activeItem$ = calendar.activeItem$;
    this.canUpdate$ = this.accessService.canUpdate$;
  }

  /**
   * Updates the display when the date changes.
   */
  private display(): Observable<string> {
    return this.period$.pipe(
      map(period => {
        if (this.isToday(period.start)) {
          return 'Today';
        } else {
          return period.start.format('D/M/YYYY');
        }
      })
    );
  }

  /**
   * Called when the start date has changed.
   * @param date The new start date.
   */
  onStartDateChanged(date: moment.Moment): void {
    this.filterService.onStartDateChanged(date);
  }

  /**
   * Whether the date is today.
   * @param date The date to check.
   */
  private isToday(date: moment.Moment): boolean {
    const today = moment();
    return date.isSame(today, 'day');
  }

  /**
   * Toggles the more filters property.
   */
  toggleMoreFilters(): void {
    const sidebar: Sidebar = {
      id: 'rooming-calendar-filters',
      template: this.filterPortal
    };
    this.sidebarService.showSidebar(sidebar);
  }

  /**
   * Called when the number of days to display changes.
   * @param days The number of days to display.
   */
  onDaysChanged(days: number): void {
    this.filterService.onDaysChanged(days);
  }

  /**
   * Selects the current date as the start date.
   * @param event The click event that occurred.
   */
  pickToday(event: any): void {
    this.onStartDateChanged(moment());
    event.stopPropagation();
  }

  /**
   * Reverts the last rooming change made.
   */
  undoLastChange(): void {
    this.roomService.undoLastChange();
  }

  /**
   * Reverts all the rooming changes made.
   */
  undoAllChanges(): void {
    this.roomService.undoAllChanges();
  }

  /**
   * Save the rooming changes made.
   */
  saveChanges(): void {
    this.roomService.saveChanges();
  }

  /**
   * Attempt to room unallocated items.
   */
  autoRoom(): void {
    this.roomService.autoRoom();
  }

  viewActiveItem() {
    this.calendar.viewActiveItem();
  }

  refresh(): void {
    this.roomService.refresh();
  }
}
