import { Component, OnInit, ChangeDetectorRef, OnDestroy } from '@angular/core';
import { FrontOfHouseService } from 'src/app/rooming-calendar/services/front-of-house/front-of-house.service';
import { CalendarItem } from 'src/app/rooming-calendar/core/calendar/calendar-item';
import { PortalService } from 'src/app/shared/components/portal';
import { SubSink } from 'subsink';
import { AuthorisationService } from 'src/app/shared/services/authorisation/authorisation.service';
import { BehaviorSubject, Observable } from 'rxjs';
import { RoomService } from 'src/app/rooming-calendar/services/room/room.service';
import { CalendarAccessService } from '../../services/access/calendar-access.service';
import { first } from 'rxjs/operators';
import * as moment from 'moment';
import { ReservationItem } from '../../core/room/reservation-item';


@Component({
  selector: 'app-reservation-check-in',
  templateUrl: './reservation-check-in.component.html',
  styleUrls: ['./reservation-check-in.component.scss'],
})
export class ReservationCheckInComponent implements OnInit, OnDestroy {
  private subSink: SubSink;

  canUpdate$ = this.accessService.canUpdate$;
  canDateCheckouts$ = this.accessService.canDateCheckouts$;
  private checkInDisabledSource: BehaviorSubject<boolean>;
  private checkOutDisabledSource: BehaviorSubject<boolean>;
  checkInDisabled$: Observable<boolean>;
  checkOutDisabled$: Observable<boolean>;
  private checkInCountSource: BehaviorSubject<number>;
  private checkOutCountSource: BehaviorSubject<number>;
  checkInCount$: Observable<number>;
  checkOutCount$: Observable<number>;
  /**
   * Items for the active reservation
   */
  items: CalendarItem[] = [];
  calendarItemByDataId: {[id: string]: CalendarItem};

  selectedItems: string[] = [];
  checkoutDate: moment.Moment;

  constructor(
    private frontOfHouseService: FrontOfHouseService,
    private portalService: PortalService,
    private cd: ChangeDetectorRef,
    private authorisation: AuthorisationService,
    private roomService: RoomService,
    private accessService: CalendarAccessService
  ) {
    this.subSink = new SubSink();
    this.checkoutDate = moment();
    this.checkInDisabledSource = new BehaviorSubject(true);
    this.checkOutDisabledSource = new BehaviorSubject(true);
    this.checkInDisabled$ = this.checkInDisabledSource.asObservable();
    this.checkOutDisabled$ = this.checkOutDisabledSource.asObservable();
    this.checkInCountSource = new BehaviorSubject<number>(0);
    this.checkInCount$ = this.checkInCountSource.asObservable();
    this.checkOutCountSource = new BehaviorSubject<number>(0);
    this.checkOutCount$ = this.checkOutCountSource.asObservable();
  }

  ngOnInit(): void {
    this.updateRoomItems();

    this.subSink.sink = this.frontOfHouseService.loading$.subscribe((loading) => {
      this.portalService.onShowLoading(loading);
    });

    this.subSink.sink = this.roomService.items$.subscribe(() => {
      this.portalService.onShowLoading(true);
      this.refresh();
      this.portalService.onShowLoading(false);
    });
  }

  updateRoomItems(): void {
    this.frontOfHouseService.reservationItems().then(items => {
      this.items = items;
      this.calendarItemByDataId = {};
      this.selectedItems = [];

      this.items.forEach(item => {
        this.calendarItemByDataId[item.data.id] = item;
      });

      this.updateEnabledActions();
      this.cd.markForCheck();
    });
  }

  ngOnDestroy() {
    this.subSink.unsubscribe();
  }

  checkIn(): void {
    this.portalService.onShowLoading(true);
    const items = [];

    this.items.forEach(item => {
      if (
        this.selectedItems.includes(item.data.id)
        && item.canCheckInOut
        && (
          item.data.roomStatusInd === 2
          || item.data.roomStatusInd === 8
        )
      ) {
        items.push(item);
      }
    });

    this.frontOfHouseService.checkIn(items).then(() => {
      this.refresh();
      this.portalService.onShowLoading(false);
    });
  }

  async checkOut(): Promise<void> {
    const canDateCheckout = await this.canDateCheckouts$.pipe(first()).toPromise();
    let checkOutDate = null;
    let today = moment();

    if (canDateCheckout && !this.checkoutDate.isSame(today, 'day')) {
      checkOutDate = this.checkoutDate.format("YYYY-MM-DD HH:mm:ss");
    }

    this.portalService.onShowLoading(true);
    const items = [];

    this.items.forEach(item => {
      if (
        this.selectedItems.includes(item.data.id)
        && item.canCheckInOut
        && item.data.roomStatusInd === 5
      ) {
        items.push(item);
      }
    });

    this.frontOfHouseService.checkOut(items, checkOutDate).then(() => {
      this.refresh();
      this.portalService.onShowLoading(false);
    });
  }

  guestRows(guests: [], rowCount: number): any[] {
    const capacity = [];

    for (let index = 0; index < rowCount; index++) {
      if (guests[index]) {
        capacity.push(guests[index]);
      } else {
        capacity.push({});
      }
    }

    return capacity;
  }

  toggleSelection(id: string): void {
    const index = this.selectedItems.indexOf(id);
    if (index < 0) {
      this.selectedItems.push(id);
    } else {
      this.selectedItems.splice(index, 1);
    }

    this.updateEnabledActions();
  }

  async updateEnabledActions(): Promise<void> {
    const canUpdate = await this.canUpdate$.pipe(first()).toPromise();
    const items = this.selectedItems.map(item => this.calendarItemByDataId[item]);
    const checkInItems = items
      .filter(item => item.canCheckInOut)
      .filter(item => (item.data.roomStatusInd === 2 || item.data.roomStatusInd === 8) && item.canCheckIn);
    const checkInCount = checkInItems.length;
    const hasCheckIn = checkInCount > 0;

    const checkOutItems = items
      .filter(item => item.canCheckInOut)
      .filter(item => item.data.roomStatusInd === 5);
    const checkOutCount = checkOutItems.length;
    const hasCheckOut = checkOutCount > 0;

    if (canUpdate) {
      this.checkInDisabledSource.next(!hasCheckIn);
      this.checkOutDisabledSource.next(!hasCheckOut);
      this.checkInCountSource.next(checkInCount);
      this.checkOutCountSource.next(checkOutCount);
    } else {
      this.checkInDisabledSource.next(true);
      this.checkOutDisabledSource.next(true);
      this.checkInCountSource.next(0);
      this.checkOutCountSource.next(0);
    }
  }

  manageGuests(item: CalendarItem, event): void {
    this.frontOfHouseService.manageGuests(item);
    event.stopPropagation();
  }

  manageRooms(item: CalendarItem, event): void {
    this.frontOfHouseService.manageRooms(item);
    event.stopPropagation();
  }

  guestDetails(groupId: string, guestId: string): void {
    this.frontOfHouseService.guestDetails(groupId, guestId);
  }

  /**
   * Updates items and runs change detection
   */
  refresh(): void {
    this.updateRoomItems();
  }

  nights(item: ReservationItem): number {
    const start = moment(item.from);
    const end = moment(item.toBooked);
    const nights = Math.ceil(end.diff(start, 'day', true));

    return nights;
  }
}
