import { Component, OnInit, Input, OnDestroy } from '@angular/core';
import { Itinerary } from '../../core/reservation/itinerary';
import { ReservationService } from '../../services/reservation/reservation.service';
import { CalendarService } from '../../services/calendar/calendar.service';
import { switchMap, map, filter } from 'rxjs/operators';
import { Observable, combineLatest } from 'rxjs';
import { ReservationPortalService } from '../../services/reservation-portal/reservation-portal.service';
import { PortalService } from 'src/app/shared/components/portal';
import { SubSink } from 'subsink';
import { FilterService } from '../../services/filter/filter.service';

@Component({
  selector: 'app-reservation-itinerary',
  templateUrl: './reservation-itinerary.component.html',
  styleUrls: ['./reservation-itinerary.component.scss']
})
export class ReservationItineraryComponent implements OnInit, OnDestroy {
  @Input() portalName: string;
  itineraries$: Observable<Itinerary[]>;
  private subsink: SubSink;
  showRooming$ = this.filterService.showRooming$;
  showExtras$ = this.filterService.showExtras$;

  constructor(
    private calendarService: CalendarService,
    private reservationService: ReservationService,
    private reservationPortalService: ReservationPortalService,
    private portalService: PortalService,
    private filterService: FilterService
  ) {
    this.subsink = new SubSink();
  }

  ngOnInit(): void {
    this.itineraries$ = combineLatest(
      this.calendarService.activeItem$,
      this.reservationPortalService.refresh$
    ).pipe(
      map(([activeItem]) => activeItem),
      filter(item => item !== null),
      switchMap(item => this.reservationService.getItinerary(item.data.reservationId))
    );

    this.subsink.sink = this.reservationPortalService.refresh$.pipe(
      switchMap(() => this.portalService.activePortal$),
      filter(activePortal => activePortal.name === this.portalName),
      switchMap(() => {
        this.portalService.onShowLoading(true);
        return this.itineraries$;
      })
    ).subscribe(() => {
      this.portalService.onShowLoading(false);
    })
  }

  onRoomingToggle(show: boolean): void {
    this.filterService.onShowRoomingChanged(show);
  }

  onExtrasToggle(show: boolean): void {
    this.filterService.onShowExtrasChanged(show);
  }

  ngOnDestroy() {
    this.subsink.unsubscribe();
  }
}
