import { Component, OnInit, Input, OnDestroy } from '@angular/core';
import { ReservationService } from '../../services/reservation/reservation.service';
import { Observable, combineLatest, of } from 'rxjs';
import { ReservationNotes } from '../../core/reservation/reservation-notes';
import { CalendarService } from '../../services/calendar/calendar.service';
import { switchMap, map, filter } from 'rxjs/operators';
import { ReservationPortalService } from '../../services/reservation-portal/reservation-portal.service';
import { SubSink } from 'subsink';
import { PortalService } from 'src/app/shared/components/portal';
import { FilterService } from '../../services/filter/filter.service';

@Component({
  selector: 'app-reservation-notes',
  templateUrl: './reservation-notes.component.html',
  styleUrls: ['./reservation-notes.component.scss']
})
export class ReservationNotesComponent implements OnInit, OnDestroy {
  @Input() portalName: string;
  expandNotes$ = this.filterService.expandNotes$;
  resNotes$: Observable<ReservationNotes>;
  private subsink: SubSink;

  constructor(
    private reservationService: ReservationService,
    private calendarService: CalendarService,
    private reservationPortalService: ReservationPortalService,
    private portalService: PortalService,
    private filterService: FilterService
    ) {
    this.subsink = new SubSink();
  }

  ngOnInit() {
    this.resNotes$ = combineLatest(
      this.calendarService.activeItem$,
      this.reservationPortalService.refresh$
    ).pipe(
      map(([activeItem]) => activeItem),
      switchMap(item => {
        if (item) {
          return this.reservationService.getReservationNotes(item.data.reservationId)
        } else {
          return of(null);
        }
      })
    );

    this.subsink.sink = this.reservationPortalService.refresh$.pipe(
      switchMap(() => this.portalService.activePortal$),
      filter(activePortal => activePortal.name === this.portalName),
      switchMap(() => {
        this.portalService.onShowLoading(true);
        return this.resNotes$
      })
    ).subscribe(() => {
      this.portalService.onShowLoading(false);
    })
  }

  onExpandToggle(expand: boolean): void {
    this.filterService.onExpandNotesChanged(expand);
  }

  ngOnDestroy() {
    this.subsink.unsubscribe();
  }
}
