import { Component, OnInit, OnDestroy, Input, ViewChild } from '@angular/core';
import { SidebarService } from 'src/app/shared/services/sidebar/sidebar.service';
import { FrontOfHouseService } from '../../services/front-of-house/front-of-house.service';
import { CalendarService } from '../../services/calendar/calendar.service';
import { first, filter, tap } from 'rxjs/operators';
import { LegacyService } from '../../services/legacy/legacy.service';
import { PortalComponent } from 'src/app/shared/components/portal/portal.component';
import { ReservationPortalService } from '../../services/reservation-portal/reservation-portal.service';
import { SubSink } from 'subsink';
import { Observable } from 'rxjs';
import { CalendarItem } from '../../core/calendar/calendar-item';
import { CalendarAccessService } from '../../services/access/calendar-access.service';

@Component({
  selector: 'app-reservation-portal',
  templateUrl: './reservation-portal.component.html',
  styleUrls: ['./reservation-portal.component.scss'],
  providers: [LegacyService, FrontOfHouseService]
})
export class ReservationPortalComponent implements OnInit, OnDestroy {
  @ViewChild('portal') portal: PortalComponent;
  private subsink: SubSink;
  activeItem$: Observable<CalendarItem>;

  constructor(
    private sidebarService: SidebarService,
    private frontOfHouseService: FrontOfHouseService,
    private calendarService: CalendarService,
    private reservationPortalService: ReservationPortalService,
    private accessService: CalendarAccessService,
  ) {
    this.subsink = new SubSink();
    this.activeItem$ = this.calendarService.activeItem$.pipe(
      tap(item => {
        if (!item) {
          this.close();
        }
      }),
      filter(item => item !== null)
    );
  }

  ngOnInit() {
    this.subsink.sink = this.reservationPortalService.openPortal$.subscribe(portal => {
      this.openPortal(portal);
    });
  }

  ngOnDestroy() {
    this.subsink.unsubscribe();
  }

  close(): void {
    this.sidebarService.close();
    this.calendarService.onReservationPortalClosed();
  }

  /**
   * Called when an action is emitted from the reservation portal.
   * @param action The action to execute.
   */
  doAction(action: string): void {
    switch (action) {
      case 'audit-trail':
        this.frontOfHouseService.auditTrail();
        break;

      default:
        break;
    }
  }

  async goToReservation(): Promise<void> {
    const activeItem = await this.calendarService.activeItem$.pipe(first()).toPromise();
    const reservationId = activeItem.data.reservationId;
    var job = this.accessService.canEditResTabs ? '2' : '0';
    const url = `/reservation.php?` + job + `+${reservationId}`;
    location.href = url;
  }

  openPortal(portal: string): void {
    setTimeout(() => {
      const portalService = this.portal.getPortalService();
      portalService.onPortalSelected(portal);
    }, 500);
  }
}
