import { Component, ChangeDetectionStrategy, TemplateRef, ViewChild } from '@angular/core';
import { ReportService } from './services/report/report.service';
import { FilterService } from './services/filter/filter.service';
import { Period } from './core/filters/period';
import { CookieService } from 'ngx-cookie-service';
import { WindowService } from './services/window/window.service';
import { SettingsService } from './services/settings/settings.service';
import { CalendarService } from './services/calendar/calendar.service';
import { RoomService } from './services/room/room.service';
import { SidebarService } from '../shared/services/sidebar/sidebar.service';
import { Sidebar } from '../shared/services/sidebar/sidebar';
import { CalendarAccessService } from './services/access/calendar-access.service';
import domtoimage from 'dom-to-image';
import { jsPDF } from 'jspdf';

@Component({
  selector: 'app-rooming-calendar',
  templateUrl: './rooming-calendar.component.html',
  styleUrls: ['./rooming-calendar.component.scss'],
  changeDetection: ChangeDetectionStrategy.OnPush,
  providers: [
    CookieService
  ]
})
export class RoomingCalendarComponent {
  /**
   * Selected accommodations
   */
  private accommodations: string[];

  /**
   * Calendar period
   */
  private period: Period;

  totalConflicts$ = this.calendarService.totalConflicts$;
  checkInTotal$ = this.roomService.checkInTotal$;
  checkOutTotal$ = this.roomService.checkOutTotal$;
  @ViewChild('checkInPortal') checkInPortal: TemplateRef<any>;
  @ViewChild('checkOutPortal') checkOutPortal: TemplateRef<any>;
  canUpdate$ = this.accessService.canUpdate$;

  constructor(
    private filterService: FilterService,
    private report: ReportService,
    private cookieService: CookieService,
    private windowService: WindowService,
    private settingsService: SettingsService,    // Do not remove. Used to update settings on changes automatically.
    private calendarService: CalendarService,
    private roomService: RoomService,
    private sidebarService: SidebarService,
    private accessService: CalendarAccessService
  ) {
    this.filterService.accommodations$.subscribe(accommodations => this.accommodations = accommodations);
    this.filterService.period$.subscribe(period => this.period = period);

    this.cookieService.set('mainWindowPage', 'room_module', 0, '/', null, null, null);
  }

  paxReport() {
    const startDate = this.period.start;
    const endDate = startDate.clone().add(this.period.days, 'days');
    this.report.paxReport('range', this.accommodations, startDate, endDate);
  }

  exportPDF(): void {
    var actions = document.getElementById('notForPDF'); //headerPrincipalName
    actions.classList.add('hidden');
    document.getElementsByClassName('main-content')[0].classList.add('forPDF');

    var node = document.getElementById('forPDF');
    var newImage;
    var filename;

    domtoimage.toJpeg(node, { bgcolor: '#fff' })
    .then(function (dataUrl) {
      actions.classList.remove('hidden');
      document.getElementsByClassName('main-content')[0].classList.remove('forPDF');
        var img = new Image();
        img.src = dataUrl;
        img.id  = 'tempPDF';
        document.body.appendChild(img);
        newImage = img.src;
        img.onload = function() {
            var pdfWidth = img.width;
            var pdfHeight = img.height;
            var doc;
            if(pdfWidth > pdfHeight) {
                doc = new jsPDF('l', 'px', [pdfWidth , pdfHeight]);
            } else {
                doc = new jsPDF('p', 'px', [pdfWidth , pdfHeight]);
            }
            var width = doc.internal.pageSize.getWidth();
            var height = doc.internal.pageSize.getHeight();
            doc.addImage(newImage, 'JPG',  10, 10, width, height);
            filename = 'Rooming Calendar.pdf';
            doc.save(filename);
        };
        document.getElementById('tempPDF').remove();
    })
    .catch(function(error) {
        this.app.modalShow("Error", error);
    });
  }

  openLegend() {
    const width = 720;
    const height = 825;
    this.windowService.openPopup('/resource/Resrequest/Application/public/html/legend_room.htm', 'Rooming Legend', width, height);
  }

  showRoomingConflicts(): void {
    this.calendarService.onConflictsDismissedChange(false);
  }

  private openSidenav(id: string, template: TemplateRef<any>): void {
    const sidebar: Sidebar = {
      id,
      template
    };
    this.sidebarService.showSidebar(sidebar);
  }

  openCheckInPortal(): void {
    this.openSidenav('rooming-calendar-bulk-check-in', this.checkInPortal);
  }

  openCheckOutPortal(): void {
    this.openSidenav('rooming-calendar-bulk-check-out', this.checkOutPortal);
  }
}
